// PremierePlugin.cpp: implementation of the CPremierePlugin class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PremierePlugin.h"
#include <windows.h>
#include <mmsystem.h>
#include "debug.h"

extern compStdParms  stdParms;
extern CGlobalSettings g_formatSettings;


///////////////////////////////////
// Stubs for premiere callbacks
int fgetAudio ( long frame, long *frameCount, long *size, long offset, BufferReturnType theBuffer, long compileSeqID )
{
  return ((CPremierePlugin *)compileSeqID)->fgetAudio(frame, frameCount, size, offset, theBuffer);
}

long fgetBlipMax ( long compileSeqID ) 
{
	return ((CPremierePlugin *)compileSeqID)->fgetBlipMax();
}

int fgetFrame ( long frame, void **buffer, long *rowbytes, compGetFrameReturnRec *getFrameReturn, char getCompressed, long compileSeqID )
{
  return ((CPremierePlugin *)compileSeqID)->fgetFrame(frame, buffer, rowbytes, getFrameReturn, getCompressed);
}


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CPremierePlugin::CPremierePlugin(char *filename )
{
  strcpy( m_filename, filename );
  m_pSettings = NULL;
  m_nSettingsSize = 0;
}

CPremierePlugin::~CPremierePlugin()
{
  if(m_pSettings)
    delete []m_pSettings;
}

flreturn_t CPremierePlugin::floutentry( flselector_t sel, ui32 par1, ui32 par2 )
{
  flreturn_t result = flo_ok;

  switch (sel)
  {
	 case flo_startup:
     result = startup( (startup_s *)par1 );
     break;
   case flo_shutdown:
     result = shutdown();
     break;
   case flo_configure:
     result = configure();
     break;
   case flo_compile:
     result = compile( (compile_s *)par1 );
     break;
   case flo_getinfo:
     result = getinfo( par1, par2 );
     break;
   case flo_setinfo:
     result = setinfo( par1, par2 );
     break;
  }

  return result;
}


flreturn_t CPremierePlugin::startup( startup_s *st )
{
	int i;

	m_hInstLibrary = LoadLibrary(m_filename); 
	if (!m_hInstLibrary)
		return flo_error;

	m_pPremEntry = (CompileEntryFunc) GetProcAddress(m_hInstLibrary, "xCompileEntry");
	if( !m_pPremEntry ) {
		FreeLibrary(m_hInstLibrary);
		return flo_error;
	}

  compInfoRec				infoRec;

	if( m_pPremEntry(compStartup, &stdParms, (long)&infoRec,0)!=comp_ErrNone ) {
		// Free the DLL module.
		FreeLibrary(m_hInstLibrary);
		return flo_error;
	}

  m_nCompilerID = infoRec.compilerID;
  m_nFileType   = infoRec.fileType;
	strcpy( m_CompilerName, infoRec.compilerName );

  // Supported formats
  long m_nSupportedVideoFormats = infoRec.extension == 'true' ? infoRec.suppOutputFormat : 0;
  m_nFloutSupported = FLO_VIDEO_RGB32;

  m_nFloutSupported |= m_nSupportedVideoFormats & PrI_RGB32 ? FLO_VIDEO_RGB32 : 0;
  m_nFloutSupported |= m_nSupportedVideoFormats & PrI_YUY2  ? FLO_VIDEO_YUY2  : 0;
  m_nFloutSupported |= m_nSupportedVideoFormats & PrI_YV12  ? FLO_VIDEO_YV12  : 0;

  compFileInfoRec fileInfoRec;
  compAudioInfoRec audioInfoRec;

	//Now query for the supported formats
	i=0;
	while( (m_pPremEntry)(compGetIndFormat, &stdParms, (long)&fileInfoRec,i)!=comp_BadFormatIndex )
	{
		i++;
		if(fileInfoRec.subtype==compUncompressed)
			break;
	}//Just look for uncompressed data

	i=0;
	while( (m_pPremEntry)(compGetAudioIndFormat, &stdParms, (long)&audioInfoRec,i)!=comp_BadFormatIndex )
	{
		i++;
		if(fileInfoRec.subtype==compUncompressed)
			break;
	}


  st->doesaudio = infoRec.compilesAudio;
  st->doesvideo = infoRec.compilesVideo;

  st->dualpass = infoRec.classID == 'yuv@' || 
                 infoRec.classID == 'CCE2';

  strcpy( st->modulename, m_CompilerName );
  st->moduleid = infoRec.classID;

  return flo_ok;
}

flreturn_t CPremierePlugin::shutdown()
{
  compInfoRec infoRec;

  m_pPremEntry(compDoShutdown , &stdParms, (long)&infoRec,0);
  FreeLibrary(m_hInstLibrary);

  return flo_ok;
}

flreturn_t CPremierePlugin::configure()
{
  compGetFilePrefsRec     FilePrefsRec;

  //User Preferences
  FilePrefsRec.compilerPrefs = (char *)m_pSettings;

  m_pPremEntry(compGetFilePrefs, &stdParms, (LONG)&FilePrefsRec,0);

  // Make sure the plugin gave fresh settings
  if( FilePrefsRec.compilerPrefs && FilePrefsRec.compilerPrefs!=(char *)m_pSettings ) {
    delete []m_pSettings;

    m_nSettingsSize = fgetPtrSize(FilePrefsRec.compilerPrefs);
    m_pSettings = new BYTE[m_nSettingsSize];
    memcpy( m_pSettings, FilePrefsRec.compilerPrefs, m_nSettingsSize );
  }

  return flo_ok;
}

flreturn_t CPremierePlugin::getinfo(ui32 par1, ui32 par2)
{
  videoinfo_s *vi;
  audioinfo_s *ai;
  cfgblockinfo_s *bi;
  fileinfo_s *fi;

  switch( par1 )
  {
    case flo_getvideoinfo:
      vi = (videoinfo_s *)par2;
      strcpy( vi->fmtname, m_CompilerName );
      strcat( vi->fmtname, " Video" );
      vi->supported_fmt = m_nFloutSupported;
      break;
    case flo_getaudioinfo:
      ai = (audioinfo_s *)par2;
      strcpy( ai->fmtname, m_CompilerName );
      strcat( ai->fmtname, " Audio" );
      break;
    case flo_getblockinfo:
      bi = (cfgblockinfo_s *)par2;
      bi->buf     = (ui8 *)m_pSettings;
      bi->bufsize = m_nSettingsSize;
      break;
    case flo_getfileinfo:
      fi = (fileinfo_s *)par2;
      switch( m_nFileType )
      {
        case 'MPEG':
        case '\0MPG':
        case 'MPG2':
          strcpy(fi->extension, "mpg");
          break;
        default:
          strcpy(fi->extension, "");        
          break;
      }
  }
  return flo_ok;
}

flreturn_t CPremierePlugin::setinfo(ui32 par1, ui32 par2)
{
  cfgblockinfo_s *bi;
  switch( par1 )
  {
    case flo_setblockinfo:
      bi = (cfgblockinfo_s *)par2;
      if(!bi->buf || !bi->bufsize)
        return flo_error;

      if(m_pSettings)
        delete []m_pSettings;

      m_nSettingsSize = bi->bufsize;
      m_pSettings = new BYTE[bi->bufsize];
      memcpy( m_pSettings, bi->buf, bi->bufsize );
      break;
  }
  return flo_ok;
}

flreturn_t CPremierePlugin::compile(compile_s *cs)
{
  videodef_s *vd=NULL;
  audiodef_s *ad=NULL;

 
  // Setup the streams of the conversion
  for( ui32 i=0; i<cs->streamcnt; i++)
  {
    switch(cs->streams[i]->streamtype)
    {
      case flo_video:
        vd = (videodef_s *)cs->streams[i];
        break;
      case flo_audio:
        ad = (audiodef_s *)cs->streams[i];
        break;
    }
  }

  if( !ad && !vd )
    return flo_invalidpar;

  return DoCompile( vd, ad , cs);
}

flreturn_t CPremierePlugin::DoCompile( videodef_s *vd, audiodef_s *ad, compile_s *cs )
{
  compDoCompileInfo CompileInfo;
  bool bDoAudio = ad>0;
  bool bDoVideo = vd>0;

  memset( &CompileInfo, 0, sizeof compDoCompileInfo );

  //Setup compiler plugin info
  CompileInfo.compilerID    = m_nCompilerID;
  CompileInfo.compilerPrefs = m_pSettings;
  
  //CompOutputRec
  CompileInfo.outputRec.doVideo = bDoVideo;
  CompileInfo.outputRec.doAudio = bDoAudio;
  
  //vidCompression
  CompileInfo.outputRec.vidCompression.subtype  = m_nFileType;
  CompileInfo.outputRec.vidCompression.depth    = 16;
  CompileInfo.outputRec.vidCompression.recompressWhen=recompNever;
  
  //CompileInfo.subtype=un
  //PostProcessing int gammaCorrection;
  CompileInfo.postProcessing.gammaCorrection = 1;
  CompileInfo.outputRec.width  =  vd->width ;
  CompileInfo.outputRec.height =  vd->height;
  
  //compTimebaseRec
  CompileInfo.outputRec.timebase.scale=			    vd->framerate_num;
  CompileInfo.outputRec.timebase.sampleSize=		vd->framerate_den;
  CompileInfo.outputRec.timebase.value=			    0;
  CompileInfo.outputRec.fieldType = compFieldsNone;
  
  //compPostProcessing
  //compFileSpec
  strcpy( CompileInfo.outputFile.name, cs->filename );
  

  ui32 nChannels, nFrequency, nDepth;
  if( bDoAudio ) 
  {
    nChannels = ad->fmt.channels;
    nFrequency = ad->fmt.frequency;
    nDepth = ad->fmt.depth;
  }
  else // some plugins require info even if we don't do audio
  {
    nChannels = 2;
    nFrequency = 44100;
    nDepth = 16;
  }

  WAVEFORMATEX AudioCompressionRecord;

  AudioCompressionRecord.wFormatTag      = WAVE_FORMAT_PCM;
  AudioCompressionRecord.nChannels       = nChannels;
  AudioCompressionRecord.nSamplesPerSec  = nFrequency;
  AudioCompressionRecord.nAvgBytesPerSec = nFrequency * nChannels * nDepth/8;
  AudioCompressionRecord.nBlockAlign     = nChannels * nDepth/8;
  AudioCompressionRecord.wBitsPerSample  = nDepth; 
  AudioCompressionRecord.cbSize          = 0;

  m_nAudBlockAlign = nChannels * nDepth/8;

  memcpy(&CompileInfo.outputRec.audCompression.AudCompRec, (char *)&AudioCompressionRecord, sizeof(WAVEFORMATEX));
  CompileInfo.outputRec.audCompression.subtype = 0;

  CompileInfo.outputRec.fieldType     = compFieldsNone;
  CompileInfo.outputRec.audrate       = nFrequency;
  CompileInfo.outputRec.audsamplesize = nDepth;
  CompileInfo.outputRec.stereo        = nChannels == 2;
  CompileInfo.outputRec.audchunksize  = (int)((nFrequency)/((double)vd->framerate_num/(double)vd->framerate_den)*AudioCompressionRecord.nBlockAlign); //samples/Frame * 2bytes/sample * 2channels 


  m_nAudSamplesPerFrame = MulDiv( nFrequency, vd->framerate_den,  vd->framerate_num);

  CompileInfo.outputRec.audInterleave = prInterleave1Frame;


  // This has to be modified to support some plugins.
  // now it is modified
  #define plug_is_cce ((m_CompilerName[0]=='C') && (m_CompilerName[6]=='C'))
  if (plug_is_cce)
  {
	  m_hOutputFile = (HANDLE) CreateFile( CompileInfo.outputFile.name, GENERIC_WRITE, 
											                    0, NULL, CREATE_ALWAYS,
											                    FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN, NULL );
  
	if (m_hOutputFile == INVALID_HANDLE_VALUE)
		m_hOutputFile=NULL;
  }
  else
	  m_hOutputFile=NULL;
		  
  //To close: CloseHandle
  CompileInfo.outputFileRef = (compFileRef) m_hOutputFile;
  CompileInfo.startFrame    =  1;
  CompileInfo.endFrame      =  cs->length;
  CompileInfo.compileSeqID  =  (long)this;

  m_cs = cs;
  m_ad = ad;
  m_vd = vd;
 
  // It's time to set the global settings so the premiere plugin
  // asks for them if it needs to
  if( ad ) {
    g_formatSettings.Set( kSettingsIsStereo, ad->fmt.channels==2 );
    g_formatSettings.Set( kSettingsIs16BitAudio, ad->fmt.depth==16 );
    g_formatSettings.Set( kSettingsGetAudRate, ad->fmt.frequency );
  }

  if( vd ) {
    long prFormat = vd->vidformat==FLO_VIDEO_YV12 ? PrI_YV12 :
                    vd->vidformat==FLO_VIDEO_YUY2 ? PrI_YUY2 : PrI_RGB32;

    g_formatSettings.Set( kSettingsVidWidth, vd->width );
    g_formatSettings.Set( kSettingsVidHeight, vd->height );
    g_formatSettings.Set( kSettingsVidFormat, prFormat );
  }

  long res = m_pPremEntry(compDoCompile, &stdParms, (long)&CompileInfo,0);

  if( plug_is_cce )
    CloseHandle( m_hOutputFile );

  return ( res==comp_ErrNone || res==comp_CompileAbort || res==comp_CompileDone )? flo_ok : flo_error; 
}

int CPremierePlugin::fgetFrame ( long frame, void **buffer, long *rowbytes, compGetFrameReturnRec *getFrameReturn, char getCompressed )
{
  flreturn_t ret;
  ret =  m_cs->getstream( m_cs->compileid, m_vd->strdef.streamid, (getstream_s *)&m_gv );
  if( ret == flo_ok || ret == flo_endfirstpass ) 
  {
    // If this is the first frame of the second pass, retrieve it.
    //  flo_endfirstpass only indicates the end of the first pass.
    //  there is no actual frame returned. 
    if( ret == flo_endfirstpass )
      m_cs->getstream( m_cs->compileid, m_vd->strdef.streamid, (getstream_s *)&m_gv );

    *buffer = m_gv.gs.buf;
    *rowbytes = 4*m_vd->width;     // FIXME, different formats...
	  getFrameReturn->returnVal         = comp_ErrNone;
	  getFrameReturn->repeatCount       = 0;
	  getFrameReturn->makeKeyFrame      = FALSE;
	  getFrameReturn->frameIsCompressed = FALSE;
	  getFrameReturn->startframe        = frame;
	  getFrameReturn->frameOnMarker     = FALSE;
  }
  else
  {
    *buffer = m_gv.gs.buf;
    *rowbytes = 4*m_vd->width;     // FIXME, different formats...
	  getFrameReturn->returnVal         = comp_CompileAbort;
	  getFrameReturn->repeatCount       = 0;
	  getFrameReturn->makeKeyFrame      = FALSE;
	  getFrameReturn->frameIsCompressed = FALSE;
	  getFrameReturn->startframe        = frame;
	  getFrameReturn->frameOnMarker     = FALSE;
  }

  return getFrameReturn->returnVal;
}


int CPremierePlugin::fgetAudio ( long frame, long *frameCount, long *size, long offset, BufferReturnType theBuffer )
{
  __int64 pos_now, pos_next_frame;

  // Mimic Adobe Premiere behaviour
  // Current sample position
  pos_now        =  (__int64)(frame-1)  *  (__int64)m_nAudSamplesPerFrame ;
  pos_next_frame =  (__int64)frame    *  (__int64)m_nAudSamplesPerFrame ;
  
  m_ga.samplecnt = (ui32)(pos_next_frame - pos_now);

  if( m_cs->getstream( m_cs->compileid, m_ad->strdef.streamid, (getstream_s *)&m_ga ) == flo_ok )
  {
    *theBuffer      = (char **)&m_ga.gs.buf;
    *size = m_ga.gs.bufsize;
    return comp_ErrNone;
  }
  else
    return comp_CompileAbort;
}


long CGlobalSettings::Get( int selector )
{
  long res = 0;
  switch( selector ) {
		case kSettingsVidWidth:
			res = m_nWidth;
      break;
		case kSettingsVidHeight:
			res =  m_nHeight;
      break;
		case kSettingsIsStereo:
			res =  m_nIsStereo;
      break;
		case kSettingsIs16BitAudio:
			res =  m_nIs16BitAudio;
      break;
		case kSettingsGetAudRate:
      res =  m_nAudRate;
      break;
    case kSettingsVidFormat:
      res =  m_nVidFormat;
      break;
    case kSettingsSubtype:
      res =  m_nSubtype;
      break;
    default:
      break;
  }
  DBG_STR(( str, "CGlobalSettings::Get() - selector:%d, result:%d\n", selector, res ))
  return res;
}

void CGlobalSettings::Set( int selector , long value )
{
  switch( selector ) {
		case kSettingsVidWidth:
			m_nWidth = value;
      break;
		case kSettingsVidHeight:
			m_nHeight = value;
      break;
		case kSettingsIsStereo:
			m_nIsStereo = value;
      break;;
		case kSettingsIs16BitAudio:
			m_nIs16BitAudio = value;
      break;
		case kSettingsGetAudRate:
			m_nAudRate = value;
      break;
    case kSettingsVidFormat:
			m_nVidFormat = value;
      break;
    case kSettingsSubtype:
			m_nSubtype = value;
      break;
    default:
      break;
  }
}