/* 
 *  Vcdread.cpp  // routines to access .dat vcd data
 *
 *	Copyright (C) Alberto Vigata - December 2000  ultraflask@yahoo.com
 *    
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <stdio.h>
#include <string.h>

#include "mism_vcd.h"
#include "vcdread.h"

struct RIFFHeader
{
  char ID[4];
  unsigned long nSize;
};



// Opens a track in the given opened file
// returns the length of the data in the track
static bool OpenTrack(TrackData *pTrackData)
{
  RIFFHeader sRiffHdr;
  bool bSuccess = false;
  if(!pTrackData || !pTrackData->pFile )
    return false;

  pTrackData->nDataStart   = 0;
  pTrackData->nSectorCount = 0;

  // Look if this is a RIFF file
  fread( &sRiffHdr, 1, sizeof RIFFHeader, pTrackData->pFile);
  
  bSuccess = !strncmp ("RIFF", sRiffHdr.ID, 4);
  
  if(bSuccess)
  {
    // RIFF found, look for CDXA ID
    fread( &sRiffHdr, 1, 4, pTrackData->pFile);
    bSuccess = !strncmp ("CDXA", sRiffHdr.ID, 4);
  }
  
  if(bSuccess)
  { 
    // CDXA chunk , look for fmt chunk
    fread( &sRiffHdr, 1, sizeof RIFFHeader, pTrackData->pFile);
    bSuccess = !strncmp ("fmt ", sRiffHdr.ID, 4);
    // skip fmt data. How long is this chunk anyways?
    char cDummy;
    if(bSuccess)
    {
      for(unsigned long i=0; i<sRiffHdr.nSize; i++)
        fread(&cDummy, 1, 1, pTrackData->pFile);
    }
  }
  
  if(bSuccess)
  {
    // look for data chunk
    fread( &sRiffHdr, 1, sizeof RIFFHeader, pTrackData->pFile);
    bSuccess = !strncmp ("data", sRiffHdr.ID, 4);
    
    // data chunk size must be multiple of 2352, the sector size
    if(bSuccess) bSuccess = bSuccess && ((sRiffHdr.nSize%2352)==0);

    // update track data struct
    if(bSuccess)
    {
      fpos_t fpCurrentPos;
      fgetpos(pTrackData->pFile, &fpCurrentPos);

      pTrackData->nDataStart = (unsigned long)fpCurrentPos;
      pTrackData->nSectorCount  = sRiffHdr.nSize / CDXA_SECTOR_SIZE;
    }
  }
  
  // There are some empty sectors at the beginning of every track
  // We don't really care about it.
  return bSuccess;
}

bool VCDIsVAlidTrack(char *pFileName)
{
  FILE *pFile;
  TrackData sTrackData;
  bool bSuccess = false;

  // Opening file
  if( !(pFile = fopen( pFileName, "rb" )) )
  {
    return false;
  }
  
  sTrackData.pFile = pFile;
  bSuccess = OpenTrack( &sTrackData);

  fclose( pFile );  
  return bSuccess;
}





bool inline ReadSector(TrackData *pTrackData)
{
  // We are aligned to a sector boundary
  if( fread(&pTrackData->sSector, CDXA_SECTOR_SIZE, 1, pTrackData->pFile) ==1 )
  {
    pTrackData->nSectorPos = 0;
    pTrackData->nSectorIdx++;    
    return true;
  }
  else
  {
    return false;
  }
}

TrackData *VCDOpenStream(char *pFileName )
{
  // Allocate TrackData structure
  TrackData *pTrackData = new TrackData;
  
  memset( pTrackData, 0, sizeof TrackData);
  
  FILE *pFile;
  bool bSuccess = false;
  
  // Opening file
  if( !(pFile = fopen( pFileName, "rb" )) )
  {
    delete pTrackData;
    return false;
  }
  
  pTrackData->pFile = pFile;
  
  bSuccess = OpenTrack(pTrackData);
  
  if(bSuccess)
  {
    // This will force a read in the first operation
    pTrackData->nSectorIdx = -1;
    VCDSetStreamPos( pTrackData, 0);
  }
  
  if(!bSuccess) delete pTrackData;
  
  return bSuccess ? pTrackData : NULL;
}

bool VCDCloseStream(TrackData *pTrackData)
{
  if( !pTrackData )
    return false;
  
  // Close file
  if( !pTrackData->pFile )
    return false;
  
  fclose(pTrackData->pFile);
  
  //Deallocate structure
  delete pTrackData;
  
  return true;
}

int VCDReadStream(TrackData *pTrackData, ui8 *buf, unsigned int nSize)
{
  unsigned int nRemBytes, nBytesToRead, nTotalSize = nSize;
  ui8* data = pTrackData->sSector.data;

  while(nSize)
  {
    if( pTrackData->nSectorPos==CDXA_SECTOR_DATASIZE )
    {
      if( !ReadSector(pTrackData) )
        return nTotalSize - nSize;
    }


    nRemBytes = CDXA_SECTOR_DATASIZE - pTrackData->nSectorPos;

    nBytesToRead = nSize <= nRemBytes ? nSize : nRemBytes;
    // Read data
    memcpy( buf, &data[pTrackData->nSectorPos], nBytesToRead);
    buf                    += nBytesToRead;
    nSize                  -= nBytesToRead;
    pTrackData->nSectorPos += nBytesToRead;
  }
  return nTotalSize - nSize;
}

int VCDSetStreamPos(TrackData *pTrackData, ui64 pos)
{
  fpos_t global_pos;

  // Pos is referred to valid user data in the sectors
  int nAlignPos           = (int)pos % CDXA_SECTOR_DATASIZE;
  unsigned int nSectorIdx = (int)pos / CDXA_SECTOR_DATASIZE;

  if(pTrackData->nSectorIdx != nSectorIdx)
  {
    // We have to grab the sector of this pos
    // Jump at the beginning
    global_pos = nSectorIdx * CDXA_SECTOR_SIZE + pTrackData->nDataStart;
    fsetpos( pTrackData->pFile, &global_pos);

    // Read the sector
    ReadSector(pTrackData);

    // Update nSectorIdx
    pTrackData->nSectorIdx = nSectorIdx;
  }
  
  pTrackData->nSectorPos = nAlignPos;

  return 1;
}

int VCDGetStreamPos(TrackData *pTrackData, ui64 *pos)
{
  *pos = pTrackData->nSectorIdx * CDXA_SECTOR_DATASIZE + pTrackData->nSectorPos;
  return 1;
}

ui64 VCDGetStreamSize(TrackData *pTrackData)
{
  return pTrackData->nSectorCount * CDXA_SECTOR_DATASIZE;
}