/* 
 *  options.cpp    -  Controls FlasKMPEG options
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

 
#include "flaskmpeg.h"
#include "resource.h"
#include "auxiliary.h"
#include <commctrl.h>
#include "error.h"
#include ".\Resizer\resizer.h"
#include ".\postprocessing.h"
#include ".\Video\Videowrapper.h"
#include ".\misc\ieee1180testdlg.h"
#include ".\runstate.h"
#include ".\misc\selectordialog.h"
#include ".\OutputPad.h"
 
extern TRunState        rs;
extern HINSTANCE        hInst;

struct TOptions{
	int         frames_compile;
};

#define GET_FRAMERATE(x) ((double)x.scale/(double)x.sampleSize)

// Dlg prototypes
void DlgVideoStartUp( HWND hDlg );

static 
struct Trunoptions{
	CFrame   in, out;
	TPPost              pp;
	TOptions            opt;
  TProfile            prof;
  TConfig             conf;
} opt_run;

HFONT myFont;


#define C_PAGES 5
typedef struct tag_dlghdr { 
   HWND hwndTab;       // tab control 
   HWND hwndDisplay;   // current child dialog box 
   RECT rcDisplay;     // display rectangle for the tab control 
   HWND DlgTabs[C_PAGES];
   /* these function pointers check for valid data in the given
    * tab (corresponding to DlgTabs); on valid data, true is returned;
	* on invalid data, an error is displayed, focus set on the
	* proper control and false is returned. 
	*
    * If this is NULL, then there are no checks and a true return
    * is assumed.
    */
   bool (*DlgTabCheck[C_PAGES])(HWND hDlg);
} DLGHDR; 
HWND video,audio,processing,files,general;
DLGHDR	*pHdr;

LRESULT CALLBACK OptionsProfileNameDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);


void EnableCropBoxes(HWND hDlg){
  DlgEnable(hDlg, IDC_CROP_TOPOFFSET);
  DlgEnable(hDlg, IDC_CROP_HEIGHT);
  DlgEnable(hDlg, IDC_CROP_LEFTOFFSET);
  DlgEnable(hDlg, IDC_CROP_WIDTH);
}

void DisableCropBoxes(HWND hDlg){
  DlgDisable(hDlg, IDC_CROP_TOPOFFSET);
  DlgDisable(hDlg, IDC_CROP_HEIGHT);
  DlgDisable(hDlg, IDC_CROP_LEFTOFFSET);
  DlgDisable(hDlg, IDC_CROP_WIDTH);
}
void EnableLetterboxBoxes(HWND hDlg){
  DlgEnable(hDlg, IDC_LETTERBOX_TOP);
  DlgEnable(hDlg, IDC_LETTERBOX_BOTTOM);
  DlgEnable(hDlg, IDC_LETTERBOX_RIGHT);
  DlgEnable(hDlg, IDC_LETTERBOX_LEFT);
}

void DisableLetterboxBoxes(HWND hDlg){
  DlgDisable(hDlg, IDC_LETTERBOX_TOP);
  DlgDisable(hDlg, IDC_LETTERBOX_BOTTOM);
  DlgDisable(hDlg, IDC_LETTERBOX_RIGHT);
  DlgDisable(hDlg, IDC_LETTERBOX_LEFT);
}
void SetCropBoxesText(HWND hDlg){
  DlgSetText(hDlg, IDC_CROP_TOPOFFSET, opt_run.pp.cropTopOffset);
  DlgSetText(hDlg, IDC_CROP_HEIGHT,    opt_run.pp.cropHeight);
  DlgSetText(hDlg, IDC_CROP_LEFTOFFSET,opt_run.pp.cropLeftOffset);
  DlgSetText(hDlg, IDC_CROP_WIDTH,     opt_run.pp.cropWidth);
}
void SetLetterboxBoxesText(HWND hDlg){
  DlgSetText(hDlg, IDC_LETTERBOX_TOP,       opt_run.pp.letterboxTop);
  DlgSetText(hDlg, IDC_LETTERBOX_BOTTOM,    opt_run.pp.letterboxBottom);
  DlgSetText(hDlg, IDC_LETTERBOX_RIGHT,     opt_run.pp.letterboxRight);
  DlgSetText(hDlg, IDC_LETTERBOX_LEFT,      opt_run.pp.letterboxLeft);
}
void GetCropBoxesText(HWND hDlg){
  opt_run.pp.cropTopOffset = DlgGetText(hDlg, IDC_CROP_TOPOFFSET);
  opt_run.pp.cropHeight    = DlgGetText(hDlg, IDC_CROP_HEIGHT);
  opt_run.pp.cropLeftOffset= DlgGetText(hDlg, IDC_CROP_LEFTOFFSET);
  opt_run.pp.cropWidth     = DlgGetText(hDlg, IDC_CROP_WIDTH);
}
void GetLetterboxBoxesText(HWND hDlg){
  opt_run.pp.letterboxTop    = DlgGetText(hDlg, IDC_LETTERBOX_TOP);
  opt_run.pp.letterboxBottom = DlgGetText(hDlg, IDC_LETTERBOX_BOTTOM);
  opt_run.pp.letterboxRight  = DlgGetText(hDlg, IDC_LETTERBOX_RIGHT);
  opt_run.pp.letterboxLeft   = DlgGetText(hDlg, IDC_LETTERBOX_LEFT);
}

// UpdateCropAndLetterboxBoxes in postprocessing tab
void UpdateCropAndLetterboxBoxes(HWND hDlg){
  // when doAR is turned on, letterbox and (if needed) crop are
  // turnen off automatically in opt_run.pp. We only need to disable
  // the checkbox buttons.
  if (opt_run.pp.doAR) {
    DlgCheck(hDlg, R_OPT_PPOST_KEEPASPECT);
    DlgEnable(hDlg, IDC_AR11);
    DlgEnable(hDlg, IDC_AR43);
    DlgEnable(hDlg, IDC_AR169);
  }
  else {
    DlgUnCheck(hDlg, R_OPT_PPOST_KEEPASPECT);
    DlgDisable(hDlg, IDC_AR11);
    DlgDisable(hDlg, IDC_AR43);
    DlgDisable(hDlg, IDC_AR169);
  }
  if(opt_run.pp.crop){
    DlgUnCheck(hDlg, IDC_NOCROP);
    EnableCropBoxes(hDlg);
  }
  else{
    DlgCheck(hDlg, IDC_NOCROP);
    DisableCropBoxes(hDlg);
  }
  if(opt_run.pp.letterbox){
    DlgUnCheck(hDlg, IDC_NOLETTERBOX);
    EnableLetterboxBoxes(hDlg);
  }
  else{
    DlgCheck(hDlg, IDC_NOLETTERBOX);
    DisableLetterboxBoxes(hDlg);
  }
  SetCropBoxesText(hDlg);
  SetLetterboxBoxesText(hDlg);
}




// Retrieve timebase from video options tab
void GetFrameRate(HWND hDlg, timebase_s *timebase){

	if(DlgIsChecked(hDlg,IDC_TB23)){ //23.976
		//HACK: LSX ENCODER DOESNT LIKE RIGHT TIMEBASE
		timebase->scale=		23976;
		timebase->sampleSize=	1000;
	}else if(DlgIsChecked(hDlg,IDC_TB24)){
		timebase->scale=		24;
		timebase->sampleSize=	1;
	}else if(DlgIsChecked(hDlg,IDC_TB29)){
		timebase->scale=		2997;
		timebase->sampleSize=	100;
	}else if(DlgIsChecked(hDlg,IDC_TB30)){
		timebase->scale=		30;
		timebase->sampleSize=	1;
	}else {
		timebase->scale=		25;
		timebase->sampleSize=	1;
	}
}

///////////////////////////////////////////////////////////
//             GLOBAL    OPTIONS
///////////////////////////////////////////////////////////
	// OnChildDialogInit - Positions the child dialog box to fall 
	//     within the display area of the tab control. 
 
	VOID WINAPI OnChildDialogInit(HWND hwndDlg) 
	{ 
    HWND hwndParent = GetParent(hwndDlg); 
    DLGHDR *pHdr = (DLGHDR *) GetWindowLong( 
        hwndParent, GWL_USERDATA); 
    SetWindowPos(hwndDlg, HWND_TOP, 
        pHdr->rcDisplay.left, pHdr->rcDisplay.top, 
        0, 0, SWP_NOSIZE); 
	} 


void GetFilter(HWND hDlg){
	// Check filter settings
	if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_NEAREST))
		opt_run.pp.filterMode=FILTER_NONE;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_BILINEAR))
		opt_run.pp.filterMode=FILTER_BILINEAR;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_BICUBIC))
		opt_run.pp.filterMode=FILTER_BICUBIC;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC))
		opt_run.pp.filterMode=FILTER_TABLEBICUBIC;
	else 
		opt_run.pp.filterMode=FILTER_TABLEBICUBIC;
}

void DlgProcessingStartUp( HWND hDlg )
{
    //CROP

    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_NEAREST );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_BILINEAR );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_BICUBIC );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC );
    switch(opt_run.prof.filter)
	{
	    case FILTER_NONE:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_NEAREST);
			break;
		case FILTER_BILINEAR:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_BILINEAR);
			break;
		case FILTER_BICUBIC:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_BICUBIC);
			break;
		case FILTER_TABLEBICUBIC:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC);
			break;
	}

	if (opt_run.prof.keepAspect & AR_43)
		DlgCheck(hDlg, IDC_AR43);
	else if (opt_run.prof.keepAspect & AR_169)
		DlgCheck(hDlg, IDC_AR169);
	else
		DlgCheck(hDlg, IDC_AR11); // default VGA output AR

	// quick fix for old profiles that still allow letterboxing/cropping
	// with keep AR
	if (opt_run.pp.doAR) {
		opt_run.pp.letterbox = false;
		if (opt_run.pp.outAR & (AR_43|AR_169))
			opt_run.pp.crop = false;
	}

	UpdateCropAndLetterboxBoxes(hDlg);

    return;
}

LRESULT CALLBACK DlgProcessing(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
				OnChildDialogInit(hDlg);
        FromConfigToPPost(&opt_run.prof, &opt_run.pp);
        DlgProcessingStartUp(hDlg);
				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			
			switch(LOWORD(wParam)){
				case R_OPT_PPOST_KEEPASPECT:
					opt_run.pp.doAR = DlgIsChecked(processing, R_OPT_PPOST_KEEPASPECT);

					UpdateCropAndLetterboxBoxes(hDlg);
					break;
				case IDC_AR11:
				case IDC_AR43:
				case IDC_AR169:
					if (DlgIsChecked(processing, IDC_AR11))
						opt_run.pp.outAR = AR_11;
					else if (DlgIsChecked(processing, IDC_AR43))
						opt_run.pp.outAR = AR_43;
					else if (DlgIsChecked(processing, IDC_AR169))
						opt_run.pp.outAR = AR_169;

					UpdateCropAndLetterboxBoxes(hDlg);
					break;
				case IDC_NOCROP:
					opt_run.pp.crop=!DlgIsChecked(hDlg, IDC_NOCROP);
					if(opt_run.pp.crop)
						EnableCropBoxes(hDlg);
					else
						DisableCropBoxes(hDlg);
					break;
				case IDC_NOLETTERBOX:
					opt_run.pp.letterbox = !DlgIsChecked(hDlg, IDC_NOLETTERBOX);
					if(opt_run.pp.letterbox )
						EnableLetterboxBoxes(hDlg);
					else
						DisableLetterboxBoxes(hDlg);
					break;
				case R_OPT_PPOST_RESIZING_NEAREST:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_BILINEAR:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_BICUBIC:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_HQBICUBIC:
					GetFilter(hDlg); break;

				break;
			}
			break;
	}
    return FALSE;

}


/* Update settings from hDlg's video section.  If all settings are valid, return
 * true; if any setting is incorrect, print a warning, set focus on the erroneous
 * entry and return false.
 *
 * The allowed sizes depend on wether we're cropping or not. If we use crop,
 * the demand that sizes are a multiple of 16 goes for the cropping window,
 * with some more freedom for the resizing dimensions.
 */
static bool DlgVideo_Check(HWND hDlg)
{
	// width must be smaller than 1024 always,
	// a multiple of 16 if we're not cropping, a multiple of 2 if crop is on.
	if(DlgGetText(hDlg, IDC_WIDTH)<=0 || DlgGetText(hDlg, IDC_WIDTH)>=4096){
		PrintError(SIZE_OOR,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	if( (!opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_WIDTH)%16)!=0)){
		PrintError(SIZE_NOT16,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	if ( (opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_WIDTH)%2)!=0)){
		PrintError(INCORRECT_SETTING,(int)hDlg,0); // FIXME: should be size not /2
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}

	// a multiple of 16 if we're not cropping, free otherwise.
	if(DlgGetText(hDlg, IDC_HEIGHT)<=0 || DlgGetText(hDlg, IDC_HEIGHT)>=4096){
		PrintError(SIZE_OOR,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_HEIGHT);
		return false;
	}
	if( (!opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_HEIGHT)%16)!=0)){
		PrintError(SIZE_NOT16,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	
	opt_run.pp.resWidth = DlgGetText(hDlg, IDC_WIDTH);
	opt_run.pp.resHeight = DlgGetText(hDlg, IDC_HEIGHT);
	return true;
}

static bool DlgProcessing_Check(HWND hDlg)
{
	GetLetterboxBoxesText(hDlg);
	GetCropBoxesText(hDlg);

	if(!CheckVideoParameters(&opt_run.pp)){
		PrintError(INCORRECT_SETTING, (int)hDlg,0);
		/* FIXME: we need to know where to set focus, but what's a
		 * clean way to do this without duplicating CheckVideoParameters() code? */
		// DlgSetFocus(hDlg, LOWORD(wParam));
		return false;
	}

	return true;
}
static bool DlgGeneral_Check(HWND hDlg)
{
	char szTemp[256];

	if(DlgIsChecked(hDlg,IDC_COMPILE_WHOLE)) return true; /* don't bother */

	GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
	if(!atof(szTemp)){
		PrintError(TIME_IS_NOT_POSITIVE, (int)hDlg,0);
		DlgSetFocus(hDlg, IDC_COMPILE_TIME);
		return false;
	}

	if(opt_run.conf.displayTimeUnits == 0) {
		/* it's in seconds; switch to frames */
		timebase_s timebase;
		GetFrameRate(video, &timebase);

		GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
		opt_run.opt.frames_compile = (int)(atof(szTemp) * GET_FRAMERATE(timebase));
	} else {
		/* it's in frames; copy it directly */
		GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
		opt_run.opt.frames_compile = atoi(szTemp);
	} 

	return true;
}

void DlgVideoStartUp( HWND hDlg )
{
 
		DlgSetText(hDlg, IDC_WIDTH, opt_run.pp.resWidth);
    DlgSetText(hDlg, IDC_HEIGHT, opt_run.pp.resHeight);
    
    // Uncheck all checkboxes
    DlgUnCheck( hDlg, IDC_TB23 );
    DlgUnCheck( hDlg, IDC_TB24 );
    DlgUnCheck( hDlg, IDC_TB29 );
    DlgUnCheck( hDlg, IDC_TB30 );
    DlgUnCheck( hDlg, IDC_TB25 );
    switch( opt_run.prof.timeBase.scale )
    {
    case 23976: //23.976
      DlgCheck(hDlg, IDC_TB23);
      break;
    case 24:    //24
      DlgCheck(hDlg, IDC_TB24);
      break; 
    case 2997: //29.97
      DlgCheck(hDlg, IDC_TB29);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.prof.recons_progressive = 0;
      break;
    case 30:    //30
      DlgCheck(hDlg, IDC_TB30);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.prof.recons_progressive = 0;
      break;
    default:	//if not, set 25
      DlgCheck(hDlg, IDC_TB25);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.prof.recons_progressive = 0;
      break;
    }

    DlgCheckBoxState( hDlg, R_OPT_IDCT_AUTOSELECT, opt_run.conf.idctAutoSelect );
    if( DlgIsChecked(hDlg, R_OPT_IDCT_AUTOSELECT) )
      DlgDisable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
    else
      DlgEnable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
    
    DlgSetText(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD, opt_run.prof.threshold);
    DlgCheckBoxState( hDlg, R_OPT_TIMEBASE_RECONS, opt_run.prof.recons_progressive);
    DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK, opt_run.prof.deinterlace);
    DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND, opt_run.prof.blend);
    if(opt_run.prof.deinterlace)
    {
      DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
      DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
    }
    else
    {
      DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
      DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
    }
    
    DlgCheck( hDlg, IDC_DUALPASSCHECK, opt_run.conf.bDualPass );
    DlgCheck( hDlg, IDC_DONTRESIZE, !opt_run.pp.bResize );
    DlgEnable( hDlg, IDC_WIDTH, opt_run.pp.bResize );
    DlgEnable( hDlg, IDC_HEIGHT, opt_run.pp.bResize );

    return;
}

LRESULT CALLBACK DlgVideo(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  int i;
  switch (message)
  {
		case WM_INITDIALOG:
      OnChildDialogInit(hDlg);
      FromConfigToPPost(&opt_run.prof, &opt_run.pp); 
      DlgVideoStartUp( hDlg );
      
      return TRUE;
      
    case WM_COMMAND:
      if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
      {
        EndDialog(hDlg, LOWORD(wParam));
        return TRUE;
      }
      TSelectorDialog selDlgInfo;
      // framerate
      switch(LOWORD(wParam))
      {
      case R_OPT_IDCT_PERFORMIEEE1180:
        {
          CIEEE1180Dlg obTest(hDlg, rs.hInst);
          // Initialize idct
          void (* pInitIdct)() = rs.video->GetIdctInfo(opt_run.conf.idctIndex)->InitIdct;
          void (* pDeInit)() = rs.video->GetIdctInfo(opt_run.conf.idctIndex)->DeInitIdct;

          if(pInitIdct)
            pInitIdct();
          obTest.IsIEEE1180(rs.video->GetIdctInfo(opt_run.conf.idctIndex)->Idct,
                            rs.video->GetIdctInfo(opt_run.conf.idctIndex)->sDescription);
          if(pDeInit)
            pDeInit();
          
        }
        break;
      case R_OPT_IDCT_SELECTIONBUTTON:
        selDlgInfo.tittle = "iDCT Selection";
        selDlgInfo.lateral_text = "Select iDCT";
        selDlgInfo.button_text  = "Ok";
        selDlgInfo.section_count = 1;
        selDlgInfo.sections_titles[0] = "Available iDCTs";

        LPFMpegIdctInfo idctInfo;
        selDlgInfo.strings[0].SetArraySize(rs.video->GetIdctCount());
        selDlgInfo.helpstrings[0].SetArraySize(rs.video->GetIdctCount());
        for(i=0; i<rs.video->GetIdctCount(); i++)
        {
          idctInfo = rs.video->GetIdctInfo(i);
          //selDlgInfo.strings[0][i] =  (char *)(idctInfo->sDescription);
          selDlgInfo.strings[0][i] =  (char *)(idctInfo->sName);
          selDlgInfo.helpstrings[0][i] =  (char *)(idctInfo->sDescription);
          selDlgInfo.section_mode[0] = MUST_SELECT | SINGLE_SELECT;
        }
        ResetSelections(&selDlgInfo);
        // Select first one
        selDlgInfo.selected[0][opt_run.conf.idctIndex] = 1;
        OpenSelectorDialog( hDlg, hInst, &selDlgInfo );
        // Get result of selection
        opt_run.conf.idctIndex = GetFirstSelected( &selDlgInfo, 0 );

        break;
      case R_OPT_IDCT_AUTOSELECT:
        if( DlgIsChecked(hDlg, R_OPT_IDCT_AUTOSELECT) )
          DlgDisable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
        else
          DlgEnable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
        break;
      case R_OPT_VIDEO_DEINTERLACE_CHECK:
        if(DlgIsChecked(hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK) )
        {
          DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
          DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
        }
        else
        {
          DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
          DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
        }
        break;
      case IDC_TB23: 
      case IDC_TB24:
        DlgEnable(hDlg, R_OPT_TIMEBASE_RECONS);
        break; 
      case IDC_TB25:
      case IDC_TB29:
      case IDC_TB30:
        DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
        opt_run.prof.recons_progressive = 0;
        break;
      case IDC_DUALPASSCHECK:
        opt_run.conf.bDualPass = DlgIsChecked( hDlg, IDC_DUALPASSCHECK );
        break;
      case IDC_DONTRESIZE:
        opt_run.pp.bResize = !DlgIsChecked( hDlg, IDC_DONTRESIZE );
        DlgVideoStartUp(hDlg);
        break;
      }
  }
  return FALSE;
}


void DlgAudioStartUp( HWND hDlg )
{
			//Setting init options
                DlgUnCheck( hDlg, R_OPT_AUDIO_DOAUDIO );
                DlgUnCheck( hDlg, R_OPT_AUDIO_DSC );
                DlgUnCheck( hDlg, R_OPT_AUDIO_NOAUDIO );

				int audiomode = -1;
				if(rs.audio)
        {
					if(rs.audio->GetFormat()!=MpegAudio  &&  rs.audio->GetFormat()!=Ac3 )
          {
						DlgDisable(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(hDlg, R_OPT_AUDIO_DSC);
						audiomode = NO_AUDIO;
					}
				}
				else{
						DlgDisable(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(hDlg, R_OPT_AUDIO_DSC);
						audiomode = NO_AUDIO;
				}

				if (audiomode < 0)
					audiomode = opt_run.prof.audioMode;

                DlgUnCheck( hDlg, IDC_44100 );
                DlgUnCheck( hDlg, IDC_48000 );
				if(opt_run.prof.outSampleFrequency==44100)
					DlgCheck(hDlg, IDC_44100);
				else
					DlgCheck(hDlg, IDC_48000);

				switch( audiomode ){
					case DO_AUDIO:
						DlgCheck(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(files, IDC_OUTPUT_FILE2);
						DlgDisable(files, IDC_FILEBUTTON2);
						if(opt_run.prof.sampleFreqSameAsInput){
							DlgDisable(hDlg, IDC_48000);
							DlgDisable(hDlg, IDC_44100);
							DlgCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						}
						else{
							DlgEnable(hDlg, IDC_48000);
							DlgEnable(hDlg, IDC_44100);
							DlgUnCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						}
						DlgEnable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
					case DSC:
						DlgCheck(hDlg, R_OPT_AUDIO_DSC);
						DlgEnable(files, IDC_OUTPUT_FILE2);
						DlgEnable(files, IDC_FILEBUTTON2);
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
					case NO_AUDIO:
						DlgCheck(hDlg, R_OPT_AUDIO_NOAUDIO);
						DlgDisable(files, IDC_OUTPUT_FILE2);
						DlgDisable(files, IDC_FILEBUTTON2);
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
				}
    return;
}

LRESULT CALLBACK DlgAudio(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
				OnChildDialogInit(hDlg);
                DlgAudioStartUp( hDlg );
				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case R_OPT_AUDIO_SAMPLING_SAMEASINPUT:
					if(DlgIsChecked(hDlg,R_OPT_AUDIO_SAMPLING_SAMEASINPUT))
					{
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
					}
					else{
						DlgEnable(hDlg, IDC_48000);
						DlgEnable(hDlg, IDC_44100);
					}
					break;
				case R_OPT_AUDIO_DOAUDIO:
					DlgDisable(files, IDC_OUTPUT_FILE2);
					DlgDisable(files, IDC_FILEBUTTON2);
					if(opt_run.prof.sampleFreqSameAsInput){
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					}
					else{
						DlgEnable(hDlg, IDC_48000);
						DlgEnable(hDlg, IDC_44100);
						DlgUnCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					}
					DlgEnable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
				case R_OPT_AUDIO_DSC:
					DlgEnable(files, IDC_OUTPUT_FILE2);
					DlgEnable(files, IDC_FILEBUTTON2);
					DlgDisable(hDlg, IDC_48000);
					DlgDisable(hDlg, IDC_44100);
					DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
				case R_OPT_AUDIO_NOAUDIO:
					DlgDisable(files, IDC_OUTPUT_FILE2);
					DlgDisable(files, IDC_FILEBUTTON2);
					DlgDisable(hDlg, IDC_48000);
					DlgDisable(hDlg, IDC_44100);
					DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
			}
			break;
	}
    return FALSE;
}


void DlgGeneralStartUp( HWND hDlg )
{
  if(opt_run.conf.compileWhole) {
    // whole file
    DlgCheck( hDlg, IDC_COMPILE_WHOLE);
		  // disable edit box and unit selection
    DlgDisable( hDlg, IDC_COMPILE_TIME);
    DlgDisable( hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
		DlgDisable( hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
  }
  else
  {
    DlgUnCheck( hDlg, IDC_COMPILE_WHOLE);
    DlgEnable(hDlg, IDC_COMPILE_TIME);
    DlgEnable(hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
    DlgEnable(hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
  }

	opt_run.opt.frames_compile = opt_run.conf.framesToCompile;
			
  /* make sure we have a sane frame count, and not our magic '300000' */
  if(opt_run.conf.compileWhole) opt_run.opt.frames_compile = 100;
  
  if(opt_run.conf.displayTimeUnits == 0) {
    	/* secs */
	    timebase_s timebase;
      GetFrameRate(video, &timebase);
      
      DlgSetDouble( hDlg, IDC_COMPILE_TIME, opt_run.conf.framesToCompile / GET_FRAMERATE(timebase));
      DlgCheck(hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
  } else if(opt_run.conf.displayTimeUnits == 1) {
    /* frames */
    DlgSetText( hDlg, IDC_COMPILE_TIME, opt_run.conf.framesToCompile);
    DlgCheck(hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
  }
  
  DlgSetText(hDlg, IDC_LURKSIZE, opt_run.conf.lurk_size );
  
  if(opt_run.conf.shutdown)
    DlgCheck(hDlg, R_GENERAL_MISC_SHUTDOWN);
  else
    DlgUnCheck(hDlg, R_GENERAL_MISC_SHUTDOWN);
  
  // 
  switch( opt_run.conf.nProcessingFormat )
  {
    case FRAME_RGB32:
      DlgCheck(hDlg, IDC_PROCESSINGFORMAT_RGB);
      break;
    case FRAME_YUY2:
      DlgCheck(hDlg, IDC_PROCESSINGFORMAT_YUY2);
      break;
    case FRAME_YV12:
      DlgCheck(hDlg, IDC_PROCESSINGFORMAT_YV12);
      break;
    default:
      DlgCheck(hDlg, IDC_PROCESSINGFORMAT_RGB);
      break;
  }

  DlgCheck( hDlg, IDC_PROCESSINGAUTO, opt_run.conf.bProcessingFormatAuto );

  DlgEnable( hDlg, IDC_PROCESSINGFORMAT_RGB, opt_run.conf.bProcessingFormatAuto==false );
  DlgEnable( hDlg, IDC_PROCESSINGFORMAT_YUY2, opt_run.conf.bProcessingFormatAuto==false );
  DlgEnable( hDlg, IDC_PROCESSINGFORMAT_YV12, opt_run.conf.bProcessingFormatAuto==false );

  return;
}

LRESULT CALLBACK DlgGeneral(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	char szTemp[1024];
	switch (message)
	{
		case WM_INITDIALOG:
			OnChildDialogInit(hDlg);
            DlgGeneralStartUp(hDlg);

				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case IDC_COMPILE_WHOLE:
					opt_run.conf.compileWhole = DlgIsChecked(hDlg,IDC_COMPILE_WHOLE);
          DlgGeneralStartUp( hDlg );
					break;
				case R_GENERAL_COMPTIME_SECONDSCOMPILE:
					if(HIWORD(wParam) == BN_CLICKED && opt_run.conf.displayTimeUnits == 1) {
						/* switch IDC_COMPILE_TIME to seconds */						
						timebase_s timebase;
						GetFrameRate(video, &timebase);

						GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
						DlgSetDouble( hDlg, IDC_COMPILE_TIME, atoi(szTemp) / GET_FRAMERATE(timebase));
						opt_run.conf.displayTimeUnits = 0;
					}
					

					break;
				case R_GENERAL_COMPTIME_FRAMESCOMPILE:
					if(HIWORD(wParam) == BN_CLICKED && opt_run.conf.displayTimeUnits == 0) {
						/* switch IDC_COMPILE_TIME to frames */
						timebase_s timebase;
						GetFrameRate(video, &timebase);

						GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
						DlgSetText( hDlg, IDC_COMPILE_TIME, (int) (atof(szTemp) * GET_FRAMERATE(timebase)));
						opt_run.conf.displayTimeUnits = 1;
					}
					break;
        case IDC_PROCESSINGAUTO:
          opt_run.conf.bProcessingFormatAuto = DlgIsChecked( hDlg, IDC_PROCESSINGAUTO );
          DlgGeneralStartUp( hDlg );
          break;
        case IDC_PROCESSINGFORMAT_YUY2:
        case IDC_PROCESSINGFORMAT_RGB:
        case IDC_PROCESSINGFORMAT_YV12:
          opt_run.conf.nProcessingFormat = DlgIsChecked( general, IDC_PROCESSINGFORMAT_RGB ) ? FRAME_RGB32 :
                                           DlgIsChecked( general, IDC_PROCESSINGFORMAT_YUY2 )? FRAME_YUY2  :
                                           DlgIsChecked( general, IDC_PROCESSINGFORMAT_YV12 ) ? FRAME_YV12  : FRAME_RGB32;
          break;
			}
			break;
	}
    return FALSE;
}

void DlgFilesStartUp( HWND hDlg )
{
	DlgSetText(hDlg, IDC_OUTPUT_FILE1, opt_run.conf.outputFile);
	DlgSetText(hDlg, IDC_OUTPUT_FILE2, opt_run.conf.audioOutFile);	
	if(opt_run.conf.warn_overwrite)
		DlgCheck(hDlg, R_FILES_WARN);
	else
    	DlgUnCheck(hDlg, R_FILES_WARN);
    return;
}

LRESULT CALLBACK DlgFiles(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	char szTemp[MAXFILENAME];

	switch (message)
	{
		case WM_INITDIALOG:
			OnChildDialogInit(hDlg);
            DlgFilesStartUp( hDlg );
			return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case IDC_FILEBUTTON1:
					strcpy( szTemp, opt_run.conf.outputFile);
					if(GetOutputFileName(hDlg, szTemp, opt_run.conf.outputDir, "FlasK MPEG - Select output video file")){
						DlgSetText(hDlg, IDC_OUTPUT_FILE1, szTemp);
					}
					break;
				case IDC_FILEBUTTON2:
					strcpy( szTemp, opt_run.conf.audioOutFile);
					if(GetOutputFileName(hDlg, szTemp, opt_run.conf.outputDir, "FlasK MPEG - Select output audio file")){
						DlgSetText(hDlg, IDC_OUTPUT_FILE2, szTemp);
					}
					break;
			}
			break;
	}
    return FALSE;
}
// DoLockDlgRes - loads and locks a dialog template resource. 
// Returns the address of the locked resource. 
// lpszResName - name of the resource 
DLGTEMPLATE * WINAPI DoLockDlgRes(LPCSTR lpszResName) 
{ 
    HRSRC hrsrc = FindResource(NULL, lpszResName, RT_DIALOG); 
    HGLOBAL hglb = LoadResource(hInst, hrsrc); 
    return (DLGTEMPLATE *) LockResource(hglb); 
} 


// OnSelChanged - processes the TCN_SELCHANGE notification. 
// hwndDlg - handle to the parent dialog box. 
VOID WINAPI OnSelChanged(HWND hwndDlg) 
{ 
	    DLGHDR *pHdr = (DLGHDR *) GetWindowLong( 
        hwndDlg, GWL_USERDATA); 
    int iSel =opt_run.conf.selectedTab= TabCtrl_GetCurSel(pHdr->hwndTab); 
 
    // Hide the current child dialog box, if any. 
    if (pHdr->hwndDisplay != NULL) 
        ShowWindow(pHdr->hwndDisplay,SW_HIDE); 
 
    // Show. 
    pHdr->hwndDisplay = pHdr->DlgTabs[iSel];
        ShowWindow(pHdr->hwndDisplay,SW_SHOW);
} 


VOID WINAPI TabbedDialogInit(HWND hwndDlg) 
{ 
    pHdr = (DLGHDR *) LocalAlloc(LPTR, sizeof(DLGHDR)); 
    DWORD dwDlgBase = GetDialogBaseUnits(); 
    int cxMargin = LOWORD(dwDlgBase) / 4; 
    int cyMargin = HIWORD(dwDlgBase) / 8; 
    TCITEM tie; 
    RECT rcTab; 


    // Save a pointer to the DLGHDR structure. 
    SetWindowLong(hwndDlg, GWL_USERDATA, (LONG) pHdr); 
 
    // Create the tab control. 
    InitCommonControls(); 
    pHdr->hwndTab = CreateWindow( 
        WC_TABCONTROL, "", 
        WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE, 
        0, 0, 100, 100, 
        hwndDlg, NULL, hInst, NULL 
        ); 
    if (pHdr->hwndTab == NULL) {
        return ;
    }
	// Set font
	SendMessage(pHdr->hwndTab, WM_SETFONT, (WPARAM)myFont, MAKELPARAM(TRUE, 0));    
 
    // Add a tab for each of the three child dialog boxes. 
    tie.mask = TCIF_TEXT | TCIF_IMAGE; 
    tie.iImage = -1; 
    tie.pszText = "Video"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 0, &tie); 
    tie.pszText = "Audio"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 1, &tie); 
    tie.pszText = "Post Processing"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 2, &tie); 
     tie.pszText = "Files"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 3, &tie); 
     tie.pszText = "General"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 4, &tie); 
 
 
    SetRectEmpty(&rcTab); 
    rcTab.right = (292 * LOWORD(dwDlgBase)) / 4; 
    rcTab.bottom = (130 * HIWORD(dwDlgBase)) / 8; 
 
    // Calculate how large to make the tab control, so 
    // the display area can accommodate all the child dialog boxes. 
    TabCtrl_AdjustRect(pHdr->hwndTab, TRUE, &rcTab); 
    OffsetRect(&rcTab, cxMargin - rcTab.left, 
            cyMargin - rcTab.top); 
 
    // Calculate the display rectangle. 
    CopyRect(&pHdr->rcDisplay, &rcTab); 
    TabCtrl_AdjustRect(pHdr->hwndTab, FALSE, &pHdr->rcDisplay); 
 
    // Set the size and position of the tab control, buttons, 
    // and dialog box. 
    SetWindowPos(pHdr->hwndTab, NULL, rcTab.left, rcTab.top, 
            rcTab.right - rcTab.left, rcTab.bottom - rcTab.top, 
            SWP_NOZORDER); 
 
  
	//Create dialogs
	// files tab has dependences from all the others (audio)
	files=		pHdr->DlgTabs[3]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_FILES), hwndDlg, (DLGPROC)DlgFiles);

	video=		pHdr->DlgTabs[0]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_VIDEO_OPTIONS), hwndDlg, (DLGPROC)DlgVideo);
	audio=		pHdr->DlgTabs[1]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_AUDIO_OPTIONS), hwndDlg, (DLGPROC)DlgAudio);
	processing=	pHdr->DlgTabs[2]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_PROCESSING_OPTIONS), hwndDlg, (DLGPROC)DlgProcessing);
	general=	pHdr->DlgTabs[4]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_GENERAL), hwndDlg, (DLGPROC)DlgGeneral);

	pHdr->DlgTabCheck[0] = DlgVideo_Check;
	pHdr->DlgTabCheck[1] = pHdr->DlgTabCheck[3] = NULL;
	pHdr->DlgTabCheck[2] = DlgProcessing_Check;
	pHdr->DlgTabCheck[4] = DlgGeneral_Check;

	// Select Tab from options
	TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.conf.selectedTab);
	OnSelChanged(hwndDlg); 
    
} 
 

// RefreshTabsFromProfile()
//    Refreses the current tabs from a given profile
void RefreshTabsFromProfile(HWND hDlg)
{
    //Select Tab from profile
    TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.conf.selectedTab);
    OnSelChanged(hDlg);

	//Get post processing options from config
	FromConfigToPPost(&opt_run.prof, &opt_run.pp);
    
    DlgProcessingStartUp( processing );
    DlgVideoStartUp( video );
    DlgAudioStartUp( audio );
    DlgGeneralStartUp( general );
    DlgFilesStartUp( files );

    return;
}

void RefreshListBox( HWND hDlg )
{
    int i;
    //Language Selection
    //DlgSetText(hDlg, R_WELCOME_LANG, GS(MENU_LANGUAGE));
    ListClean(GetDlgItem(hDlg, IDC_PROFILES_LIST));
	for(i=0; i<rs.profiler->GetCount() ; i++){
		ListAddText(GetDlgItem(hDlg, IDC_PROFILES_LIST), (rs.profiler->Get(i))->profile_name );
	}
	ListSetCur(GetDlgItem(hDlg, IDC_PROFILES_LIST), rs.profiler->GetSelectedIndex() );
    return;
}

void RefreshOptionsDialog( HWND hDlg )
{
    // Get selected profile
    memcpy( &opt_run.prof, rs.profiler->GetSelected(), sizeof(TProfile) );

	//Get post processing options from config
	FromConfigToPPost(&opt_run.prof, &opt_run.pp);

    RefreshListBox(hDlg);
    RefreshTabsFromProfile(hDlg);
    return;
}

void RetrieveProfile( HWND hDlg )
{


    	//Video options
        opt_run.pp.resWidth  = DlgGetText( video, IDC_WIDTH);
        opt_run.pp.resHeight = DlgGetText( video, IDC_HEIGHT);

    //iDCT options
    if( opt_run.conf.idctAutoSelect = DlgIsChecked( video, R_OPT_IDCT_AUTOSELECT ) )
      opt_run.conf.idctIndex = rs.video->SelectIdctDefault();


		DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK, opt_run.prof.deinterlace);
		DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND, opt_run.prof.blend);

        //Retrieve timebase
		GetFrameRate( video, &opt_run.prof.timeBase );
		// opt_run.prof.secondsToCompile = opt_run.opt.seconds;
		opt_run.conf.framesToCompile  = opt_run.opt.frames_compile;
		opt_run.prof.recons_progressive = DlgIsChecked(video, R_OPT_TIMEBASE_RECONS);

		opt_run.pp.deinterlace   = DlgIsChecked(video, R_OPT_VIDEO_DEINTERLACE_CHECK);
		opt_run.pp.blend_fields  = DlgIsChecked(video, R_OPT_VIDEO_DEINTERLACE_BLEND);
		opt_run.pp.threshold     = DlgGetText(video, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
		//Audio options
		    if(DlgIsChecked(audio, R_OPT_AUDIO_DSC))
				opt_run.prof.audioMode=DSC;
			else if(DlgIsChecked(audio, R_OPT_AUDIO_DOAUDIO)){
				    opt_run.prof.audioMode=DO_AUDIO;
        			if(DlgIsChecked(audio, IDC_48000)) opt_run.prof.outSampleFrequency=48000;
		        	else opt_run.prof.outSampleFrequency=44100;
        			if(DlgIsChecked(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT)) opt_run.prof.sampleFreqSameAsInput=true;
		        	else opt_run.prof.sampleFreqSameAsInput=false;
			    }
				else 
                    opt_run.prof.audioMode=NO_AUDIO;

        //Processing
		//Files
		DlgGetText( files, IDC_OUTPUT_FILE1, opt_run.conf.outputFile);	
		DlgGetText( files, IDC_OUTPUT_FILE2, opt_run.conf.audioOutFile);
		opt_run.conf.warn_overwrite = DlgIsChecked(files, R_FILES_WARN);
		
		//General
		opt_run.conf.shutdown = DlgIsChecked(general, R_GENERAL_MISC_SHUTDOWN);
		if(DlgIsChecked(general,IDC_COMPILE_WHOLE)){ //If we don't have the CompileWhole
			opt_run.conf.framesToCompile = 3000000; // at least 25hrs at 30fps
		}
		else
		{
			opt_run.conf.framesToCompile =  opt_run.opt.frames_compile;
		}
    opt_run.conf.lurk_size = DlgGetText( general, IDC_LURKSIZE );

        // Post Processing
	    FromPPostToConfig(&opt_run.prof, &opt_run.pp);
        // Right now, all the data from the current options is in opt_run.prof
        return;
}

// Mesage handler for output options
LRESULT CALLBACK OutputOptionsDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	
	static LOGFONT logFont;
    char szTemp[MAX_PATH];

	switch (message)
	{
		case WM_INITDIALOG:
      // Get selected profile
      memcpy( &opt_run.prof, (TProfile *)lParam, sizeof(TProfile) );
      // Get curretn configurations
      memcpy( &opt_run.conf, &rs.conf, sizeof(TConfig));

			//Get post processing options from config
			FromConfigToPPost(&opt_run.prof, &opt_run.pp);
      opt_run.pp.nInWidth = 640;
      opt_run.pp.nInHeight = 480;

			strcpy(logFont.lfFaceName,"MS Sans Serif");
			logFont.lfWeight=FW_NORMAL;
			logFont.lfHeight=8; 
			myFont=CreateFontIndirect(&logFont);
			TabbedDialogInit(hDlg);
			

      RefreshListBox(hDlg);

			// SET LANGUAGE
			DlgSetText(hDlg, IDOK,GS(OPT_OK));
			DlgSetText(hDlg, IDCANCEL,GS(OPT_CANCEL));
			DlgSetText(hDlg, IDC_SHOWPAD ,GS(OPT_SHOWPAD));
			DlgSetText(hDlg, IDC_1, GS(OPT_SHOWPAD_EXP));
			SetText(hDlg, GS(OPT_TITLE));
			// Setting Tabs
			TabSetText(pHdr->hwndTab, 0, GS(OPT_VIDEO) );
			TabSetText(pHdr->hwndTab, 1, GS(OPT_AUDIO) );
			TabSetText(pHdr->hwndTab, 2, GS(OPT_PPOST) );
			TabSetText(pHdr->hwndTab, 3, GS(OPT_FILES) );
			TabSetText(pHdr->hwndTab, 4, GS(OPT_GENERAL) );
			// Video
			DlgSetText(video, R_OPT_FRAMESIZE, GS(OPT_FRAMESIZE));
			DlgSetText(video, R_OPT_WIDTH, GS(OPT_WIDTH));
			DlgSetText(video, R_OPT_HEIGHT, GS(OPT_HEIGHT));
			DlgSetText(video, R_OPT_FRAMESIZE_EXP, GS(OPT_FRAMESIZE_EXP));
			DlgSetText(video, R_OPT_OUTPAD_EXP, GS(OPT_OUTPAD_EXP));
			DlgSetText(video, R_OPT_TIMEBASE, GS(OPT_TIMEBASE));
			DlgSetText(video, R_OPT_TIMEBASE_RECONS, GS(R_OPT_TIMEBASE_RECONS) );
			DlgSetText(video, R_OPT_TIMEBASE_EXP, GS(OPT_TIMEBASE_EXP));
			DlgSetText(video, R_OPT_IDCT_TITLE, GS(OPT_IDCT_TITLE));
			DlgSetText(video, R_OPT_IDCT_EXP, GS(OPT_IDCT_EXP));
			DlgSetText(video, R_OPT_VIDEO_NOTE, GS(OPT_VIDEO_NOTE));
			
			DlgSetText(video, R_OPT_VIDEO_VIDEOSTRUCTURE, GS(R_OPT_VIDEO_VIDEOSTRUCTURE));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_CHECK, GS(R_OPT_VIDEO_DEINTERLACE_CHECK));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_BLEND, GS(R_OPT_VIDEO_DEINTERLACE_BLEND));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_THRESHOLD_CAPTION, GS(R_OPT_VIDEO_DEINTERLACE_THRESHOLD_CAPTION));
			
			// Audio
			DlgSetText(audio, R_OPT_AUDIO_AUDIOMODE_EXP, GS(OPT_AUDIO_AUDIOMODE_EXP));
			DlgSetText(audio, R_OPT_AUDIO_AUDIOMODE, GS(OPT_AUDIO_AUDIOMODE));
			DlgSetText(audio, R_OPT_AUDIO_DSC, GS(OPT_AUDIO_DSC));
			DlgSetText(audio, R_OPT_AUDIO_DOAUDIO, GS(OPT_AUDIO_DOAUDIO));
			DlgSetText(audio, R_OPT_AUDIO_NOAUDIO, GS(OPT_AUDIO_NOAUDIO));
			DlgSetText(audio, R_OPT_AUDIO_DSC_EXP, GS(OPT_AUDIO_DSC_EXP));
			DlgSetText(audio, R_OPT_AUDIO_DOAUDIO_EXP, GS(OPT_AUDIO_DOAUDIO_EXP));
			DlgSetText(audio, R_OPT_AUDIO_NOAUDIO_EXP, GS(OPT_AUDIO_NOAUDIO_EXP));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_EXP, GS(OPT_AUDIO_SAMPLING_EXP));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_TITLE, GS(OPT_AUDIO_SAMPLING_TITLE));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT, GS(OPT_AUDIO_SAMPLING_SAMEASINPUT));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT_EXP, GS(OPT_AUDIO_SAMPLING_SAMEASINPUT_EXP));
			// Post Processing
			DlgSetText(processing, R_OPT_PPOST_RESIZING_TITLE  , GS(R_OPT_PPOST_RESIZING_TITLE ));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_NEAREST, GS(R_OPT_PPOST_RESIZING_NEAREST));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_BILINEAR, GS(R_OPT_PPOST_RESIZING_BILINEAR));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_BICUBIC, GS(R_OPT_PPOST_RESIZING_BICUBIC));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_HQBICUBIC, GS(R_OPT_PPOST_RESIZING_HQBICUBIC));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_NOTE, GS(R_OPT_PPOST_RESIZING_NOTE));
			DlgSetText(processing, R_OPT_PPOST_CROP_TITLE, GS(R_OPT_PPOST_CROP_TITLE));
			DlgSetText(processing, IDC_NOCROP  , GS(IDC_NOCROP ));
			DlgSetText(processing, IDC_NOLETTERBOX, GS(IDC_NOLETTERBOX));
			DlgSetText(processing, R_OPT_PPOST_CROP_TOPOFFSET, GS(R_OPT_PPOST_CROP_TOPOFFSET));
			DlgSetText(processing, R_OPT_PPOST_CROP_HEIGHT, GS(R_OPT_PPOST_CROP_HEIGHT));
			DlgSetText(processing, R_OPT_PPOST_CROP_LEFTOFFSET, GS(R_OPT_PPOST_CROP_LEFTOFFSET));
			DlgSetText(processing, R_OPT_PPOST_CROP_WIDTH, GS(R_OPT_PPOST_CROP_WIDTH));
			DlgSetText(processing, R_OPT_PPOST_CROP_TOP  , GS(R_OPT_PPOST_CROP_TOP ));
			DlgSetText(processing, R_OPT_PPOST_CROP_BOTTOM, GS(R_OPT_PPOST_CROP_BOTTOM));
			DlgSetText(processing, R_OPT_PPOST_CROP_LEFT, GS(R_OPT_PPOST_CROP_LEFT));
			DlgSetText(processing, R_FILES_AUDIOBOX_TITLE, GS(R_FILES_AUDIOBOX_TITLE));
			DlgSetText(processing, R_OPT_PPOST_CROP_RIGHT, GS(R_OPT_PPOST_CROP_RIGHT));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT, GS(R_OPT_PPOST_KEEPASPECT));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT_EXP, GS(R_OPT_PPOST_KEEPASPECT_EXP));
			DlgSetText(processing, R_OPT_PPOST_SHOWPAD_EXP, GS(R_OPT_PPOST_SHOWPAD_EXP));
			DlgSetText(processing, R_OPT_PPOST_CROP_SETTINGS_EXP, GS(R_OPT_PPOST_CROP_SETTINGS_EXP));
			DlgSetText(processing, R_OPT_PPOST_CROP_LETTERBOX_EXP, GS(R_OPT_PPOST_CROP_LETTERBOX_EXP));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT, GS(R_OPT_PPOST_KEEPASPECT));
			DlgSetText(processing, IDC_SHOWPAD, GS(OPT_SHOWPAD));
			// Files
			DlgSetText(files, R_FILES_OUTFILES_TITLE  , GS(R_FILES_OUTFILES_TITLE ));
			DlgSetText(files, R_FILES_VIDEOBOX_TITLE, GS(R_FILES_VIDEOBOX_TITLE));
			DlgSetText(files, R_FILES_OUTVIDEO_SEX, GS(R_FILES_OUTVIDEO_SEX));
			DlgSetText(files, R_FILES_AUDIOBOX_TITLE, GS(R_FILES_AUDIOBOX_TITLE));
			DlgSetText(files, R_FILES_AUDIOBOX_EXP, GS(R_FILES_AUDIOBOX_EXP));
			DlgSetText(files, R_FILES_WARN, GS(R_FILES_WARN));
			DlgSetText(files, IDC_FILEBUTTON1, GS(1113));
			DlgSetText(files, IDC_FILEBUTTON2, GS(1113));
			// General
			DlgSetText(general, R_GENERAL_COMPTIME_TITLE  , GS(R_GENERAL_COMPTIME_TITLE ));
			DlgSetText(general, R_GENERAL_COMPTIME_SECONDSCOMPILE, GS(R_GENERAL_COMPTIME_SECONDSCOMPILE));
			DlgSetText(general, R_GENERAL_COMPTIME_FRAMESCOMPILE, GS(R_GENERAL_COMPTIME_FRAMESCOMPILE));
			DlgSetText(general, IDC_COMPILE_WHOLE  , GS(IDC_COMPILE_WHOLE ));
			DlgSetText(general, R_GENERAL_COMPTIME_COMPTIME_EXP1, GS(R_GENERAL_COMPTIME_COMPTIME_EXP1));

			// DlgSetText(general, R_GENERAL_COMPTIME_COMPTIME_EXP2, GS(R_GENERAL_COMPTIME_COMPTIME_EXP2));
			DlgSetText(general, R_GENERAL_SEARCHSIZE_TITLE, GS(R_GENERAL_SEARCHSIZE_TITLE));			
			DlgSetText(general, R_GENERAL_SEARCHSIZE_BOX, GS(R_GENERAL_SEARCHSIZE_BOX));
			DlgSetText(general, R_GENERAL_SEARCHSIZE_EXP, GS(R_GENERAL_SEARCHSIZE_EXP));
			DlgSetText(general, R_GENERAL_MISC_TITLEIDC_STATIC, GS(R_GENERAL_MISC_TITLEIDC_STATIC));
			DlgSetText(general, R_GENERAL_MISC_SHUTDOWN, GS(R_GENERAL_MISC_SHUTDOWN));			


			//Load from config
			//Audio output
			//If its not an AC3 disable DO_AUDIO
/*			if(!audio->isAC3){
				DlgDisable(hDlg, IDC_DOAUDIO);
				if(opt_run.prof.audioMode==DO_AUDIO) opt_run.prof.audioMode=DSC;
			}
			else{
				DlgEnable(hDlg, IDC_DOAUDIO);
			}*/


			return TRUE;
		case WM_NOTIFY:
			switch(((LPNMHDR)lParam)->code){
				case TCN_SELCHANGE:
					/* is all data valid? */
					if(pHdr->DlgTabCheck[opt_run.conf.selectedTab] &&
						!pHdr->DlgTabCheck[opt_run.conf.selectedTab](pHdr->hwndDisplay)) {
						/* there's a problem; set the focused tab back */
						TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.conf.selectedTab);
					    return FALSE;
					}

					/* we're OK */
					OnSelChanged(hDlg);

					break;
			}
			break;
		case WM_COMMAND:
			//Notifications
			switch(LOWORD(wParam))
			{
				case IDOK:
					/* is all data valid? */
					if(pHdr->DlgTabCheck[opt_run.conf.selectedTab] &&
						!pHdr->DlgTabCheck[opt_run.conf.selectedTab](pHdr->hwndDisplay)) {
						/* there's a problem */
					    return FALSE;
					}

          RetrieveProfile(hDlg);

          rs.profiler->AddNSelect( &opt_run.prof );

          // Set Global Run State profile from selected one
          rs.prof = *rs.profiler->GetSelected();

          // Save options
          memcpy( &rs.conf, &opt_run.conf, sizeof(TConfig));

          rs.conf.selected_profile = rs.profiler->GetSelectedIndex();


					DeleteObject(myFont);
					EndDialog(hDlg, LOWORD(wParam));
					return TRUE;
				break;

				case R_OPT_NEWPROFILE_BUTTON:
                //case R_OPT_SAVECURRENT_BUTTON:

					// Changed by edwin van eggelen 12-05-2001

					// Current settings are stored into profile
                    if (DialogBoxParam( hInst, 
                                        MAKEINTRESOURCE(IDD_OUTPUT_OPTIONS_PROFILE_NAME),
                                        hDlg, 
                                        (DLGPROC)OptionsProfileNameDlg,
                                        (LPARAM)szTemp) == IDOK)
                    {
                        //The string is in szTemp
                        RetrieveProfile(hDlg);
                        strcpy( opt_run.prof.profile_name, szTemp);
                        rs.profiler->AddProfile( &opt_run.prof );

						rs.profiler->Select( rs.profiler->FindIndexWithName(szTemp));
                        RefreshOptionsDialog(hDlg);
                    }
                    break;

				case R_OPT_SAVECURRENT_BUTTON:

					{

						int selected_item;

					    // Changed by edwin van eggelen 12-05-2001

					    // Current settings are stored into profile

                        RetrieveProfile(hDlg);

						//memcpy( rs.profiler->GetSelected(), &opt_run.prof,sizeof(TProfile) );

                        selected_item=rs.profiler->GetSelectedIndex();

						rs.profiler->DeleteProfile(rs.profiler->GetSelectedIndex());

                        rs.profiler->AddProfile(&opt_run.prof);

						//select previous select profile

						rs.profiler->Select(selected_item);

						RefreshOptionsDialog(hDlg);

					}	                        

					    break;


                case R_OPT_DELETEPROFILE_BUTTON:
                        //Delete selected profile
                        rs.profiler->DeleteProfile( ListGetCur(GetDlgItem(hDlg, IDC_PROFILES_LIST)) );
                        RefreshOptionsDialog(hDlg);
						break;


				case IDC_SHOWPAD:
					/* is all data valid? */
					if(pHdr->DlgTabCheck[opt_run.conf.selectedTab] &&
						!pHdr->DlgTabCheck[opt_run.conf.selectedTab](pHdr->hwndDisplay)) {
						/* there's a problem; set the focused tab back */
						TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.conf.selectedTab);
					    return FALSE;
					}
          {
            COutputPadDlg obOutputPadDlg( hDlg, hDlg, hInst );
            //obOutputPadDlg.Show( &opt_run.pp );
            DlgProcessingStartUp(hDlg);
            DlgVideoStartUp(video);
          }
					break;

				case IDCANCEL:
					EndDialog(hDlg, LOWORD(wParam));
					return TRUE;
				break;
				
			}
            if(HIWORD(wParam)==CBN_SELCHANGE){
                rs.profiler->Select( ListGetCur( GetDlgItem(hDlg, IDC_PROFILES_LIST)) );
				RefreshOptionsDialog( hDlg );

				return TRUE;
			}
			break;
	}
    return FALSE;
}

LRESULT CALLBACK OptionsProfileNameDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    static char  *string_buffer=NULL;

	switch (message)
	{
		case WM_INITDIALOG:
                string_buffer = (char *) lParam;
				return TRUE;

		case WM_COMMAND:

            DlgGetText(hDlg, IDC_EDITBOX, string_buffer );

			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
            }
			break;
	}
    return FALSE;
}
