/* 
 *  Crop.cpp 
 *                                     
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "letterbox.h"
#include "Auxiliary.h"
#include "Debug.h"


void LetterboxPlanar( Pixel8 *dst, int width, int height, TLetterboxConfig *config, Pixel8 color)
{
  if(!dst)
    return;
  
  Pixel8  *buf;
  Pixel32 *buf4;
  Pixel32 color32 = color<<24 | color<<16 | color<<8 | color;

  int ww, wr;

  if(config->top)
  {
    memset( dst, color, config->top * width );
  }

  if( config->bottom )
  {
    memset( dst+width *(height - config->bottom), color, config->bottom*width);
  }
  if(config->left)
  {
    int h = height; 
    int mod = width - config->left;

    buf4 = (Pixel32 *)dst;
    while(h--)
    {
      ww = config->left/4; // 4 byte chunks
      wr = config->left%4; // remainder
      // 4 byte
      while(ww--)
        *buf4++ = color32;

      // single byte
      buf = (Pixel8 *)buf4;
      while(wr--)
        *buf++ = color;
      buf4 = (Pixel32 *)(buf + mod);
    }
  }
  if(config->right)
  {
    int h = height; 
    int mod = width - config->right;
    
    buf = dst + mod;
    while(h--)
    {
      ww = config->right/4; // 4 byte chunks
      wr = config->right%4; // remainder

      // single byte
      while(wr--)
        *buf++ = color;

      buf4 = (Pixel32 *)buf;
      // 4 byte
      while(ww--)
        *buf4++ = color32;

      buf = (Pixel8 *)(buf4) + mod;
    }
  }
}


int LetterboxYV12(CFrame *fr, TLetterboxConfig *config) 
{
  TLetterboxConfig sUVConfig;

  int sw = fr->GetWidth();
  int sh = fr->GetHeight();

  int shw = sw>>1;
  int shh = sh>>1;

  int sysize = sw*sh;
  int scrsize = shw * shh;

  sUVConfig.bottom = config->bottom >> 1;
  sUVConfig.top    = config->top    >> 1;
  sUVConfig.left   = config->left   >> 1;
  sUVConfig.right  = config->right  >> 1;

  Pixel8 *pSrc = (Pixel8 *)fr->GetBuffer();


  LetterboxPlanar( pSrc, sw, sh,  config, 0);
  // V
  LetterboxPlanar( pSrc + sysize, 
                   shw, shh,  &sUVConfig, 128 );
  // U
  LetterboxPlanar( pSrc + sysize + scrsize,
                   shw, shh,  &sUVConfig, 128 );


  return 0;
}

int Letterbox(CFrame *source, TLetterboxConfig *config) 
{
  if(!source )
    return 0;

  switch( source->GetFormat() )
  {
  case FRAME_YV12:
    return LetterboxYV12( source, config);
    break;
  default:
    break;
  }
  return 1;
}


int StopCrop(TLetterboxConfig *config) {

	return 1;
}

FlLetterbox::FlLetterbox()
{
  memset( &m_cfg, 0, sizeof m_cfg );
  m_bConfigured = false;
}

int FlLetterbox::Configure( void *conf, int confsize)
{
  
  FLASSERT( confsize==sizeof(TLetterboxConfig) )
  TLetterboxConfig *cf = (TLetterboxConfig *)conf;
  
  m_cfg = *cf;
  m_bConfigured = true;
  
  return flfil_ok;
}

int FlLetterbox::GetFilterConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlLetterbox::GetFilterConf - You need to configure first\n"))
      return 0;
  }
  
  if(!fc)
    return 0;
  
  *fc = m_fc;
  
  return 1;
}

int FlLetterbox::ValFilterConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlLetterbox::ValFilterConf - You need to configure first\n"))
      return flfil_error;
  }
  
  if( m_cfg.left  > fc->iw/2  ||
      m_cfg.right > fc->iw/2  ||
      m_cfg.bottom > fc->ih/2 ||
      m_cfg.top    > fc->ih/2 ||
      m_cfg.left  < 0         ||
      m_cfg.right < 0         ||
      m_cfg.bottom< 0         ||
      m_cfg.top   < 0 ) {
    DBG_STR((str, "FlLetterbox::ValFilterConf - Bad Configuration\n"))
    return flfil_error;
  }
  fc->olag = 0;
  fc->op = 1;

  fc->od = fc->id;
  fc->ow = fc->iw;
  fc->oh = fc->ih;
  fc->oprovided = 0;

  fc->ocanmodify = 1;
  
  m_fc = *fc;
  
  return flfil_ok;
}

int FlLetterbox::StartSimple()
{
  return m_bConfigured == true ? 1 : 0;
}

int FlLetterbox::ProcessSimple( CFrame *in,  CFrame *out )
{
  FLASSERT( out==NULL )
  return Letterbox( in, &m_cfg );
}

int FlLetterbox::StopSimple()
{
  return 1;
}
