/* 
 *  flfilterchain.cpp  
 *                    
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "debug.h"
#include "flfilterchain.h"


FlFilterChain::FlFilterChain()
{
  m_frc = 0;
  m_bConfigured = false;
}

int FlFilterChain::Start()
{

  flfilter_conf fc;

  if(!m_bConfigured)
  {
    DBG_STR((str, "FlFilterChain::Start - You need to configure first\n"))
    return 0;
  }


  for( int i=0; i<m_filc; i++ )
  {
    if(!m_fil[i]->Start())
    {
      DBG_STR((str, "FlFilterGraph::Start - %d filter failed to start", i))
      return 0;
    }

    // if this is not the last filter, create a buffer for output
    if( i<m_filc-1 )
    {
      // retrieve current configuration of filter
      m_fil[i]->GetConf( &fc );

      // allocate frame
      m_fr[m_frc] = new CFrame();
      m_fr[m_frc++]->Set( fc.ow, fc.oh, fc.iformat );
    }
  }

  return 1;
}

int FlFilterChain::Configure( void *conf, int confsize )
{
  FlFilter **filters = ((TFilterChainCfg *)conf)->filters;
  int filtercnt = ((TFilterChainCfg *)conf)->filtercnt;
  
  m_filc =0;

  for( int i=0; i<filtercnt; i++ )
  {
    // store filter
    m_fil[i] = filters[i];
    m_filc++;
  }

  m_bConfigured = true;
  return flfil_ok;
}

int FlFilterChain::GetConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlFilterChain::GetConf - You need to configure first\n"))
    return 0;
  }

  if(!fc)
    return 0;

  *fc = m_fc;

  return 1;
}

int FlFilterChain::Validate( flfilter_conf *fc )
{
  flfilter_conf myfs;

  if(!m_bConfigured)
  {
    DBG_STR((str, "FlFilterChain::Validate - You need to configure first\n"))
    return 0;
  }
  
  fc->lag = 0;
  fc->od = fc->id;
  fc->ow = fc->iw;
  fc->oh = fc->ih;
  fc->op = 0;

  for( int i=0; i<m_filc; i++ )
  {
    myfs.iw = fc->ow;
    myfs.ih = fc->oh;
    myfs.id = fc->od;
    myfs.iformat = fc->iformat;

    if(!m_fil[i]->Validate(&myfs))
    {
      DBG_STR((str, "FlFilterChain::Validate - %d filter query failed", i))
      return 0;
    }

    fc->lag += myfs.lag;
    fc->od = myfs.od;
    fc->ow = myfs.ow;
    fc->oh = myfs.oh;
    fc->op = 0;
  }

  m_fc = *fc;

  return 1;
}

int FlFilterChain::Process(CFrame *in, CFrame *out)
{
  CFrame *pin[nMaxFilters], *pout[nMaxFilters];

  // If no filters just set the frame
  if( m_filc==0 )
  {
    out->SetFrame( in );
  }
  else
  {
    for( int i=0; i<m_filc; i++ )
    { 
      // Prepare pointers for input and output buffers
      // if first set input to in, else to the output of the previous filter
      pin[i] = i==0 ? in : m_fr[i-1];
      
      // if last set output to out, else to its output
      pout[i] = i==m_filc-1 ? out : m_fr[i];
      
      // Process
      m_fil[i]->Process(pin[i], pout[i]);
    }
  }

  return 0;
}

int FlFilterChain::Stop()
{
  // delete temporary buffers
  for( int i=0; i<m_frc; i++ )
  {
    delete m_fr[i];
    m_fr[i] = NULL;
  }
  m_frc = 0;
  m_bConfigured = false;
  return 0;
}
