/* 
 *  flcroppad.cpp
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "flcroppad.h"
#include "debug.h"
 

#define AREA_SIZE 36


FlCropPad::FlCropPad()
{
  m_clx = m_cly = 0;
  m_bLocked = false;
  m_nLockedType = 0;
}

bool FlCropPad::SetConfig( int nWidth, int nHeight, TCropConfig *cfg )
{
  m_nWidth = nWidth;
  m_nHeight = nHeight;
  m_cfg = *cfg;
  BuildAreas();
  return true;
}


bool FlCropPad::OnDrag( int x, int y )
{
  bool res = false;
  
  if( m_bLocked )
    res = Notify( x, y, m_nLockedType );

  return res;
}

bool FlCropPad::OnHover( int x, int y, LPCTSTR &mousetype)
{
  for( int i=0; i<9; i++ ) {
    if( m_vAreas[i].Poll( x, y ) ) {
      int type = m_bLocked ? m_nLockedType : m_vAreas[i].GetType();
      switch( type )
      {
        case topleft:
        case bottomright:
          mousetype = IDC_SIZENWSE;
          break;
        case top:
        case bottom:
          mousetype = IDC_SIZENS;
          break;
        case topright:
        case bottomleft:
          mousetype = IDC_SIZENESW;
          break;
        case left:
        case right:
          mousetype = IDC_SIZEWE;
          break;
        case center:
          mousetype = IDC_SIZEALL;
          break;
      }
      return true;
    }
  }
  return false;
}


bool FlCropPad::BuildAreas()
{ 
  int w = m_nWidth;
  int h = m_nHeight;

  int b = m_cfg.bottom;
  int l = m_cfg.left;
  int r = m_cfg.right;
  int t = m_cfg.top;

  // corners
  m_vAreas[0].Set( l, t, AREA_SIZE, AREA_SIZE, topleft );
  m_vAreas[1].Set( r, t, AREA_SIZE, AREA_SIZE, topright );
  m_vAreas[2].Set( l, b, AREA_SIZE, AREA_SIZE, bottomleft );
  m_vAreas[3].Set( r, b, AREA_SIZE, AREA_SIZE, bottomright );

  // sides
  int sidew = (r - l) - AREA_SIZE;
  m_vAreas[4].Set( (l+r)/2, t, sidew, AREA_SIZE, top );
  m_vAreas[5].Set( (l+r)/2, b, sidew, AREA_SIZE, bottom );

  int sideh = (b - t) - AREA_SIZE;
  m_vAreas[6].Set( l, (b+t)/2, AREA_SIZE, sideh, left );
  m_vAreas[7].Set( r, (b+t)/2, AREA_SIZE, sideh, right );

  // center
  m_vAreas[8].Set( (l+r)/2, (b+t)/2, sidew, sideh, center );

  return true;
}

int FlCropPad::SnapCoordinate( int c, int oldc )
{
  int rem, res;

  if( c >= oldc ) {
    // calculate the difference with the old
    //  point
    rem = (c-oldc)%16;
    res = rem < 8 ? c-rem : c+(16-rem);
  }
  else { // c< oldc
    rem = (oldc -c)%16;
    res = rem < 8 ? c+rem : c-(16-rem);
  }

  return res;
}

bool FlCropPad::OnClick( int x, int y )
{
  bool bSuccess = false;
  m_clx = x;
  m_cly = y;
  for( int i=0; i<9; i++ ) {
    if( m_vAreas[i].Poll( x, y ) ) {
      m_bLocked = true;
      m_nLockedType = m_vAreas[i].GetType();
      bSuccess = true;
    }
  }
  return bSuccess;
}

bool FlCropPad::OnRelease()
{
  m_bLocked = false;
  return true;
}

bool FlCropPad::Notify( int x, int y, int nType )
{
  int w = m_nWidth;
  int h = m_nHeight;
  int xoff, yoff;

  TCropConfig cfg;

  // clip x, y first
  //x = x<0 ? 0 : x>w ? w : x;
  //y = y<0 ? 0 : y>h ? h : y;

  cfg = m_cfg;
  switch( nType )
  {
    case topleft:
      cfg.top = SnapCoordinate(y, cfg.top);
      cfg.left = SnapCoordinate(x, cfg.left);
      break;
    case top:
      cfg.top = SnapCoordinate(y, cfg.top);
      break;
    case topright:
      cfg.right = SnapCoordinate(x, cfg.right);
      cfg.top = SnapCoordinate(y, cfg.top);
      break;
    case left:
      cfg.left = SnapCoordinate(x, cfg.left);
      break;
    case center:
      xoff = x - m_clx;
      yoff = y - m_cly;

      if( (cfg.right - cfg.left) == w )
        xoff = 0;

      if( (cfg.bottom - cfg.top) == w )
        yoff = 0;

      cfg.left += xoff;
      cfg.right += xoff;
      cfg.top += yoff;
      cfg.bottom += yoff;

      m_clx = x;
      m_cly = y;
      
      break;
    case right:
      cfg.right = SnapCoordinate(x, cfg.right);
      break;
    case bottomleft:
      cfg.bottom = SnapCoordinate(y, cfg.bottom);
      cfg.left = SnapCoordinate(x, cfg.left);
      break;
    case bottom:
      cfg.bottom = SnapCoordinate(y, cfg.bottom);
      break;
    case bottomright:
      cfg.right = SnapCoordinate(x, cfg.right);
      cfg.bottom = SnapCoordinate(y, cfg.bottom);
      break;
  }

  FlCrop oCrop;

  oCrop.Configure( &cfg, sizeof (TCropConfig) );

  flfilter_conf fc;
  memset( &fc, 0, sizeof flfilter_conf );
  fc.iw = w;
  fc.ih = h;

  // Validate configuration
  if( oCrop.ValFilterConf(&fc) ) 
  {
    // if nothing to do
    if( memcmp(&cfg, &m_cfg, sizeof TCropConfig)==0 )
      return false;

    DBG_STR((str, "Crop pad changed settings\n"))
    m_cfg = cfg;
    BuildAreas();
    return true;
  }
  return false;
}