/* 
 *  flbrightness.cpp 
 *                                     
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "flbrightness.h"
#include "Debug.h"


static const __int64 mmmask_0064 = 0x0040004000400040;

static void Luminance_Filter(unsigned char *src, int area, int LumGain, int LumOffset )
{
	__int64 LumGainMask = ((__int64)LumGain<<48) + 
                        ((__int64)LumGain<<32) + 
                        ((__int64)LumGain<<16) + 
                         (__int64)LumGain;

	__int64 LumOffsetMask = ((__int64)LumOffset<<48) + 
                          ((__int64)LumOffset<<32) + 
                          ((__int64)LumOffset<<16) + 
                           (__int64)LumOffset;

	__asm
	{
		mov			eax, [src]
		mov			esi, 0x00
		mov			edi, [area]
		pxor		mm0, mm0
		movq		mm5, [LumOffsetMask]
		movq		mm6, [LumGainMask]
		movq		mm7, [mmmask_0064]

lumconv:
		movq		mm1, [eax+esi]
		movq		mm2, mm1

		punpcklbw	mm1, mm0
		punpckhbw	mm2, mm0

		;pmullw		mm1, mm6
		;pmullw		mm2, mm6

		;paddw		mm1, mm7
		;paddw		mm2, mm7

		;psrlw		mm1, 7
		;psrlw		mm2, 7

		paddw		mm1, mm5
		paddw		mm2, mm5

		packuswb	mm1, mm0
		packuswb	mm2, mm0

		add			esi, 0x08
		cmp			esi, edi
		movd		[eax+esi-8], mm1
		movd		[eax+esi-4], mm2
		jl			lumconv
	}
}


FlBrightness::FlBrightness()
{
  memset( &m_cfg, 0, sizeof m_cfg );
  m_bConfigured = false;
}

void FlBrightness::Luminance(CFrame *in) {
  switch( in->GetFormat() ) {
    case FRAME_YV12:
      Luminance_Filter( in->GetBuffer(), in->GetWidth()*in->GetHeight(), 128, m_cfg.brightness-128 );
      break;
    default:
      break;
  }
}

int FlBrightness::StartSimple()
{
  return m_bConfigured == true ? 1 : 0;
}

int FlBrightness::Configure( void *conf, int confsize)
{
  FLASSERT( confsize==sizeof(TBrightnessCfg) )
  TBrightnessCfg cfg = *(TBrightnessCfg *)conf;

  m_cfg = cfg;
  m_bConfigured = true;
  return flfil_ok;
}

int FlBrightness::GetFilterConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlBrightness::GetConf - You need to configure first\n"))
    return 0;
  }

  if(!fc)
    return 0;

  *fc = m_fc;

  return 1;
}

int FlBrightness::ValFilterConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlBrightness::ValFilterConf - You need to configure first\n"))
    return flfil_error;
  }
  
  if( m_cfg.brightness < 0 || m_cfg.brightness>255) {
    DBG_STR((str, "FlBrightness::ValFilterConf - Bad configuration\n"))
    return flfil_error;
  }
  
  if( fc->iw%8 != 0 ) {
    DBG_STR((str, "FlBrightness::ValFilterConf - Width must be multiple of 8\n"))
    return flfil_error;
  }

  fc->od = fc->id;
  fc->ow = fc->iw;
  fc->oh = fc->ih;
  fc->oprovided = 0;
  fc->op = 1;
  fc->ocanmodify = 1;

  m_fc = *fc;

  return 1;
}

int FlBrightness::ProcessSimple( CFrame *in,  CFrame *out )
{
  FLASSERT( out==NULL )
  Luminance( in );
  return true;
}

int FlBrightness::StopSimple()
{
  return 1;
}