// VideoSource.cpp: implementation of the CVideoSource class.
//
//////////////////////////////////////////////////////////////////////

#include "VideoSource.h"
#include "postprocessing.h"
#include "debug.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction

CVideoSource::CVideoSource()
{
  m_nWidth = m_nHeight = 0;
}

//  
// Actual running thread

ui32 CVideoSource::GetFrame(CFrame **ppFrame)
{
  // If there are more than 8 frames to be
  // processed in the queue
  // awake the decoder to start working
  if(m_pProcessedQueue->GetSize()>=8)
    ScheduleCommand(getFrame);

  DBG_STR((str, "%d ", m_pProcessedQueue->GetSize() ));

  if(!(*ppFrame = m_pOutputQueue->RemoveFrame()))
    return FRAME_NOTPRESENT;
  if((*ppFrame)->IsLastFrame())
    return FRAME_NOTPRESENT;

  return FRAME_OK;
}

bool CVideoSource::Start( CFrameSource *pSource, TPPost *pp )
{
  m_pFrameSource = pSource; 

  if( !pp )
    return false;

  if( !m_pp.Set(pp) )
    return false;
  if( !m_pp.Start() )
    return false;

  m_nWidth = m_pp.GetWidth();
  m_nHeight = m_pp.GetHeight();

  m_pProcessedQueue = new CListFrameBuffer( m_nWidth, m_nHeight, pp->nProcessingFormat, 10);
  
  m_pOutputQueue    = new CQueueFrameBuffer();
  
  ScheduleCommand(noCommand);
  
  Create();
  return true;
}

bool CVideoSource::Stop()
{
  // Terminate the thread
  Exit();
  
  m_pp.Stop();

  if( m_pProcessedQueue )
  {
    delete m_pProcessedQueue;
    m_pProcessedQueue = NULL;
  }
    
  
  if( m_pOutputQueue )
  {
    delete m_pOutputQueue;
    m_pOutputQueue = NULL;
  }

  return true;
}

DWORD CVideoSource::ThreadProc()
{
  m_nState = stStopped;

  TCommand sCommand;
  CFrame *pOutputFrame;
  CFrame *pDecodedFrame;

  while(1)
  {
    if(GetCommand(&sCommand))
    {
      switch( sCommand.nCommand )
      {
      case noCommand:
        break;
      case getFrame:
        m_nState = stGetFrame;
        break;
      case exit:
        // Exit
        ReplyCommand(&sCommand);
        return 0;
        break;
      }
      ReplyCommand(&sCommand);
    }
    else
    {
      // If we are stopped, wait for a command
      if(m_nState==stStopped)
        WaitCommand();
    }
    switch( m_nState )
    {
    case stGetFrame:
      
      // If the queue is almost empty means that you've processed
      // all those.
      // stop
      if(m_pProcessedQueue->GetSize()<=2)
      {
        m_nState = stStopped;
        break;
      }   

      // Get a free frame from the processed buffer
      if(!(pOutputFrame = m_pProcessedQueue->GetFreeFrame() ))
      {
        m_nState = stStopped;
        break;
      }

      // Get a new frame
      if(m_pFrameSource->GetFrame(&pDecodedFrame))
      {
        m_pp.Process(pDecodedFrame, pOutputFrame);
        // Add time info
        pOutputFrame->SetPresTime( pDecodedFrame->GetPresTime() );
        // Add processed frame to the output queue
        m_pOutputQueue->AddFrame( pOutputFrame );
        // Release decoded
        pDecodedFrame->Release();
      }
      else
      {
        // Likely the last frame in the stream
        pOutputFrame->Erase();
        pOutputFrame->SetLastFrame();

        m_pOutputQueue->AddFrame( pOutputFrame );

      }
      break; 
    }
  }
}