/* 
 *  resizer.cpp  
 *	Copyright (C) Alberto Vigata - ultraflask@yahoo.com - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include <windows.h>
#include <commctrl.h>
#include <crtdbg.h>
#include <assert.h>
#include <stdio.h>
#include <math.h>
#include "..\debug.h"

#include "..\flaskmpeg.h"

#include "resizer.h"

extern HINSTANCE g_hInst;



extern void NearestInitialize(long nSrcWidth, long nSrcHeight, long nDstWidth, long nDstHeight );
extern void NearestRun( Pixel8 *src, Pixel8 *dst, long nSrcWidth, long nSrcHeight, 
                      long nDstWidth, long nDstHeight, long nSrcPitch, long nDestPitch );

extern void BilinearRun( Pixel8 *src, Pixel8 *dst, long nSrcWidth, long nSrcHeight, 
                       long nDstWidth, long nDstHeight, long nSrcPitch, long nDstPitch  );



#define SAR(DAR,hor,vert) (DAR*((double)hor/(double)vert))


// xoffset, yoffset referes to the offset from the source picture
// we assume src width = src pitch
// xoffset, yoffset = offset where to put the resized image in the dest frame
// rdw = resized destination width
// rdh = resized destination height
void ResizerRunYV12( Pixel8 *src, Pixel8 *dst, 
                     int src_w, int src_h, int dst_w, int dst_h, 
                     int xoffset, int yoffset,
                     int rdw, int rdh )
{
  int srcpitch = src_w;
  int dstpitch = dst_w;

  int src_size = src_h * srcpitch;
  int chroma_srcsize = src_size >> 2;

  int dst_size = dst_h * dstpitch;
  int chroma_dstsize = dst_size>>2;
 
  int sw = src_w;
  int sh = src_h;
  int hsw = sw>>1;
  int hsh = sh>>1;
  int sp  = srcpitch;
  int hsp = sp>>1;

  int dw = rdw;
  int dh = rdh;
  int hdw = dw>>1;
  int hdh = dh>>1;
  int dp = dstpitch;
  int hdp = dp>>1;

  
  
  Pixel8 *sy = src;
  Pixel8 *sv = src + src_size ;
  Pixel8 *su = src + src_size + chroma_srcsize;

  Pixel8 *dy = dst +                                  yoffset * dp  +  xoffset;
  Pixel8 *dv = dst + dst_size                  + (yoffset>>1) * hdp + (xoffset>>1);
  Pixel8 *du = dst + dst_size + chroma_dstsize + (yoffset>>1) * hdp + (xoffset>>1);

/*
  // Y
  NearestRun( src, dst, src_w, src_h, dst_w, dst_h, src_w, dst_w );
  // U
  NearestRun( src + src_size, dst + dst_size, 
              src_w>>1, src_h>>1, dst_w>>1, dst_h>>1, src_w, dst_w );
  // V
  NearestRun( src + src_size + (src_w>>1), dst + dst_size + (dst_w>>1), 
    src_w>>1, src_h>>1, dst_w>>1, dst_h>>1, src_w, dst_w );
*/
  // Y
  BilinearRun( sy, dy, sw, sh, dw, dh, sp, dp );
  // U
  BilinearRun( sv, dv, hsw, hsh, hdw, hdh, hsp, hdp );
  // V
  BilinearRun( su, du, hsw, hsh, hdw, hdh, hsp, hdp );
  
}


FlResize::FlResize()
{
  m_bConfigured = false;
}

int FlResize::Configure(void *conf, int confsize )
{
  FLASSERT( confsize==sizeof(TResizeCfg) )
  TResizeCfg *rc = (TResizeCfg *)conf;

  // check some settings
  if(rc->doar) 
  {
    if( rc->idar>1  || rc->idar<=0 ) 
    {
      rc->idar = 3.0/4.0;
      DBG_STR((str,"FlResize::Configure - incorrect idar. Adjusted to 4:3\n"))
    }

    if( rc->odar>1  || rc->odar<=0 )
    {
      rc->idar = 3.0/4.0;
      DBG_STR((str,"FlResize::Configure - incorrect odar. Adjusted to 4:3\n"))
    }
  }
  if( rc->oheight< 4 ||
      rc->owidth < 4 ){
    m_bConfigured = false;
    return flfil_error;
  }

  m_cfg = *rc;
  m_bConfigured = true;
  return flfil_ok;
}

int FlResize::ValFilterConf(flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlResize::Validate - You need to configure first\n"))
      return flfil_error;
  }
  
  // if input is 0,0 is not valid
  if( fc->iw==0 ||
      fc->ih==0 )
      return flfil_error;

  // the same delay
  fc->od = fc->id;

  // output res
  fc->ow = m_cfg.owidth;
  fc->oh = m_cfg.oheight;

  // no on-place, no lag
  fc->op = 0;
  fc->olag = 0;
  fc->oprovided = 0;
  fc->ocanmodify = 1;
  
  // store this validation data for start
  m_fc = *fc;
  
  return 1;
}

int FlResize::GetFilterConf( flfilter_conf *fc )
{
  if(!m_bConfigured)
  {
    DBG_STR((str, "FlResize::GetConf - You need to configure first\n"))
      return 0;
  }
  
  if(!fc)
    return flfil_error;
  
  *fc = m_fc;
  
  return 1;
}

int FlResize::StartSimple()
{
  double destSAR;
  int ow = m_cfg.owidth;
  int oh = m_cfg.oheight;

  if( m_cfg.doar )
  {
    destSAR = SAR( m_cfg.odar, ow, oh );
    
    
    // This is my aspect ratio code, illustrated with glorious ascii art 
    if ( (double)oh / (double)ow < m_cfg.idar/destSAR ) { 
      // add vert bands
      // first case: input |###| must fit into user frame |_______|, so we will
      // keep the output y value and center x: |__###__|
      m_nResHeight = oh;
      m_nResWidth  = oh * destSAR/m_cfg.idar;
      if (m_nResWidth & 1) 
        m_nResWidth -= 1; // round to multiple of two

      m_nResXOffset = (ow - m_nResWidth) / 2;
      m_nResYOffset = 0;
    }
    else { 
      // add horz bands
      // second case: the input must go into a frame that's too high, 
      // We keep the x value and center y, so
      //           |   |
      //  |###| -> |###|
      //           |   |
      m_nResWidth = ow;
      m_nResHeight = ow * m_cfg.idar/destSAR;
      if (m_nResHeight & 1) 
        m_nResHeight -= 1; // round to multiple of two

      m_nResXOffset = 0;
      m_nResYOffset = (oh - m_nResHeight) / 2;
    }

    m_nResXOffset = m_nResXOffset/2 *2;
    m_nResYOffset = m_nResYOffset/2 *2;

    m_lcfg.top = m_nResYOffset;;
    m_lcfg.bottom = oh - m_nResHeight - m_lcfg.top;
    m_lcfg.left = m_nResXOffset;
    m_lcfg.right = ow - m_nResWidth - m_lcfg.left;

    // start letterbox filter
    m_pLetterbox = new FlLetterbox;
    m_pLetterbox->Configure( &m_lcfg, sizeof TLetterboxConfig );
    flfilter_conf fc;

    fc.iformat = m_fc.iformat;
    fc.iw = m_fc.iw;
    fc.ih = m_fc.ih;
    fc.id = 0;
    fc.icanmodify = 1;
    fc.iprovided = 1;
    fc.input = NULL;

    m_pLetterbox->ValFilterConf( &fc );
    m_pLetterbox->StartSimple();

  }
  else
  {
    m_nResWidth = ow;
    m_nResHeight = oh;

    m_nResXOffset = 0;
    m_nResYOffset = 0;
  }
    
  // TODO
  return flfil_ok;
}

int FlResize::ProcessSimple(CFrame *in, CFrame *out)
{

  int sw = in->GetWidth();
  int sh = in->GetHeight();
  int dw = out->GetWidth();
  int dh = out->GetHeight();
  FLASSERT( sw==m_fc.iw && sh==m_fc.ih &&
            dw==m_fc.ow && dh==m_fc.oh )

  
  Pixel8 *src = (Pixel8 *)in->GetBuffer();
  Pixel8 *dst = (Pixel8 *)out->GetBuffer();

  ResizerRunYV12( src, dst, sw, sh, dw, dh, m_nResXOffset, m_nResYOffset,
                   m_nResWidth, m_nResHeight);

  if( m_cfg.doar )
    m_pLetterbox->ProcessSimple( out, NULL );

  //ResizerRun(in, out);
  return flfil_ok;
}

int FlResize::StopSimple()
{
  if( m_cfg.doar ) {
    if( m_pLetterbox ) {
      delete m_pLetterbox;
      m_pLetterbox = NULL;
    }
  }
  // TODO
  return flfil_ok;
}