/* 
 *  ProgressGraph.cpp
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "ProgressGraph.h"


#define NORMAL_FRAMECOLOR RGB(36, 90, 180)
#define KEY_FRAMECOLOR RGB(78, 121, 220)
#define KNOBCOLOR RGB(221, 159, 77)
#define BACKGROUNDCOLOR RGB(60, 60, 60)
#define PANELCOLOR RGB(80, 80, 80)
#define WHITE RGB(255, 255, 255)
#define RED RGB(255, 0, 0)

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CProgressGraph::CProgressGraph()
{
  m_nMinFramesize = 0;
  m_nMaxFramesize = 0;
  m_nTotalFramesize = 0;
  m_nFramecount = 0;  
  m_nFramedelay = 0;
}

CProgressGraph::~CProgressGraph()
{

}

bool CProgressGraph::Initialize(HWND hWnd, long nWidth, long nHeight, long nFrameDelay)
{

  if ( CCanvas::Initialize(hWnd, nWidth, nHeight) )
  {
    // Set widths of the vectors
    m_nBarCount = m_nWidth / PROGRESSBAR_WIDTH;
    m_vFrames.resize( m_nBarCount );
    m_vBars.resize( m_nBarCount );
  
    TProgressBar sBar;
    TProgressFrame sFrame;
  
    m_nFramedelay = nFrameDelay;

    memset( &sBar, 0, sizeof TProgressBar );
    memset( &sFrame, 0, sizeof TProgressFrame );
  
    // Set defaults
    for(int i=0; i<m_nBarCount; i++)
    {
      m_vFrames[i] = sFrame;
      m_vBars[i] = sBar;
    }

    m_nMinFramesize = 999999999;
    m_nMaxFramesize = 0;
    m_nTotalFramesize = 0;
    m_nFramecount = 0;  

    m_hWnd = hWnd;

    return true;
  }
  return false;
}

void CProgressGraph::StreamInfo(ui64 size )
{
  m_nTotalSize = size; 

  if( !m_nFramecount || !m_nFramedelay )
    return;

  // Calculate avg bitrate in bytes/sec
  m_nAverageBitrate = (m_nTotalSize * (ui64)100000) / (ui64)(m_nFramecount * m_nFramedelay);
  // Convert to kilobits / sec
  m_nAverageBitrate = ( m_nAverageBitrate * (ui64)8 ) / (ui64)1000;
}
void CProgressGraph::NewFrame(long size, bool bKeyframe )
{
  if(!m_bInitialized)
    return;

  m_nFramecount++;
  m_nTotalFramesize += size;

  if(size > m_nMaxFramesize) m_nMaxFramesize = size;
  if(size < m_nMinFramesize) m_nMinFramesize = size;

  // Calculate avg bitrate in bytes/sec
  m_nAverageVideoBitrate = (m_nTotalFramesize * 100000) / (m_nFramecount * m_nFramedelay);
  // Convert to kilobits / sec
  m_nAverageVideoBitrate = ( m_nAverageVideoBitrate * 8 ) / 1000;

  // Add the new frame to the list
  m_vFrames.pop_front();

  TProgressFrame sFrame;
  sFrame.bKeyFrame = bKeyframe;
  sFrame.nValue = size;

  m_vFrames.push_back( sFrame );


  //Find Max in frames
  int maxpos=0;
  int maxvalue=0;

  for( int i=0; i<m_nBarCount; i++ )
  {
    if(m_vFrames[i].nValue > maxvalue)
    {
      maxpos = i;
      maxvalue = m_vFrames[i].nValue;
    }
  }

  // Adjust the bars
  for( i=0; i<m_nBarCount; i++)
  {
    if(maxvalue) m_vBars[i].nHeight = (m_vFrames[i].nValue * m_nHeight) / maxvalue;
    m_vBars[i].bKeyFrame = m_vFrames[i].bKeyFrame;
  }
}

void CProgressGraph::Draw()
{
  if(!m_bInitialized)
    return;

  // Draw the rectangles


  TRectangle sRectpar;
  TRect sRect;  

  sRectpar.bFilled = true;

  // Draw Background
  SetBackground(BACKGROUNDCOLOR);

  // Adjust the bars
  for( int i=0; i<m_nBarCount; i++)
  {
    if( m_vBars[i].nHeight != 0 )
    {


      sRect.nLeft = i*PROGRESSBAR_WIDTH;
      sRect.nRight = sRect.nLeft + PROGRESSBAR_WIDTH;
      sRect.nTop = m_nHeight - m_vBars[i].nHeight;
      sRect.nBottom = m_nHeight;

      sRectpar.nBorder = 0;
      sRectpar.nColor = m_vBars[i].bKeyFrame ? KEY_FRAMECOLOR : NORMAL_FRAMECOLOR;
      sRectpar.sRect = sRect;
      DrawRectangle(&sRectpar);
    }
  }

  // Draw keyframes knots
  for( i=0; i<m_nBarCount; i++)
  {
    if( m_vBars[i].nHeight!=0 && m_vBars[i].bKeyFrame )
    {
     
      sRect.nLeft = i*PROGRESSBAR_WIDTH - 1;
      sRect.nRight = sRect.nLeft + PROGRESSBAR_WIDTH + 1;
      sRect.nTop = m_nHeight - m_vBars[i].nHeight - 1;
      sRect.nBottom = sRect.nTop + PROGRESSBAR_WIDTH + 1;
      
      sRectpar.nBorder = 0;
      sRectpar.nBorderColor = KNOBCOLOR;
      sRectpar.nColor = KNOBCOLOR;

      sRectpar.sRect = sRect;
      DrawRectangle(&sRectpar);
    }
  }

  // Legend
  DrawRectangle( 10, 10, 13, 13, KNOBCOLOR);
  DrawText( "Keyframes", 15, 7, WHITE );

  // Stats
  long nStatsLeft  = m_nWidth - 150;
  long nStatsRight = m_nWidth - 10;

  DrawRectangle( nStatsLeft, 10, nStatsRight, 60, NORMAL_FRAMECOLOR, 2, WHITE);
  char szTemp[256];

  if( m_nMinFramesize >= 1024 )
    sprintf(szTemp, "Min: %d kb", m_nMinFramesize >> 10);
  else
    sprintf(szTemp, "Min: %d bytes", m_nMinFramesize );

  DrawText( szTemp, nStatsLeft +2, 12, WHITE );

  sprintf(szTemp, "Max: %d kb", m_nMaxFramesize >> 10);
  DrawText( szTemp, nStatsLeft +2, 23, WHITE );

  if( m_nAverageVideoBitrate > 1000 )
    sprintf(szTemp, "Video bitrate: %.3f Mbits/s", (float)m_nAverageVideoBitrate/1000.0 );
  else
    sprintf(szTemp, "Video bitrate: %.d kbits/s", m_nAverageVideoBitrate );

  DrawText( szTemp, nStatsLeft +2, 34, WHITE );
  
  if( m_nAverageBitrate > 1000 )
    sprintf(szTemp, "Total bitrate: %.3f Mbits/s", (float)m_nAverageBitrate/1000.0 );
  else
    sprintf(szTemp, "Total bitrate: %.d kbits/s", m_nAverageBitrate );

  DrawText( szTemp, nStatsLeft +2, 45, WHITE );
  // Draw the whole thing
  CCanvas::DrawCanvas(0,0);
}
