/* 
 *  Profile.cpp 
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */
 

#include <windows.h>
#include "Profile.h"
#include "Auxiliary.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CProfile::CProfile(char *program_directory)
{
    n_selected = 0;
    if(program_directory)
        strcpy(this->program_directory, program_directory);
}

CProfile::~CProfile()
{

}

static char *validateName( char *name )
{
  static char validname[1024];
  strcpy( validname, name );
  validateFileName( validname );
  return validname;
}

int CProfile::LoadProfileFromDisk(char *fileName)
{
    FILE *input_profile;
    TProfile *myProfile;

    // Open file with this name
    if(!(input_profile=fopen(fileName, "rb")) )
        return 0;
    // create temporary profile
    myProfile = (TProfile *)malloc( sizeof(TProfile) );

    // Load profile from disk
    if( fread( myProfile, 1, sizeof(TProfile), input_profile ) != sizeof(TProfile) )
    {
        free(myProfile);
        fclose(input_profile);
        return 0;
    }
    // If the profile version doesnt match with this version, drop it
    if(myProfile->nProfileVersion!=PROFILE_VERSION)
    {
      free(myProfile);
      fclose(input_profile);
      return 0;
    }
    // Add profile to our profile list
    profiles.AddItem( myProfile );
    
    fclose(input_profile);
    free(myProfile);
    return 1;
}

// Refresh: Load all available profiles from disk
int CProfile::Refresh()
{
	WIN32_FIND_DATA		find_data;
	char				directory[MAX_PATH], szTemp[MAX_PATH];
	HANDLE				search_handle;
	int					i;

	
  sprintf(directory, "%s\\Profiles\\*.profile.flask", program_directory );

  profiles.EmptyArray();

	i=0;
	search_handle = FindFirstFile(directory, &find_data);
	if(search_handle==INVALID_HANDLE_VALUE){
		profiles.EmptyArray();
	}
	else{
    sprintf(szTemp,"%s\\Profiles\\%s", program_directory, find_data.cFileName );
		LoadProfileFromDisk( szTemp );

        while( FindNextFile(search_handle, &find_data ) ){
            sprintf(szTemp,"%s\\Profiles\\%s", program_directory, find_data.cFileName );
            LoadProfileFromDisk( szTemp );
		}
		FindClose(search_handle);
	}
  
  Sort();

  SelectDefault();


  return profiles.GetCount();
}

int CProfile::AddProfile(TProfile *profile_data )
{
	char szTemp[MAX_PATH];
	FILE *file;

  if(!profile_data)
      return 0;

  profile_data->nProfileVersion = PROFILE_VERSION;

	strcpy(szTemp, program_directory);
	strcat(szTemp,"\\Profiles" );
	//Creating Profiles folder
	CreateDirectory(szTemp, NULL);

	sprintf(szTemp, "%s\\%s.profile.flask", szTemp, validateName(profile_data->profile_name) );
	if(!(file=fopen(szTemp, "wb"))){
		//Ooops
		return 0;
	}
	else
	{
		fwrite( profile_data, sizeof(TProfile), 1, file);
		fclose(file);
        // Now, reload profiles
		return Refresh();
	}
}

int CProfile::DeleteProfile(int profileIndex)
{
    char szTemp[MAX_PATH];

    // Cant delete profile 0
    if( profileIndex <=0 || profileIndex > (profiles.GetCount()-1) )
        return NULL;

	sprintf(szTemp, "%s\\Profiles\\%s.profile.flask", program_directory, validateName(profiles[profileIndex].profile_name) );
  DeleteFile(szTemp);
  return Refresh();
}

int CProfile::GetCount()
{
    return profiles.GetCount();
}
    
TProfile * CProfile::Get( int profileIndex)
{
    if( profileIndex <0 || profileIndex > (profiles.GetCount()-1) )
        return NULL;
    return &profiles[profileIndex];
}

TProfile * CProfile::GetSelected()
{
    return &profiles[n_selected];
}

int CProfile::GetSelectedIndex()
{
    return n_selected;
}

int CProfile::Select(int profileIndex)
{
    if( profileIndex <0 || profileIndex > (profiles.GetCount()-1) )
        return NULL;
    n_selected = profileIndex;
    return 1;
}

bool CProfile::IsDefault()
{
  return strcmp(profiles[n_selected].profile_name, "Default")==0;
}

// Sets passed profiled as default
bool CProfile::SetDefault(TProfile *prof)
{
  // Change the name of the profile to "Default"
  strcpy( prof->profile_name, "Default");
  
  DeleteProfile(GetDefaultIndex());
  AddProfile(prof);

  return true;
}

bool CProfile::IsSameProfile(TProfile *prof)
{
  return memcmp( prof, GetSelected(), sizeof(TProfile) ) == 0;
}

TProfile *CProfile::GetDefault()
{
  for( int i=0; i<profiles.GetCount(); i++ )
  {
    if( strcmp(Get(i)->profile_name, "Default")==0 )
      break;
  }
  if( i >= profiles.GetCount() )
    i--;
  return Get(i);
}

int CProfile::GetDefaultIndex()
{
  for( int i=0; i<profiles.GetCount(); i++ )
  {
    if( strcmp(Get(i)->profile_name, "Default")==0 )
      break;
  }
  if( i >= profiles.GetCount() )
    i--;
  return i;
}

void CProfile::Sort()
{
  // First take the default and put it at the beginning
  Swap(GetDefaultIndex(), 0);

  // Use stupid and simple bubble algorithm
  int max, nItems = profiles.GetCount();
  while( nItems > 1 ) // Dont include 0. The default and 1
  {
    max = 1;
    for(int i=1; i<nItems; i++)
    {
      // if profile at i is greater than the current max
      if(_stricmp(profiles[i].profile_name, profiles[max].profile_name)>0)
        max = i;
    }
    // Swap the maximum to the last
    Swap(nItems-1, max);
    nItems--;
  }
}

void CProfile::Swap(int a, int b)
{
  int nItems = profiles.GetCount();
  // Check the limits of a and b
  if( a>=nItems || b>=nItems )
    return;

  TProfile sTempProf;

  sTempProf = profiles[a];
  profiles[a] = profiles[b];
  profiles[b] = sTempProf;

}
void CProfile::SelectDefault()
{
  Select( GetDefaultIndex() );
}

bool CProfile::AddNSelect( TProfile *prof )
{
  if( !prof )
    return false;

  // If the current profile is different from the selected one
  if ( IsSameProfile( prof ) == false )
  {
    // Set as default 
    SetDefault( prof );
    // And select it
    SelectDefault();
  }
  return true;
}

int CProfile::FindIndexWithName(char *profile_name)

{

  for( int i=0; i<profiles.GetCount(); i++ )

  {

    if( strcmp(Get(i)->profile_name, profile_name)==0 )

      break;

  }

  if( i >= profiles.GetCount() )

    return GetDefaultIndex();

  else

    return i;

}