/* 
 *  options.cpp    -  Controls FlasKMPEG options
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "resource.h"
#include "OutputPad.h"
#include "PostProcessing.h"
#include "RunState.h"
#include "Auxiliary.h"
#include "Canvas.h"
#include "debug.h"
 
extern TRunState        rs;
extern HINSTANCE        hInst;

#define NORMAL_FRAMECOLOR RGB(36, 90, 180)
#define KEY_FRAMECOLOR RGB(78, 121, 220)
#define KNOBCOLOR RGB(221, 159, 77)
#define BACKGROUNDCOLOR RGB(60, 60, 60)
#define PANELCOLOR RGB(80, 80, 80)
#define WHITE RGB(255, 255, 255)
#define RED RGB(255, 0, 0)


////////////////////////////////////////////////////////
//             OUTPUT    PAD
////////////////////////////////////////////////////////
LRESULT CALLBACK COutputPadDlg::DisplayProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam){

  HDC hDC;
  PAINTSTRUCT lp;
  LPCTSTR mousetype;

#define GET_X(lp)                        ((int)(short)LOWORD(lp))
#define GET_Y(lp)                        ((int)(short)HIWORD(lp))

	switch (message)
	{
    case WM_ERASEBKGND:
        return TRUE;
		case WM_PAINT:
      hDC= BeginPaint(hWnd, &lp);
      Draw();
      EndPaint(hWnd, &lp);
			break;
    case WM_MOVE:
      Draw();
      break;
    case WM_MOUSEMOVE:
      if( wParam&MK_LBUTTON ) {
        if( pp.crop && m_cp.OnDrag( GET_X(lParam), GET_Y(lParam) ) ){

          if( m_cp.OnHover(  GET_X(lParam), GET_Y(lParam) , mousetype ) )
            SetCursor( LoadCursor(NULL, mousetype));

          // parameters changed
          TCropConfig cfg = *m_cp.GetConfig();
          pp.cropLeftOffset = cfg.left;
          pp.cropWidth      = cfg.right - cfg.left;
          pp.cropTopOffset  = cfg.top;
          pp.cropHeight     = cfg.bottom - cfg.top;
    
          m_bSettingsChanged = true;
          Draw();

        } // !pp.crop || !m_cp.OnDrag
        else if( pp.letterbox && m_lp.OnDrag( GET_X(lParam), GET_Y(lParam) ) )  {
          if( m_lp.OnHover(  GET_X(lParam), GET_Y(lParam) , mousetype ) )
            SetCursor( LoadCursor(NULL, mousetype));
          // parameters changed
          TLetterboxConfig cfg = *m_lp.GetConfig();
          pp.letterboxLeft = cfg.left;
          pp.letterboxRight = cfg.right;
          pp.letterboxTop  = cfg.top;
          pp.letterboxBottom  = cfg.bottom;
    
          m_bSettingsChanged = true;
          Draw();
        }
      }
      else { // !wParam&MK_LBUTTON
        if( pp.crop && m_cp.OnHover(  GET_X(lParam), GET_Y(lParam) , mousetype ) )
          SetCursor( LoadCursor(NULL, mousetype));
        else if ( pp.letterbox && m_lp.OnHover(  GET_X(lParam), GET_Y(lParam) , mousetype ) )
          SetCursor( LoadCursor(NULL, mousetype));
        else
          SetCursor( LoadCursor(NULL, IDC_ARROW));
      }
      break;
    case WM_LBUTTONDOWN:
      SetCapture(hWnd);
      if( pp.crop )
        if( m_cp.OnClick(  GET_X(lParam), GET_Y(lParam)  ) )
          break;

      if( pp.letterbox )
        if( m_lp.OnClick(  GET_X(lParam), GET_Y(lParam)  ) )
          break;


      break;
    case WM_LBUTTONUP:
      ReleaseCapture();
      m_cp.OnRelease();
      m_lp.OnRelease();
      break;
		case WM_COMMAND:

			break;
    default:
      return DefWindowProc(hWnd, message, wParam, lParam);
      break;
	}
  return FALSE;
}


// Snap second window to the first
void Snap(HWND win1, HWND win2){
	RECT   rc;
		// Get rect win1
		GetWindowRect(win1, &rc);
		// Set position of second window
		WindowMove(win2, rc.left, rc.bottom );


}



// EnableCrop : Enable crop buttons
void COutputPadDlg::EnableCrop(){
	DlgEnable(m_hDlg, CROP_TOPOFFSET_PLUS);
	DlgEnable(m_hDlg, CROP_TOPOFFSET_MINUS);
	DlgEnable(m_hDlg, CROP_HEIGHT_PLUS);
	DlgEnable(m_hDlg, CROP_HEIGHT_MINUS);
	DlgEnable(m_hDlg, CROP_LEFTOFFSET_PLUS);
	DlgEnable(m_hDlg, CROP_LEFTOFFSET_MINUS);
	DlgEnable(m_hDlg, CROP_WIDTH_PLUS);
	DlgEnable(m_hDlg, CROP_WIDTH_MINUS);
	DlgEnable(m_hDlg, WIDTH_MINUS1);
	DlgEnable(m_hDlg, WIDTH_PLUS1);
	DlgEnable(m_hDlg, HEIGHT_MINUS1);
	DlgEnable(m_hDlg, HEIGHT_PLUS1);
}
// DisableCrop : Disable crop buttons
void COutputPadDlg::DisableCrop(){
	DlgDisable(m_hDlg, CROP_TOPOFFSET_PLUS);
	DlgDisable(m_hDlg, CROP_TOPOFFSET_MINUS);
	DlgDisable(m_hDlg, CROP_HEIGHT_PLUS);
	DlgDisable(m_hDlg, CROP_HEIGHT_MINUS);
	DlgDisable(m_hDlg, CROP_LEFTOFFSET_PLUS);
	DlgDisable(m_hDlg, CROP_LEFTOFFSET_MINUS);
	DlgDisable(m_hDlg, CROP_WIDTH_PLUS);
	DlgDisable(m_hDlg, CROP_WIDTH_MINUS);
	DlgDisable(m_hDlg, WIDTH_MINUS1);
	DlgDisable(m_hDlg, WIDTH_PLUS1);
	DlgDisable(m_hDlg, HEIGHT_MINUS1);
	DlgDisable(m_hDlg, HEIGHT_PLUS1);
}

void COutputPadDlg::EnableLetterbox(){
	DlgEnable(m_hDlg, LETTERBOX_TOP_PLUS);
	DlgEnable(m_hDlg, LETTERBOX_TOP_MINUS);
	DlgEnable(m_hDlg, LETTERBOX_BOTTOM_PLUS);
	DlgEnable(m_hDlg, LETTERBOX_BOTTOM_MINUS);
	DlgEnable(m_hDlg, LETTERBOX_LEFT_PLUS);
	DlgEnable(m_hDlg, LETTERBOX_LEFT_MINUS);
	DlgEnable(m_hDlg, LETTERBOX_RIGHT_PLUS);
	DlgEnable(m_hDlg, LETTERBOX_RIGHT_MINUS);
}
void COutputPadDlg::DisableLetterbox(){
	DlgDisable(m_hDlg, LETTERBOX_TOP_PLUS);
	DlgDisable(m_hDlg, LETTERBOX_TOP_MINUS);
	DlgDisable(m_hDlg, LETTERBOX_BOTTOM_PLUS);
	DlgDisable(m_hDlg, LETTERBOX_BOTTOM_MINUS);
	DlgDisable(m_hDlg, LETTERBOX_LEFT_PLUS);
	DlgDisable(m_hDlg, LETTERBOX_LEFT_MINUS);
	DlgDisable(m_hDlg, LETTERBOX_RIGHT_PLUS);
	DlgDisable(m_hDlg, LETTERBOX_RIGHT_MINUS);
}

void COutputPadDlg::UpdatePadButtons()
{
  DlgCheck(m_hDlg, ASPECT_CHECK, pp.doAR);
  DlgCheck(m_hDlg, IDC_LETTERBOX, pp.letterbox);
  DlgCheck(m_hDlg, IDC_CROP, pp.crop);

  DlgCheck( m_hDlg, IDC_CROP, pp.crop );
  DlgCheck( m_hDlg, IDC_LETTERBOX, pp.letterbox );


  DlgCheck( m_hDlg, IDC_DEINTERLACE, pp.deinterlace );
  DlgCheck( m_hDlg, IDC_VIDEOCONTROLCHECK, pp.videocontrols );
  DlgEnable( m_hDlg, IDC_BRIGHTSLIDER, pp.videocontrols );
  
  SendDlgItemMessage(m_hDlg, IDC_BRIGHTSLIDER, TBM_SETPOS, (WPARAM) (BOOL)true,(LPARAM) (LONG)pp.brightness );
}


static COutputPadDlg *gDlg;

static LRESULT CALLBACK DlgProcHook(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  return gDlg->DlgProc(hWnd, message, wParam, lParam);
}

static LRESULT CALLBACK DisplayProcHook(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  return gDlg->DisplayProc(hWnd, message, wParam, lParam);
}


COutputPadDlg::COutputPadDlg( HWND hParentWnd, HWND hMainControl, HINSTANCE hInst)
{ 
  m_hDlg = NULL;
  m_hParentWnd = hParentWnd; m_hInst = hInst;
  m_hMainControl = hMainControl;
  gDlg = this;
  m_nFrameCount =0;
}

bool COutputPadDlg::Show(TPPost *pPPost, FlPPReceiver *ppRec )
{
  if(!pPPost || !ppRec )
    return false;

  if(m_hDlg)
    return true;

  m_nFrameCount = 0;

  m_pReceiver = ppRec;

  ppc = pp = *pPPost;

  m_hDlg = CreateDialog( m_hInst, MAKEINTRESOURCE(DLG_OUTPUT_PAD), m_hParentWnd, (DLGPROC) DlgProcHook );

  ShowWindow( m_hDlg, SW_SHOW );

  return true;
}

bool COutputPadDlg::Hide()
{ 
  if(!m_hDlg)
    return false;
  
  if(m_pVideoRenderer) {
    delete m_pVideoRenderer;
    m_pVideoRenderer = NULL;
  }
  m_pp.Stop();
  DestroyWindow(m_hDlg);
  DestroyWindow(m_hDisplay);
  m_hDisplay = NULL;
  m_hDlg = NULL;
  return true;
}

void COutputPadDlg::SetupTooltips()
{
  HINSTANCE hInst = rs.hInst;

#define ATTACH(id, title, text){ DlgAttachToolTip( hInst, m_hDlg, id, title, text); }


  // Play controls
  ATTACH( IDC_CROP, "Crop", "Crops the video\n\rUse the mouse to setup\n\rthe cropping area." )
  ATTACH( IDC_LETTERBOX, "Letterbox", "Applies courtains to the video\n\rUse the mouse to setup\n\rthe areas." )
  ATTACH( IDC_DEINTERLACE, "Play", "Plays video." )
  ATTACH( ASPECT_CHECK, "Keep Aspect Ratio", "Maintains aspect ratio of source." )
  ATTACH( IDC_VIDEOCONTROLCHECK, "Brightness", "Use the slider\n\rto modify video brightness." )
  ATTACH( IDC_RESET, "Reset", "Resets to default settings" )
  ATTACH( R_OUTPUT_PAD_OUTPUTSIZE, "Resize", "Resizes the video" )
 

#undef ATTACH
}
void COutputPadDlg::SetWindows(HWND hParentWnd, HWND hMainControl )
{
  m_hParentWnd = hParentWnd;
  m_hMainControl = hMainControl;
}

void COutputPadDlg::PositionWindows( bool bExcludeOutputPad )
{
  // Position video window
	RECT rcDesktop, rcVideo, rcPad, rcControl, rcMain;
	GetWindowRect( GetDesktopWindow(), &rcDesktop);
  GetWindowRect( m_hDisplay, &rcVideo );
  GetWindowRect( m_hDlg, &rcPad );
  GetWindowRect( m_hMainControl, &rcControl );
  GetWindowRect( m_hParentWnd, &rcMain );

  // Setup video window next to main window
  int width = rcVideo.right - rcVideo.left;
  
  if( (rcMain.right + 5 + width) > (rcDesktop.right-rcDesktop.left) )
    WindowMove( m_hDisplay, rcDesktop.right - width - 5, rcMain.top + 25 );
  else
    WindowMove( m_hDisplay, rcMain.right +10, rcMain.top + 25 );
  GetWindowRect( m_hDisplay, &rcVideo );

  // Setup control window below video
  WindowMove( m_hMainControl, rcControl.left, rcVideo.bottom + 10 );
  GetWindowRect( m_hMainControl, &rcControl );

  // Setup toolbar next to control
  if( bExcludeOutputPad == false )
    WindowMove( m_hDlg, rcControl.right +5, rcControl.top );
  
 // SetWindowPos( m_hDlg, HWND_BOTTOM, 0,0,0,0, SWP_NOSIZE|SWP_NOMOVE );

}

LRESULT CALLBACK COutputPadDlg::DlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  ATOM at;
  FlPostProcess postpr;
  
  switch (message)
  {
		case WM_INITDIALOG:
      m_hDlg = hDlg;
      
      SetupTooltips();
      // Initialize strings
      SetText(hDlg, GS(OUTPUT_PAD_TITLE));
      // Crop
      DlgSetText(hDlg, IDC_CROP, GS(OUTPUT_PAD_CROP) );
      DlgSetText(hDlg, OPT_OUTPAD_TOPOFFSET, GS(OPT_OUTPAD_TOPOFFSET));
      DlgSetText(hDlg, OPT_OUTPAD_CROPHEIGHT, GS(OPT_OUTPAD_CROPHEIGHT));
      DlgSetText(hDlg, OPT_OUTPAD_LEFTOFFSET, GS(OPT_OUTPAD_LEFTOFFSET));
      DlgSetText(hDlg, OPT_OUTPAD_CROPWIDTH, GS(OPT_OUTPAD_CROPWIDTH));
      // Output Size
      DlgSetText(hDlg, R_OUTPUT_PAD_OUTPUTSIZE, GS(OUTPUT_PAD_OUTPUTSIZE));
      DlgSetText(hDlg, OPT_OUTPAD_WIDTH, GS(OPT_OUTPAD_WIDTH));
      DlgSetText(hDlg, OPT_OUTPAD_HEIGHT, GS(OPT_OUTPAD_HEIGHT));
      DlgSetText(hDlg, IDOK , GS(OUTPUT_PAD_HIDE));
      
      DlgSetText(hDlg, IDC_LETTERBOX, GS(OUTPUT_PAD_LETTERBOX));
      DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_TOP, GS(OPT_OUTPAD_LETTERBOX_TOP));
      DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_BOTTOM, GS(OPT_OUTPAD_LETTERBOX_BOTTOM));
      DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_LEFT, GS(OPT_OUTPAD_LETTERBOX_LEFT));
      DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_RIGHT, GS(OPT_OUTPAD_LETTERBOX_RIGHT));
      //Reset
      DlgSetText(hDlg, IDC_RESET, GS(OUTPUT_PAD_RESET));
      
      // Set brightness trackbar
      SendDlgItemMessage(hDlg, IDC_BRIGHTSLIDER, TBM_SETRANGE, (WPARAM) TRUE,(LPARAM) MAKELONG(0, 255)); 
      
      //Set crop checkbox
      UpdatePadButtons();
      
      
      // Register class for window
      WNDCLASSEX csex;
      memset( &csex, 0, sizeof WNDCLASSEX );
      csex.style = 0;
      csex.cbSize = sizeof WNDCLASSEX;
      csex.lpfnWndProc = DisplayProcHook;
      csex.hInstance = hInst;
      csex.hbrBackground = 0;
      csex.lpszClassName = "SimpleWindow";
      
      at = RegisterClassEx( &csex );
      
      // Create display window
      m_hDisplay = CreateWindow( "SimpleWindow", "Video Panel Pad", WS_BORDER ,
                                  0,0, 100, 100, m_hParentWnd, 
                                  NULL, hInst, NULL );
      
      // Set as toolbar windows
      SetWindowLong( hDlg, GWL_EXSTYLE, WS_EX_TOOLWINDOW );
      SetWindowLong( m_hDisplay, GWL_EXSTYLE, WS_EX_TOOLWINDOW );
      
      
      
      WindowShow( m_hDisplay );
      Snap(hDlg, m_hDisplay);
      
      // Start postprocessing for the first time
      ppc.crop = 0;
      m_pp.Set(&ppc);
      m_pp.Start();
      m_bSettingsChanged = true;
      WindowClientResize( m_hDisplay, m_pp.GetWidth(), m_pp.GetHeight() );    
      PositionWindows();
      return TRUE;
    case WM_MOVE:
      //Snap(hDlg, m_hDisplay);
      break;
    case WM_HSCROLL:
      int nPos;
      switch( LOWORD(wParam) ) {
        case TB_THUMBTRACK:
          nPos = SendDlgItemMessage(hDlg, IDC_BRIGHTSLIDER, TBM_GETPOS, 0, 0);
          pp.brightness = nPos;
          m_bSettingsChanged = true;
          Draw();
          break;
      }
      break;
    case WM_COMMAND:
      if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
      {
        DestroyWindow(m_hDisplay);
        DestroyWindow(m_hDlg);
        return LOWORD(wParam) == IDOK ? TRUE : FALSE;
      }

      switch (LOWORD(wParam)) {
        case WIDTH_PLUS:
        case WIDTH_MINUS:
        case HEIGHT_PLUS:
        case HEIGHT_MINUS:
          if( pp.bResize == false ) {
            pp.resHeight = rs.video->GetHeight();
            pp.resWidth  = rs.video->GetWidth();
            pp.bResize = true;
          }
          break;
      }

      switch (LOWORD(wParam)){
        case IDC_VIDEOCONTROLCHECK:
          pp.videocontrols = DlgIsChecked( hDlg, IDC_VIDEOCONTROLCHECK );
          UpdatePadButtons();
        case IDC_DEINTERLACE:
          pp.deinterlace = DlgIsChecked( hDlg, IDC_DEINTERLACE );
          break;
        case IDC_RESET:
          pp.crop            = 0;
          pp.cropWidth      = rs.video->pictureWidth;
          pp.cropHeight     = rs.video->pictureHeight;
          pp.cropLeftOffset = 0;
          pp.cropTopOffset  = 0;
        
          pp.letterbox       = 0;
          pp.letterboxBottom = 0;
          pp.letterboxLeft   = 0;
          pp.letterboxRight  = 0;
          pp.letterboxTop    = 0;
        
          pp.deinterlace = 0;
          pp.doAR = false;
        
          pp.videocontrols  = false;
          pp.brightness     = 128;
        
          pp.bResize         = false;
          pp.resWidth        = rs.video->pictureWidth;
          pp.resHeight       = rs.video->pictureHeight;

       
          UpdatePadButtons();
          break;
        case IDC_CROP:
          pp.crop=DlgIsChecked(hDlg, IDC_CROP);
          if(pp.crop){
            pp.cropWidth      = m_pp.GetWidth();
            pp.cropHeight     = m_pp.GetHeight();
            pp.cropLeftOffset = 0;
            pp.cropTopOffset  = 0;
          }
          else {
            // if crop is turned off, we must set resWidth/Height to
            // multiples of 16
            pp.resHeight = 16 * ( pp.resHeight / 16 );
            pp.resWidth = 16 * (pp.resWidth / 16 );
          }
        
          UpdatePadButtons();
          break;
        
        case IDC_LETTERBOX:
          pp.letterbox=DlgIsChecked(hDlg, IDC_LETTERBOX);
          if(pp.letterboxBottom<=0 &&
            pp.letterboxLeft  <=0 && 
            pp.letterboxRight <=0 &&
            pp.letterboxTop   <=0){
            pp.letterboxBottom =0;
            pp.letterboxLeft   =0;
            pp.letterboxRight  =0;
            pp.letterboxTop    =0;
          }
        
          UpdatePadButtons();
          break;
        case ASPECT_CHECK:
          pp.doAR = DlgIsChecked(hDlg, ASPECT_CHECK);
          if (pp.doAR) {
            pp.letterbox = false;
            if (pp.outAR & (AR_43|AR_169))
              pp.crop = false;
          }
        
          UpdatePadButtons();
          break;
        
        case WIDTH_PLUS:
          pp.resWidth+=16;
          if( !CheckVideoParameters(&pp) ) pp.resWidth-=16;
          break;
        case WIDTH_MINUS:
          pp.resWidth-=16;
          if( pp.crop && pp.cropWidth>pp.resWidth ) pp.cropWidth = pp.resWidth;
          if( !CheckVideoParameters(&pp) ) pp.resWidth+=16;
          break;
        case HEIGHT_PLUS:
          pp.resHeight+=16;
          if( !CheckVideoParameters(&pp) ) pp.resHeight-=16;
          break;
        case HEIGHT_MINUS:
          pp.resHeight-=16;
          if( pp.crop && pp.cropHeight>pp.resHeight ) pp.cropHeight = pp.resHeight;
          if( !CheckVideoParameters(&pp) ) pp.resHeight+=16;
          break;
      }
      m_bSettingsChanged = true;

      ppc = pp;
      ppc.crop = 0;
      m_pp.Set( &ppc );
      postpr.Set( &pp );

      WindowClientResize( m_hDisplay, m_pp.GetWidth(), m_pp.GetHeight() );
      WindowClientResize( m_hParentWnd, postpr.GetWidth() , postpr.GetHeight() );
      Draw();

      switch( LOWORD(wParam) )
      {
        case WIDTH_PLUS:
        case WIDTH_MINUS:
        case HEIGHT_PLUS:
        case HEIGHT_MINUS:
        case IDC_RESET:
          PositionWindows( true );
          break;
      }
      break;
    case WM_DESTROY:
      // hold the lock while stopping the post processing
      CFlAutoLock lock(&m_csGlobal);
      m_pp.Stop();
      m_hDlg = NULL;
      break;
  }
  return FALSE;
}


int COutputPadDlg::PutFrame( CFrame *fr )
{
  CFlAutoLock lock(&m_csGlobal);

  if(m_hDlg)
    frIn = *fr;  
  
  Draw();
  
  return 0;
}

void COutputPadDlg::Draw()
{
  char        szTemp[256];
  int width, height;

  if(!m_hDlg)
    return;
  
  CFlAutoLock lock(&m_csGlobal);

  if( m_bSettingsChanged )
  {
    m_pp.Stop();
    ppc = pp;
    ppc.crop = 0;
    m_pp.Set(&ppc);
    m_pp.Start();
    m_cv.Initialize( m_hDisplay, m_pp.GetWidth(), m_pp.GetHeight() );
    m_bSettingsChanged = false;

    m_pReceiver->PutPPSettings(&pp);

    TCropConfig cfg;
    cfg.left = pp.cropLeftOffset;
    cfg.right = pp.cropLeftOffset + pp.cropWidth;
    
    cfg.top = pp.cropTopOffset;
    cfg.bottom = pp.cropTopOffset + pp.cropHeight;

    TLetterboxConfig lcfg;
    lcfg.bottom = pp.letterboxBottom;
    lcfg.top    = pp.letterboxTop;
    lcfg.left   = pp.letterboxLeft;
    lcfg.right  = pp.letterboxRight;

    m_cp.SetConfig( m_pp.GetWidth(), m_pp.GetHeight(), &cfg);
    m_lp.SetConfig( m_pp.GetWidth(), m_pp.GetHeight(), &lcfg);
  }
	width = m_pp.GetWidth();
	height = m_pp.GetHeight();

  if( !width || !height )
    return;

  //Postprocess
  frOut.Set( width, height, FRAME_YV12 );
  frRgb.Set( width, height, FRAME_RGB32 );

  m_pp.Process( &frIn, &frOut );

  frRgb.SetFrame( &frOut );

  m_cv.DrawRGB32( frRgb.GetBuffer(), 
                  frRgb.GetWidth(), frRgb.GetHeight(), 0, 0);



  if (pp.crop) 
  {
    m_cv.DrawRectangle( pp.cropLeftOffset,
                       pp.cropTopOffset,
                       pp.cropLeftOffset+pp.cropWidth,
                       pp.cropTopOffset+pp.cropHeight,
                       0, 2, RGB(255,255,255), false);
  }

  if (pp.letterbox) 
  {
    int startx, starty;
    startx = pp.letterboxLeft;
    m_cv.DrawLine( startx, 0, 
                   startx, height-1,
                  RED );

    startx = width - pp.letterboxRight - 1;
    m_cv.DrawLine( startx, 0,
                   startx, height-1,
                   RED );

    starty = pp.letterboxTop;
    m_cv.DrawLine( 0, starty,
                   width-1, starty, RED );

    starty = height - pp.letterboxBottom - 1;
    m_cv.DrawLine( 0, starty, 
                   width-1, starty, RED );

  }

  long lw = width, lh = height;
	if (pp.letterbox) 
  {
		lw -= (pp.letterboxLeft + pp.letterboxRight);
		lh -=  (pp.letterboxTop + pp.letterboxBottom);
	}
	sprintf(szTemp, "Resize %d x %d -> %d x %d", frIn.GetWidth(), frIn.GetHeight(),
			              lw, lh);

  m_cv.DrawText( szTemp, 3, 3, WHITE );

  if (!pp.doAR){
    sprintf( szTemp, "AR input %.03f output %.03f", 1.0/pp.iDAR, (double)width / (double)height);
  }
  else{
    sprintf( szTemp, "AR input %.03f output %s", 1.0/pp.iDAR,
            (pp.outAR & AR_169 ? "16:9 var. pixel" : 
            (pp.outAR & AR_43 ? "4:3 var. pixel" : "1:1 square pixel")));
  }
  
  m_cv.DrawText( szTemp, 3, 20, WHITE );

	if(pp.crop)
  {
		sprintf(szTemp, "Crop left/right %d/%d, top/bottom %d/%d -> %d x %d",
						pp.cropLeftOffset, 
						frOut.GetWidth()-pp.cropWidth-pp.cropLeftOffset,
						pp.cropTopOffset,
						frOut.GetHeight()-pp.cropHeight-pp.cropTopOffset,
						pp.cropWidth, pp.cropHeight);
		
    m_cv.DrawText( szTemp, 3, 37, WHITE );
	}
  
  m_cv.DrawCanvas(0,0);


}


