/* 
 *  MasterClock.cpp
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

// This is the implementation of a common clock in our system
// All methods have to be reentrant

#include "MasterClock.h"
#include <windows.h>
#include <winbase.h>

CMasterClock::CMasterClock(ClkType nMode)
{
  m_nMode = nMode;
}

CMasterClock::~CMasterClock()
{
}

// Determine if the high performance counter is avaliable.
bool CMasterClock::Initialize()
{
  bool nResult=true;

  switch( m_nMode )
  {
    case ClkSystem:
      nResult = QueryPerformanceFrequency( (PLARGE_INTEGER)&m_tiFrequency ) == TRUE;
      break;
    default:
      break;
  }
  return nResult;
}


void CMasterClock::Set(mmtick mmtkTime)
{

  // Lock the critical section first
  CFlAutoLock lock(&m_csGlobal);

  switch( m_nMode )
  {
    case ClkSystem:
      QueryPerformanceCounter( (PLARGE_INTEGER)&m_tiStartTime );

      m_mmtkStartTime = FROMCOUNT_TO_TICK(m_tiStartTime, m_tiFrequency);

      // delay = clk  -  our_clk
      m_mmtkStartDelay = m_mmtkStartTime - mmtkTime;
      break;
    case ClkSlave:
      m_tkTime = mmtkTime;
      break;
  }
}

mmtick CMasterClock::GetTickCount()
{
  // Lock the critical section first
  CFlAutoLock lock(&m_csGlobal);
  mmtick nTickCount;

  switch( m_nMode )
  {
  case ClkSystem:
    tick tiCurrentTime;
    QueryPerformanceCounter( (PLARGE_INTEGER)&tiCurrentTime );
    nTickCount = FROMCOUNT_TO_TICK(tiCurrentTime, m_tiFrequency) - m_mmtkStartDelay;
    break;
  case ClkSlave:
    nTickCount = m_tkTime;
    break;
  }
  return nTickCount;
}

void CMasterClock::Wait( mmtick tkTime, mmtick tkWindowPeriod )
{
  // Crop by the window
  if( tkTime > tkWindowPeriod )
    tkTime = tkWindowPeriod;

  // Sleep the thread for that number of ms
  Sleep( DWORD(tkTime / 10) );

  return;
}

void CMasterClock::WaitUntil( mmtick tkTime )
{
  while(1)
  {
    if( GetTickCount() >= tkTime )
      return;
    Sleep(10);
  }
  return;
}