/* 
 *  FrameBuffer.h
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#ifndef FRAMEBUFFER_H
#define FRAMEBUFFER_H




#include <windows.h>
#include "flasktypes.h"
#include "framesource.h"
#include "thread.h"
#include <list>
using namespace std;


class CListFrameBuffer : public CFrameBuffer
{
public:
    CListFrameBuffer(int nWidth, int nHeight, int nFormat, int nElements)
    {
      CFrame *pFrame;
      for(int i=0; i<nElements; i++)
      {
        pFrame = new CFrame(this);
        pFrame->Set( nWidth, nHeight, nFormat);

        // Add frame to the list
        m_lsFrames.push_back(pFrame);
      }
      m_evListIsNotEmpty.Reset();
    }
    ~CListFrameBuffer()
    {
      // Delete frames in the list
      int nSize = m_lsFrames.size();
      CFrame *pFrame;
      for(int i=0; i<nSize; i++)
      {
        pFrame = m_lsFrames.front();
        delete pFrame;
        m_lsFrames.pop_front();
      }
    }

    CFrame *GetFreeFrame()
    {
      int nSize;
      {
        CFlAutoLock lockObject(&m_csObject);
        nSize = m_lsFrames.size();
      }
      // If there were no frames wait for one
      if(!nSize)
        m_evListIsNotEmpty.Wait();
      {
        CFlAutoLock lockObject(&m_csObject);
        // Now grab one      
        CFrame *pFrame = m_lsFrames.front();
        // Add ref frame
        pFrame->AddRef();
        // remove frame from queue
        m_lsFrames.pop_front();
        //
        if(!m_lsFrames.size())
          m_evListIsNotEmpty.Reset();
        return pFrame;
      }

    }
    int GetSize()
    {
      CFlAutoLock lockObject(&m_csObject);
      return m_lsFrames.size();
    }    
    // Add this frame to the free list
    void AddFreeFrame(CFrame *pFrame)
    {
      CFlAutoLock lockObject(&m_csObject);
      m_lsFrames.push_back(pFrame);
      // And signal that the queue is not empty
      m_evListIsNotEmpty.Set();
    }

private:
  CFlCritSec m_csObject;
  CFlEvent m_evListIsNotEmpty;
  list<CFrame *> m_lsFrames;
};


// This queue just stores the pointers. It doesnt own the buffers.
class CQueueFrameBuffer
{
public:
  CQueueFrameBuffer()
  {
    m_lsFrames.clear();
    m_evListIsNotEmpty.Reset();
  }
  ~CQueueFrameBuffer()
  {
    // Delete frames in the list
    int nSize = m_lsFrames.size();
    CFrame *pFrame;
    for(int i=0; i<nSize; i++)
    {
      pFrame = m_lsFrames.front();
      delete pFrame;
      m_lsFrames.pop_front();
    }
  }

  int GetSize()
  {
    CFlAutoLock lockObject(&m_csObject);
    return m_lsFrames.size();
  }
  
  CFrame *RemoveFrame()
  {
    int nSize;
    {
      CFlAutoLock lockObject(&m_csObject);
      nSize = m_lsFrames.size();
    }
    // If there were no frames wait for one
    if(!nSize)
      m_evListIsNotEmpty.Wait();
    {
      CFlAutoLock lockObject(&m_csObject);    
      // Now grab one      
      CFrame *pFrame = m_lsFrames.front();

      // remove frame from queue
      m_lsFrames.pop_front();
      if(!m_lsFrames.size())
        m_evListIsNotEmpty.Reset();
    return pFrame;      
    }

  }
  
  // Add this frame to the free list
  void AddFrame(CFrame *pFrame)
  {
    CFlAutoLock lockObject(&m_csObject);
    m_lsFrames.push_back(pFrame);
    // And signal that the queue is not empty
    m_evListIsNotEmpty.Set();
  }
  
private:
  CFlCritSec m_csObject;
  CFlEvent m_evListIsNotEmpty;
  list<CFrame *> m_lsFrames;
};


#endif 