/* 
 *  FrameSource.h
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#ifndef FRAMESOURCE_H
#define FRAMESOURCE_H


#define FRAME_INTERLACED  0x01
#define FRAME_PROGRESSIVE 0x02

#define MAX_BUF 2048*2048*32

#include <windows.h>
#include "flasktypes.h"

// frame formats
#define FRAME_RGB  0
#define FRAME_YV12 2
#define FRAME_YUY2 1


class CFrame
{
public:
  CFrame( CFrameBuffer *pFrameBuffer )
  {
    m_pFrameBuffer = pFrameBuffer;
    m_nPresTime     = 0;
    m_nWidth = m_nHeigth = 0;;
    m_nFormat = 0;
    m_nDepth = 0;
    m_nRef = 0;
    m_pData = 0;
    m_nFrameFlags = 0;
    m_bOwnBuffer = false;
    memset( &m_sBitmapInfo, 0, sizeof( m_sBitmapInfo ) );
  }
  ~CFrame(){
    if(m_pData && m_bOwnBuffer)
      delete []m_pData;
  };
  
  CFrame &operator =(CFrame &oFrame)
  {
    if(!oFrame.IsValid())
      return *this;
    // free my buffer.
    if(m_pData && m_bOwnBuffer)
      delete []m_pData;
    m_nWidth = oFrame.GetWidth();
    m_nHeigth = oFrame.GetHeigth();
    m_nPresTime = oFrame.GetPresTime();
    m_nFormat = oFrame.GetFormat();
    m_nDepth = oFrame.GetDepth();
    m_nFrameFlags = oFrame.GetFlags();
    m_bOwnBuffer = true;
    Alloc();
    memcpy( m_pData, oFrame.GetBuffer(), oFrame.GetBufferSize() );
    return *this;
  }
  void Release()
  {
    CFlAutoLock lockObject(&m_csObject);
    //ASSERT(m_nRef>0);
    m_nRef--;
    if(m_nRef==0)
      if(m_pFrameBuffer)
        m_pFrameBuffer->AddFreeFrame(this);
  }
  void AddRef()
  {
    CFlAutoLock lockObject(&m_csObject);
    m_nRef++;
  }
  void SetPresTime( ui64 nPresTime ) { m_nPresTime = nPresTime; }
  void SetSize( ui32 nWidth, ui32 nHeigth )
  {
    m_nWidth = nWidth;
    m_nHeigth = nHeigth;
  }
  void SetFormat ( ui32 nFormat ){ m_nFormat = nFormat; }
  void SetDepth ( ui32 nDepth ){ m_nDepth = nDepth; }
  void SetBuffer ( ui8 *pData ){ m_pData = pData; }
  void SetFlags ( ui32 nFlags ){ m_nFrameFlags = nFlags; }
  ui8* GetBuffer(){ return m_pData; }
  ui32 GetWidth(){ return m_nWidth; }
  ui32 GetHeigth(){ return m_nHeigth; }
  ui64 GetPresTime(){ return m_nPresTime; }
  ui32 GetFormat(){ return m_nFormat; }
  ui32 GetDepth(){ return m_nDepth; }
  ui32 GetFlags(){ return m_nFrameFlags; }

  bool Alloc()
  {
    m_bOwnBuffer = true;
    bool bSuccess = true;
    int bufSize;
    switch(m_nFormat)
    {
      case FRAME_RGB:
        bufSize = m_nWidth * m_nHeigth * (m_nDepth/8);
        if(bufSize>MAX_BUF)
          return false;
        m_pData = new ui8[bufSize];
        break;
    }
    return true;
  }
  ui32 GetBufferSize()
  {
    int bufSize=0;
    switch(m_nFormat)
    {
    case FRAME_RGB:
      bufSize = m_nWidth * m_nHeigth * (m_nDepth/8);
      if(bufSize>MAX_BUF)
        bufSize = 0;
      break;
    }
    return bufSize;
  }
  BITMAPINFO *GetBmpInfo()
  {
    m_sBitmapInfo.bmiHeader.biWidth  = m_nWidth;
    m_sBitmapInfo.bmiHeader.biHeight = m_nHeigth;
    m_sBitmapInfo.bmiHeader.biCompression = m_nFormat==FRAME_RGB ? BI_RGB : m_nFormat;
    m_sBitmapInfo.bmiHeader.biSize = sizeof (BITMAPINFOHEADER);
    m_sBitmapInfo.bmiHeader.biPlanes = 1;
    m_sBitmapInfo.bmiHeader.biBitCount = (WORD)m_nDepth;
    return &m_sBitmapInfo;
  }

  bool IsValid(){
    return (m_pData!=NULL) && (m_nWidth!=0) && (m_nHeigth!=0);
  }
private:
  CFrameBuffer *m_pFrameBuffer;
  
  bool   m_bOwnBuffer;
  ui64   m_nPresTime;
  ui32   m_nWidth, m_nHeigth;
  ui32   m_nFormat;
  ui32   m_nDepth;
  ui32   m_nRef;
  ui8   *m_pData;
  ui32   m_nFrameFlags;

  BITMAPINFO m_sBitmapInfo;
  CFlCritSec m_csObject;
};

class CFrameSource
{
public:
  virtual bool GetFrame(CFrame *pFrame)=0;
};

// Format definition
typedef struct YV12ImageTag
{
  unsigned char *Y;
  int			  Yxsize;
  int			  Yysize;
  unsigned char *U;
  unsigned char *V;
  int			  Cxsize;
  int			  Cysize;
} TYV12;

#endif 