/* 
 *  demux.h 
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef DEMUX_H
#define DEMUX_H


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <windows.h>
#include <winbase.h>

#include "..\mism.h"


#define IDCT_MMX	1
#define IDCT_NORMAL	2
#define IDCT_IEEE	3

#define MPEG1_CLK_REF  (i64)90000
#define MPEG2_CLK_REF  (i64)27000000


// Stream IDs
#define PACK_ID					 0xBA
#define PROGRAM_STREAM_MAP       0xBC
#define PRIVATE_STREAM_1         0xBD
#define PADDING_STREAM           0xBE
#define PRIVATE_STREAM_2         0xBF
#define ECM_STREAM               0xF0
#define EMM_STREAM               0xF1
#define PROGRAM_STREAM_DIRECTORY 0xFF
#define DSMCC_STREAM             0xF2
#define ITUTRECH222TYPEE_STREAM  0xF8
#define SUBSTREAM_AC3            0x80
#define VIDEO_STREAM             0xE0
#define AUDIO_STREAM             0xC0


// PES definition
struct PESinfo
{

bool  pack_header_parsed; // flag that indicates if the PES was the first in the Pack
unsigned int  pack_bytes; // number of bytes inside the Pack from the byte containing
                          // the last bit of SCRbase field

int   streamID;		  // streamID of the PES
int	  subStreamID;	  // subStreamID of the PES

                      // Pack info
__int64   SCR;            // System Clock Reference referred to a 27Mhz clock
int    muxrate;        // Pack muxrate in bytes per second

                      // PES info
__int64   PTS;            // Presentation Time Stamp referred to a 27Mhz clock
unsigned int payloadSize;    // size in bytes of data contained inside the PES packet
bool  isValidData;     // flag to specify if the data returned is valid or not
};


// MPEG2 PACK HEADER
// SCRbase and SCRext
//                     32 .......................................0 9.........0
// [PACK_START_CODE][01---1--][--------][-----1--][--------][-----1--][-------1]

#define ZERO			(i64)0
#define GET_SCRBASE(x) ( ((ui64)x[4]&0xF8)>>3  | \
						  (ui64)x[3]<<5        | \
						 ((ui64)x[2]&0x03)<<13 | \
						 ((ui64)x[2]&0xF8)<<12 | \
						  (ui64)x[1]<<20       | \
						 ((ui64)x[0]&0x03)<<28 | \
						 ((ui64)x[0]&0x38) <<27 )

#define GET_SCREXT(x)  ( ((ui64)x[5]&0xFE)>>1     | \
				         ((ui64)x[4]&0x03)<<7 )

// muxrate
//      22 ......................0         stl
// ... [--------][--------][------11][rrrrr---]
#define GET_MPEG2MUXRATE(x) ( (ui32)x[6]<<14 | ((ui32)x[7])<<6 | ((ui32)x[8]&0x03)>>2)

#define GET_MPEG2STUFFING(x) ((x[9]&0x07)) 
// MPEG1 PACK HEADER
//                       SCR                                            muxrate
//                       32........................................0    22.......................0
// [PACK_START_CODE][0010---1][--------][-------1][--------][-------1][1-------][--------][-------1]
#define GET_SCR(x)     ( ((ui64)x[4]&0xFE) >>1     | \
				         ((ui64)x[3])      <<7     | \
                         ((ui64)x[2]&0xFE) <<14    | \
                         ((ui64)x[1])      <<22    | \
                         ((ui64)x[0]&0x0E) <<29 )
#define GET_MPEG1_PTS(x) (GET_SCR(x)) //they follow the same pattern

#define GET_MPEG1_MUXRATE(x) ( ((ui32)x[7]&0xFE) >>1     | \
                               ((ui32)x[6])	     <<7     | \
							   ((ui32)x[5]&0x7F) <<15 )

// MPEG2 PES packet (first 5 bytes)
//                     Packet lenght                 PTS       Header length
//                     15...............0                      7......0            
// [PACKET_START_CODE][--------][--------][10------][xx------][--------]
#define GET_MPEG_PACKET_LENGHT(x) ( ((ui16)x[0]<<8) | (ui16)x[1] )
#define GET_MPEG2_PTS_FLAGS(x) ( ((ui8)x[3]&0xC0)>>6  )
// MPEG2 PES packet (optional parameters)
//                         PTS                      
//                         32........................................0
// [PACKET_START_CODE][001x---1][--------][-------1][--------][-------1]
#define GET_MPEG2_PTS(x)   ( ((ui64)x[4]&0xFE) >>1     | \
      				         ((ui64)x[3])      <<7     | \
                             ((ui64)x[2]&0xFE) <<14    | \
                             ((ui64)x[1])      <<22    | \
                             ((ui64)x[0]&0x0E) <<29 )


#define IS_MPEG2PACKHEADER(x) (    ((x[0]&0xC4)==0x44) && ((x[2]&0x04)==0x04) \
&& ((x[4]&0x04)==0x04) &&  (x[5]&0x01)        )

#define IS_MPEG1PACKHEADER(x) (    ((x[0]&0xF1)==0x21) && (x[2]&0x01) \
&&  (x[4]&0x01) )

#define GET_UINT16(x)   ( ((ui16)x[0]<<8)|((ui16)x[1]))

#define POS_ENG_SIZE 2048 

class CDemux
{
public:

	// Rewinds the stream to the beginning of the previous
	// PES
	bool RewindPreviousPES( ui8 nStreamID );
	void StartReadLPES();
	bool ReadLPES(unsigned char **buffer, PESinfo *PES);
	i64 GetTime();

	CDemux();
	~CDemux();

	bool ReadPES(unsigned char **buffer, PESinfo *PES);

	void ResetSCR();
	void ResetPTS();

  bool SetStreamPos(ui64 pos);
  ui64 GetStreamPos();
  char *GetFileName();
  ui64 GetStreamSize();

  // Retrieve the beginning of the
  // pack that best closes this position
  ui64 GetSyncPoint(ui64 pos);

	i64 getSCR();
	i64	getPTS();

	int         isMPEG2;
	ui32		muxRate;


	bool		PTSread;
	int		SetInput(LPTWorkingMism pMismInfo);

	bool				EndOfFile;

protected:
	ui8    inbuf[65536];
	i64     SCR, SCRbase,SCRext, PTS;
	int     AlignPack();


private:
  fmHandle strHandle;
  int (*mismReadStream)(fmHandle strHandle, ui8 *buf, unsigned int size); 
  int ReadStream(ui8 *buf, unsigned int size);
  LPTWorkingMism inp;
	bool firstTime;
	 i64 delta;
	 i64 lastSCR;
	ui32 lastPackBytes;
	ui32 lastMuxRate;
	ui32 pack_bytes;

	// Private methods
	bool GetBackByte( ui8 *byte );
	bool GetFordByte( ui8 *byte );
  bool GetFordDWord( ui32 *dword );	
	bool GetForWord( ui16 *word );

  // Performance-wise functions
  // to seek the stream backwards and fordward.

  // Creates the positioning buffer
  // And positions the pointers from the current position
  void StartPositioningEngine()
  {
    m_pPosEngBfr = new ui8[POS_ENG_SIZE];
    // Get position
    ui64 nActualPos = GetStreamPos();
    SetPosEngPos( nActualPos );
  };
  void StopPositioningEngine()
  {
    // Seek the stream to the actual position
    SetStreamPos( m_nBasePos + m_nPosEngPtr );
    // Delete buffer
    delete []m_pPosEngBfr;
  }
  // Grabs previous or next chunk of data.
  bool GrabPrevChunk() // positions the pointer at the end
  {
    if( ((i64)m_nBasePos - POS_ENG_SIZE)  < 0 )
      return false;
    
    SetStreamPos( m_nBasePos - POS_ENG_SIZE );
    m_nBasePos -= POS_ENG_SIZE;
    m_nPosEngSize = ReadStream( m_pPosEngBfr, POS_ENG_SIZE );    
    m_nPosEngPtr  = m_nPosEngSize;
    return true;
  }
  bool GrabNextChunk() // positions the pointer at the beginning
  {
    if( (m_nBasePos + POS_ENG_SIZE)  > GetStreamSize() )
      return false;
    
    SetStreamPos( m_nBasePos + POS_ENG_SIZE );
    m_nBasePos += POS_ENG_SIZE;
    m_nPosEngSize = ReadStream( m_pPosEngBfr, POS_ENG_SIZE );    
    m_nPosEngPtr  = 0;
    return true;
  }

  ui64 GetPosEngPos(){
    return m_nBasePos + m_nPosEngPtr;
  }
  bool SetPosEngPos(ui64 pos){
    m_nPosEngPtr = (i32)(pos % POS_ENG_SIZE);
    m_nBasePos   = pos - m_nPosEngPtr;
    // Now, read the chunk
    // First position the stream at the base
    SetStreamPos( m_nBasePos );
    m_nPosEngSize = ReadStream( m_pPosEngBfr, POS_ENG_SIZE );
    return true;
  };
  // Properties
  ui8 *m_pPosEngBfr;
   i32 m_nPosEngPtr;
  ui32 m_nPosEngSize;
  ui64 m_nBasePos;

	
	
};




#endif  DEMUX_H