/* 
 *  AudioCompressor.cpp 
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <windows.h>
#include <mmsystem.h>
#include <malloc.h>
#include <winbase.h>
#include <math.h>
#include <crtdbg.h>

#include "AudioCompressor.h"

#undef AUDCOMPRESSOR_TRACE

CAudioCompressor::CAudioCompressor()
{
  m_bEnabled = false;
}

CAudioCompressor::~CAudioCompressor()
{
}

int CAudioCompressor::CreateCompressor(CAsyncBuffer *ibuf)
{
	if(!ibuf)
		return 0;
	this->inbuf = ibuf;
	return CreateCompressor();
}
int CAudioCompressor::read(char *buffer)
{
	char *insamp;
	inbuf->ReadBuffer( &insamp, RMS_WINDOW*4 );
  if(m_bEnabled)
	  Compress( insamp, RMS_WINDOW*4 );
	memcpy( buffer, insamp, RMS_WINDOW*4 );

	return RMS_WINDOW*4;
}

int CAudioCompressor::CreateCompressor()
{
	TKnot myKnot;
	int i;

  knots.EmptyArray();

	//Initialize knots
	for(i=DEFAULT_KNOTS-1; i>=0; i--)
	{
		myKnot.in_dBLevel  = -(float)(i+1)*((float)DB_RANGE/(float)(DEFAULT_KNOTS+1));
		myKnot.out_dBLevel = myKnot.in_dBLevel;
		knots.AddItem( &myKnot );
	}
	AdjustKnots(5);
  past_gain=0;
	return 1;
}

int CAudioCompressor::AdjustKnots(int step)
{
	int i;
	// Adjust out_DBLevels of every knot
	float free_range;
	float knot_step;

	if(step<0)
		step=0;
	if(step>=COMPRESSOR_STEPS)
		step=COMPRESSOR_STEPS-1;

	for(i=0; i<knots.GetCount(); i++)
	{
		// workout free dynamic range up to FS
		free_range = 0 - knots[i].in_dBLevel;
		// workout knot_step
		knot_step  = free_range/(float)COMPRESSOR_STEPS;

		knots[i].out_dBLevel = knots[i].in_dBLevel + knot_step*step;
	}

	return 1;
}

float CAudioCompressor::GetPower(char *data, int data_count)
{
	int i;
	double sum,tot,mono;
	sum=0;
	for(i=0; i<data_count>>1; i+=2)
	{
		mono = ( ((short *)data)[i] + ((short *)data)[i+1] )/2;
		sum += mono*mono;
	}
	tot = sum/ (double)(data_count>>2);
	if(tot==0) tot=10;
	return (float) (10*log10(  tot /(double)(32768*32768)) );
}

float inline CAudioCompressor::GetOutPowFromInPow(float in_pow)
{
	float slope, offset;
	TKnot this_knot, prev_knot;
	int   i;

	// Figure out between to what knots
	//   this in_pow belongs

	prev_knot.in_dBLevel = -DB_RANGE;
	prev_knot.out_dBLevel = -DB_RANGE;

	if(in_pow>=0) return in_pow;

	for(i=0; i<=knots.GetCount(); i++)
	{	
		if(i<knots.GetCount())
		{
			this_knot = knots[i];
		}
		else{
			this_knot.in_dBLevel = 0;
			this_knot.out_dBLevel = prev_knot.out_dBLevel;
		}

		if(in_pow > this_knot.in_dBLevel)
		{
			prev_knot = this_knot;
			continue;
		}
		
		slope = (this_knot.out_dBLevel - prev_knot.out_dBLevel)
			    /(this_knot.in_dBLevel - prev_knot.in_dBLevel);

		offset = prev_knot.out_dBLevel - slope*prev_knot.in_dBLevel;

		return (slope*in_pow+offset);


	}
	return in_pow;
}

#define CLIP(x) (x> 32767 ? 32767 : x <-32767 ? -32767 : x)
int CAudioCompressor::AdjustPower(char *data, int data_count, float power_gain)
{
	int i;
	float linear_gain;
	short *samples;
	samples = (short *)data;
	// convert power_gain into linear gain
	linear_gain = (float)pow(10.0, power_gain/20.0);

	for(i=0; i<data_count>>1; i+=2)
	{
		samples[i]    = (short)( CLIP((float)samples[i]*linear_gain));
		samples[i+1]  = (short)(CLIP((float)samples[i+1]*linear_gain));
	}
	return 1;
}

int CAudioCompressor::Compress(char *data, int data_count)
{
	float in_power, out_power, gain;

	 in_power = GetPower(data, data_count);
	out_power = GetOutPowFromInPow(in_power);

#ifdef AUDCOMPRESSOR_TRACE
	_RPT3(_CRT_WARN, "Gain: %0.2f dB     Out Power: %0.2f dB    In Power: %0.2f dB\n", out_power-in_power, out_power, in_power);
#endif

	gain = out_power - in_power;
	past_gain = float(0.95*past_gain + 0.05*gain);
	AdjustPower( data, data_count, past_gain);

	return 1;

}
