/* 
 *  VideoSequenceCompressor.cpp
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "stdafx.h"
#include "VideoSequenceCompressor.h"
#include "..\include\floutapi.h"

//////////////////////////////////////////////////////////////////////////////
//
//	IMITATING WIN2K AVISAVEV() BEHAVIOR IN 0x7FFFFFFF EASY STEPS
//
//	ICM_COMPRESS_FRAMES_INFO:
//
//		dwFlags			Trashed with address of lKeyRate in tests. Something
//						might be looking for a non-zero value here, so better
//						set it.
//		lpbiOutput		NULL.
//		lOutput			0.
//		lpbiInput		NULL.
//		lInput			0.
//		lStartFrame		0.
//		lFrameCount		Number of frames.
//		lQuality		Set to quality factor, or zero if not supported.
//		lDataRate		Set to data rate in 1024*kilobytes, or zero if not
//						supported.
//		lKeyRate		Set to the desired maximum keyframe interval.  For
//						all keyframes, set to 1.		
//
//	ICM_COMPRESS:
//
//		dwFlags			Equal to ICCOMPRESS_KEYFRAME if a keyframe is
//						required, and zero otherwise.
//		lpckid			Always points to zero.
//		lpdwFlags		Points to AVIIF_KEYFRAME if a keyframe is required,
//						and zero otherwise.
//		lFrameNum		Ascending from zero.
//		dwFrameSize		Always set to 7FFFFFFF (Win9x) or 00FFFFFF (WinNT)
//						for first frame.  Set to zero for subsequent frames
//						if data rate control is not active or not supported,
//						and to the desired frame size in bytes if it is.
//		dwQuality		Set to quality factor from 0-10000 if quality is
//						supported.  Otherwise, it is zero.
//		lpbiPrev		Set to NULL if not required.
//		lpPrev			Set to NULL if not required.
//
//////////////////////////////////////////////////////////////////////////////

VideoSequenceCompressor::VideoSequenceCompressor() {
	pPrevBuffer		= NULL;
	pOutputBuffer	= NULL;
	pConfigData		= NULL;
}

VideoSequenceCompressor::~VideoSequenceCompressor() {

	delete pbiInput;
	delete pbiOutput;

	finish();

	delete pConfigData;
	delete pOutputBuffer;
	delete pPrevBuffer;
}


bool VideoSequenceCompressor::EnumerateVideoCodecs(int width, int height)
{
  // Fill video codec list
  m_vVideoCodecList.RemoveAll();
  BITMAPINFO       bmpInfo;
  BITMAPINFOHEADER bi;
  
  memset(&bi, 0, sizeof(BITMAPINFOHEADER));
  bi.biSize        = sizeof(BITMAPINFOHEADER);
  bi.biWidth       = width;
  bi.biHeight      = height;
  bi.biPlanes      = 1;
  bi.biBitCount    = 24;
  bi.biCompression = BI_RGB;
  bi.biSizeImage   = 0;
  
  bmpInfo.bmiHeader = bi;
  
  HIC hic; 
  int i=0,j=0;
  for (i=0; i<m_vInstalledCodecList.GetSize(); i++) 
  { 
      hic = m_vInstalledCodecList[i].hic;

      DWORD dwSupportedFormats = 0;

      // RGB
      bmpInfo.bmiHeader.biCompression = BI_RGB;
      bmpInfo.bmiHeader.biBitCount    = 24;
      // Skip this compressor if it can't handle the format. 
      dwSupportedFormats = ICCompressQuery(hic, &bmpInfo, NULL) == ICERR_OK ? FLO_VIDEO_RGB32 : 0;

      // YV12
      bmpInfo.bmiHeader.biCompression = '21VY';
      bmpInfo.bmiHeader.biBitCount    = 12;
      // Skip this compressor if it can't handle the format. 
      dwSupportedFormats |= ICCompressQuery(hic, &bmpInfo, NULL) == ICERR_OK ? FLO_VIDEO_YV12 : 0;

      // YUY2
      bmpInfo.bmiHeader.biCompression = '2YUY';
      bmpInfo.bmiHeader.biBitCount    = 16;
      // Skip this compressor if it can't handle the format. 
      dwSupportedFormats |= ICCompressQuery(hic, &bmpInfo, NULL) == ICERR_OK ? FLO_VIDEO_YUY2 : 0;

      m_vInstalledCodecList[i].dwSupportedFormats = dwSupportedFormats;

      if( !dwSupportedFormats )
        continue; 
 
      TVideoCodecInfo cnfo;
      cnfo = m_vInstalledCodecList[i];
    
      m_vVideoCodecList.SetAtGrow( j++, cnfo );
      
  }
  return true;
}

BITMAPINFOHEADER *VideoSequenceCompressor::GetOutputFormat(){return (BITMAPINFOHEADER *)pbiOutput;};
int VideoSequenceCompressor::GetOutputFormatSize(){return pbiOutput->bmiHeader.biSize + pbiOutput->bmiHeader.biClrUsed*4;};

void VideoSequenceCompressor::init(HIC hic, BITMAPINFO *pbiInput, BITMAPINFO *pbiOutput, long lQ, long lKeyRate) {
	ICINFO	info;
	LRESULT	res;
	int cbSizeIn, cbSizeOut;

	cbSizeIn = pbiInput->bmiHeader.biSize + pbiInput->bmiHeader.biClrUsed*4;
	cbSizeOut = pbiOutput->bmiHeader.biSize + pbiOutput->bmiHeader.biClrUsed*4;

	this->hic		= hic;
	this->pbiInput	= (BITMAPINFO *) new char[cbSizeIn];
  this->pbiOutput	= (BITMAPINFO *) new char[cbSizeOut];
	this->lKeyRate	= lKeyRate;

	memcpy(this->pbiInput, pbiInput, cbSizeIn);
	memcpy(this->pbiOutput, pbiOutput, cbSizeOut);

	lKeyRateCounter = 1;

	// Retrieve compressor information.

	res = ICGetInfo(hic, &info, sizeof info);

	if (!res)
		throw "Unable to retrieve video compressor information.";

	// Analyze compressor.

	this->dwFlags = info.dwFlags;

	if (info.dwFlags & VIDCF_TEMPORAL) {
		if (!(info.dwFlags & VIDCF_FASTTEMPORALC)) {
			// Allocate backbuffer

			if (!(pPrevBuffer = new char[pbiInput->bmiHeader.biSizeImage]))
				throw "VideoSequenceCompressor::init - Memory allocation error";
		}
	}

	if (info.dwFlags & VIDCF_QUALITY)
		lQuality = lQ;
	else
		lQuality = 0;

	// Allocate destination buffer

	lMaxPackedSize = ICCompressGetSize(hic, pbiInput, pbiOutput);

	if (!(pOutputBuffer = new char[lMaxPackedSize]))
		throw "VideoSequenceCompressor::init - Memory allocation error";

	// Save configuration state.
	//
	// Ordinarily, we wouldn't do this, but there seems to be a bug in
	// the Microsoft MPEG-4 compressor that causes it to reset its
	// configuration data after a compression session.  This occurs
	// in all versions from V1 through V3.
	//
	// Stupid fscking Matrox driver returns -1!!!

	cbConfigData = ICGetStateSize(hic);

	if (cbConfigData > 0) {
		if (!(pConfigData = new char[cbConfigData]))
			throw "VideoSequenceCompressor::init - Memory allocation error";

		cbConfigData = ICGetState(hic, pConfigData, cbConfigData);

		// As odd as this may seem, if this isn't done, then the Indeo5
		// compressor won't allow data rate control until the next
		// compression operation!

		if (cbConfigData)
			ICSetState(hic, pConfigData, cbConfigData);
	}

	lMaxFrameSize = 0;
	lSlopSpace = 0;
}

void VideoSequenceCompressor::setDataRate(long lDataRate, long lUsPerFrame, long lFrameCount) {

	if (lDataRate && (dwFlags & VIDCF_CRUNCH))
		lMaxFrameSize = MulDiv(lDataRate, lUsPerFrame, 1000000);
	else
		lMaxFrameSize = 0;

	// Indeo 5 needs this message for data rate clamping.

	// The Morgan codec requires the message otherwise it assumes 100%
	// quality :(

	// The original version (2700) MPEG-4 V1 requires this message, period.
	// V3 (DivX) gives crap if we don't send it.  So special case it.

	ICINFO ici;

	ICGetInfo(hic, &ici, sizeof ici);

	{
		ICCOMPRESSFRAMES icf;

		memset(&icf, 0, sizeof icf);

		icf.dwFlags		= (DWORD)&icf.lKeyRate;
		icf.lStartFrame = 0;
		icf.lFrameCount = lFrameCount;
		icf.lQuality	= lQuality;
		icf.lDataRate	= lDataRate;
		icf.lKeyRate	= lKeyRate;
		icf.dwRate		= 1000000;
		icf.dwScale		= lUsPerFrame;

		ICSendMessage(hic, ICM_COMPRESS_FRAMES_INFO, (WPARAM)&icf, sizeof(ICCOMPRESSFRAMES));
	}
}

void VideoSequenceCompressor::start() {
	LRESULT	res;

	// Start compression process

	res = ICCompressBegin(hic, pbiInput, pbiOutput);

	if (res != ICERR_OK)
		throw "Unable to start video compression";

	// Start decompression process if necessary

	if (pPrevBuffer) {
		res = ICDecompressBegin(hic, pbiOutput, pbiInput);

		if (res != ICERR_OK) {
			ICCompressEnd(hic);
			throw "Unable to start video compression";
		}
	}

	fCompressionStarted = true;
	lFrameNum = 0;
}

void VideoSequenceCompressor::finish() {
	if (!fCompressionStarted)
		return;

	if (pPrevBuffer)
		ICDecompressEnd(hic);

	ICCompressEnd(hic);

	fCompressionStarted = false;

	// Reset MPEG-4 compressor

	if (cbConfigData && pConfigData)
		ICSetState(hic, pConfigData, cbConfigData);
}

void VideoSequenceCompressor::dropFrame() {
	if (lKeyRate && lKeyRateCounter>1)
		--lKeyRateCounter;

	++lFrameNum;
}

void *VideoSequenceCompressor::packFrame(void *pBits, bool *pfKeyframe, long *plSize) {
	DWORD dwChunkId = 0;
	DWORD dwFlags=0, dwFlagsIn = ICCOMPRESS_KEYFRAME;
	DWORD res;
	DWORD sizeImage;
	long lAllowableFrameSize=0;//xFFFFFF;	// yes, this is illegal according
											// to the docs (see below)

	// Figure out if we should force a keyframe.  If we don't have any
	// keyframe interval, force only the first frame.  Otherwise, make
	// sure that the key interval is lKeyRate or less.  We count from
	// the last emitted keyframe, since the compressor can opt to
	// make keyframes on its own.

	if (!lKeyRate) {
		if (lFrameNum)
			dwFlagsIn = 0;
	} else {
		if (--lKeyRateCounter)
			dwFlagsIn = 0;
		else
			lKeyRateCounter = lKeyRate;
	}

	// Figure out how much space to give the compressor, if we are using
	// data rate stricting.  If the compressor takes up less than quota
	// on a frame, save the space for later frames.  If the compressor
	// uses too much, reduce the quota for successive frames, but do not
	// reduce below half datarate.

	if (lMaxFrameSize) {
		lAllowableFrameSize = lMaxFrameSize + (lSlopSpace>>2);

		if (lAllowableFrameSize < (lMaxFrameSize>>1))
			lAllowableFrameSize = lMaxFrameSize>>1;
	}

	// A couple of notes:
	//
	//	o  ICSeqCompressFrame() passes 0x7FFFFFFF when data rate control
	//	   is inactive.  Docs say 0.  We pass 0x7FFFFFFF here to avoid
	//	   a bug in the Indeo 5 QC driver, which page faults if
	//	   keyframe interval=0 and max frame size = 0.

	sizeImage = pbiOutput->bmiHeader.biSizeImage;

//	pbiOutput->bmiHeader.biSizeImage = 0;

	// Compress!

	if (dwFlagsIn)
		dwFlags = AVIIF_KEYFRAME;

	res = ICCompress(hic, dwFlagsIn,
			(LPBITMAPINFOHEADER)pbiOutput, pOutputBuffer,
			(LPBITMAPINFOHEADER)pbiInput, pBits,
			&dwChunkId,
			&dwFlags,
			lFrameNum,
			lFrameNum ? lAllowableFrameSize : 0xFFFFFF,
			lQuality,
			dwFlagsIn & ICCOMPRESS_KEYFRAME ? NULL : (LPBITMAPINFOHEADER)pbiInput,
			dwFlagsIn & ICCOMPRESS_KEYFRAME ? NULL : pPrevBuffer);

	// _RPT2(0,"Compressed frame %d: %d bytes\n", lFrameNum, pbiOutput->bmiHeader.biSizeImage);

	++lFrameNum;

	*plSize = pbiOutput->bmiHeader.biSizeImage;

	// If we're using a compressor with a stupid algorithm (Microsoft Video 1),
	// we have to decompress the frame again to compress the next one....

	if (res==ICERR_OK && pPrevBuffer && (!lKeyRate || lKeyRateCounter>1)) {
		res = ICDecompress(hic, dwFlags & AVIIF_KEYFRAME ? 0 : ICDECOMPRESS_NOTKEYFRAME
				,(LPBITMAPINFOHEADER)pbiOutput
				,pOutputBuffer
				,(LPBITMAPINFOHEADER)pbiInput
				,pPrevBuffer);
	}

	pbiOutput->bmiHeader.biSizeImage = sizeImage;

	if (res != ICERR_OK)
		throw "Video compression";

	// Update quota.

	if (lMaxFrameSize) {
		lSlopSpace += lMaxFrameSize - *plSize;

		_RPT3(0,"Compression: allowed %d, actual %d, slop %+d\n", lAllowableFrameSize, *plSize, lSlopSpace);
	}

	// Was it a keyframe?

	if (dwFlags & AVIIF_KEYFRAME) {
		*pfKeyframe = true;
		lKeyRateCounter = lKeyRate;
	} else {
		*pfKeyframe = false;
	}

	return pOutputBuffer;
}

bool VideoSequenceCompressor::CodecIsInstalled(DWORD fccHandler, int *idx )
{
  // just go through all codecs to see if this one is installed
  for( int i=0; i< m_vVideoCodecList.GetSize(); i++ )
  {
    if( m_vVideoCodecList[i].fccHandler == fccHandler )
    {
      *idx = i;
      return true;
    }
  }
  return false;
}

bool VideoSequenceCompressor::DeInitVideoCompressor()
{
  finish();
  return true;
}
bool VideoSequenceCompressor::InitVideoCompressor(DWORD fccHandler, HIC hic, int width, int height, int rate, int scale,
                                                  DWORD dwInFccHandler, int nInBitCount, PBYTE settings, long settingssize )
{
  if( !hic )
  {
    // the video compressor was not open. Try to open it from the fccHandler
    if( !(hic = ICOpen(ICTYPE_VIDEO, fccHandler, ICMODE_COMPRESS )) )
      return false;
  }
  // At this point we have an opened compressor at hic.
  BITMAPINFOHEADER       inBmp, *outBmp;
  
  memset(&inBmp, 0, sizeof(BITMAPINFOHEADER));
  inBmp.biSize        = sizeof(BITMAPINFOHEADER);
  inBmp.biWidth       = width;
  inBmp.biHeight      = height;
  inBmp.biPlanes      = 1;
  inBmp.biBitCount    = nInBitCount;
  inBmp.biCompression = dwInFccHandler;
  inBmp.biSizeImage   = 0;
  
  // If settings are provided, set them
  if( settings )
    ICSetState( hic, settings, settingssize );

  // Make sure we support the input format
  if( ICCompressQuery( hic, &inBmp, NULL ) != ICERR_OK )
    return false;
    
  // Get the size of the format
  int outBmpSize = ICCompressGetFormatSize( hic, &inBmp );
  if( !outBmpSize )
    return false;

  outBmp = (BITMAPINFOHEADER *)new BYTE[outBmpSize];
  if( ICCompressGetFormat( hic, &inBmp, outBmp )!= ICERR_OK )
    return false;

  try
  {
    init( hic, (BITMAPINFO *)&inBmp, (BITMAPINFO *)outBmp, 10000, 1);
    int nUsPerFrame = MulDiv(1000000, scale, rate);
    setDataRate( 1000000, nUsPerFrame, 0x0FFFFFFF);
    start();
  }
  catch( ... )
  {
    return false;
  }
  // delete outBmp
  delete []outBmp;

  return true;
}

// Retrieves the current codecs installed in the system
// and opens them.
bool VideoSequenceCompressor::GetInstalledCodecs()
{
  // Fill video codec list
  m_vInstalledCodecList.RemoveAll();
  
  ICINFO codecInfo;
  HIC hic; 
  int i=0,j=0;
  for (i=0; ICInfo(ICTYPE_VIDEO, i, &codecInfo); i++) 
  { 
    hic = ICOpen(codecInfo.fccType, codecInfo.fccHandler, ICMODE_COMPRESS ); 
    if (hic) 
    {
      if(codecInfo.fccType != ICTYPE_VIDEO)
      {
        ICClose(hic);
        continue;
      }
      // Find out the compressor name. 
      ICGetInfo(hic, &codecInfo, sizeof(codecInfo)); 
      
      TVideoCodecInfo cnfo;
      cnfo.hic                   = hic;
      cnfo.fccHandler            = codecInfo.fccHandler;
      memcpy( cnfo.szDescription, codecInfo.szDescription, 128*sizeof(WCHAR) );
      memcpy( cnfo.szDriver     , codecInfo.szDriver     , 128*sizeof(WCHAR) );
      memcpy( cnfo.szName       , codecInfo.szName       , 16*sizeof(WCHAR) );

      // Get State
#if 1
      DWORD size;
      size = ICGetStateSize(hic);
      if( size )
      {
        PBYTE pv = new BYTE[size];
        ICGetState( hic, pv, size );
        cnfo.SetSettings( pv, size );
        cnfo.SetSettings1st( pv, size);
        cnfo.SetSettings2nd( pv, size );
        delete []pv;
      }
#endif
      m_vInstalledCodecList.SetAtGrow( j++, cnfo );
      
    }      
  }
  EnumerateVideoCodecs( 640, 480 );
  return true;
}

bool VideoSequenceCompressor::CloseCompressors()
{
  int i=0;
  for (i=0; i<m_vInstalledCodecList.GetSize(); i++) 
    ICClose( m_vInstalledCodecList[i].hic );
  return true;
}
