/* 
 *  OpenDMLAVIOutput.cpp
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "stdafx.h"
#include "error.h"
#include "OpenDMLAVIOutput.h"
#include "OptionsDlg.h"
#include <mmsystem.h>
#include <mmreg.h>
#include <msacm.h>
#include "debug.h"
#include "settingsmanager.h"

#pragma warning( disable : 4731  )

 
//
//	Note!
//
//		If this DLL is dynamically linked against the MFC
//		DLLs, any functions exported from this DLL which
//		call into MFC must have the AFX_MANAGE_STATE macro
//		added at the very beginning of the function.
//
//		For example:
//
//		extern "C" BOOL PASCAL EXPORT ExportedFunction()
//		{
//			AFX_MANAGE_STATE(AfxGetStaticModuleState());
//			// normal function body here
//		}
//
//		It is very important that this macro appear in each
//		function, prior to any calls into MFC.  This means that
//		it must appear as the first statement within the 
//		function, even before any object variable declarations
//		as their constructors may generate calls into the MFC
//		DLL.
//
//		Please see MFC Technical Notes 33 and 58 for additional
//		details.
//

/////////////////////////////////////////////////////////////////////////////
// COpenDMLAVIOutputApp

BEGIN_MESSAGE_MAP(COpenDMLAVIOutputApp, CWinApp)
	//{{AFX_MSG_MAP(COpenDMLAVIOutputApp)
		// NOTE - the ClassWizard will add and remove mapping macros here.
		//    DO NOT EDIT what you see in these blocks of generated code!
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// COpenDMLAVIOutputApp construction

COpenDMLAVIOutputApp::COpenDMLAVIOutputApp() : m_pAVIFile(NULL)
{
}

/////////////////////////////////////////////////////////////////////////////
// The one and only COpenDMLAVIOutputApp object
COpenDMLAVIOutputApp theApp;

static flohandle_t id = 0;

flreturn_t floutentry_getids( flohandle_t **ids )
{
  *ids = &id;
  return (flreturn_t)1;
}

flreturn_t floutentry( flohandle_t id, flselector_t sel, ui32 par1, ui32 par2 )
{
  AFX_MANAGE_STATE(AfxGetStaticModuleState());
  flreturn_t result = flo_ok;

  switch (sel)
  {
	 case flo_startup:
     result = theApp.startup( (startup_s *)par1 );
     break;
   case flo_shutdown:
     result = theApp.shutdown();
     break;
   case flo_compile:
     result = theApp.compile( (compile_s *)par1 );
     break;
   case flo_getinfo:
     result = theApp.getinfo( par1, par2 );
     break;
   case flo_setinfo:
     result = theApp.setinfo( par1, par2 );
     break;
   case flo_configure:
     result = theApp.configure();
     break;
  }

  return result;
}


flreturn_t COpenDMLAVIOutputApp::startup( startup_s *st )
{

  strcpy( st->modulename, "AVI Output" );
  st->moduleid = ODMLtype;

  strcpy( m_vVideoFormatName, "AVI Video");
  strcpy( m_vAudioFormatName, "AVI Audio");
  UpdateCodecNames();

  st->doesvideo = 1;
  st->doesaudio = 1;
  st->dualpass  = 1;

  m_oVidComp.GetInstalledCodecs();
  m_oAudComp.EnumerateAudioCodecs(44100, 2, 16, true );
  m_oAudComp.EnumerateAudioCodecs(48000, 2, 16, false); 
  return flo_ok;
}

flreturn_t COpenDMLAVIOutputApp::shutdown()
{
  m_oVidComp.CloseCompressors();  
  return flo_ok;
}

flreturn_t COpenDMLAVIOutputApp::configure()
{
  if( m_oAudComp.EnumerateAudioCodecs(44100, 2, 16, true ) &&
      m_oAudComp.EnumerateAudioCodecs(48000, 2, 16, false) && 
      m_oVidComp.EnumerateVideoCodecs(640, 480) )
  {    
    ShowOptionsDlg();
  }
  return flo_ok;
}

flreturn_t COpenDMLAVIOutputApp::getinfo(ui32 par1, ui32 par2)
{
  videoinfo_s *vi;
  audioinfo_s *ai;
  cfgblockinfo_s *bi;
  fileinfo_s *fi;

  switch( par1 )
  {
    case flo_getvideoinfo:
      vi = (videoinfo_s *)par2;
      UpdateCodecNames();
      strcpy( vi->fmtname, m_vVideoFormatName );
      vi->supported_fmt = m_oStateManager.GetVidSupportedFormats();
      break;
    case flo_getaudioinfo:
      UpdateCodecNames();
      ai = (audioinfo_s *)par2;
      strcpy( ai->fmtname, m_vAudioFormatName );
      break;
    case flo_getblockinfo:
      bi = (cfgblockinfo_s *)par2;

      long nBufferSize;
      sm.CreateBlock( (PBYTE)bi->buf, nBufferSize,
                      m_oStateManager.GetFccHandler(),
                      (DWORD)m_oStateManager.GetAcmDriverID(),
                      m_oStateManager.GetCWfx(),
                      m_oVidComp.m_vVideoCodecList );
      bi->bufsize = nBufferSize;
      break;
    case flo_getfileinfo:
      fi = (fileinfo_s *)par2;
      strcpy(fi->extension, "avi");
      break;
  }
  return flo_ok;
}

flreturn_t COpenDMLAVIOutputApp::setinfo(ui32 par1, ui32 par2)
{
  cfgblockinfo_s *bi;
  switch( par1 )
  {
    case flo_setblockinfo:
      bi = (cfgblockinfo_s *)par2;

      DWORD nFourCC;
      DWORD nHadId;

      if ( sm.CreateFromBlock( bi->buf, bi->bufsize, 
                               nFourCC, nHadId, m_oStateManager.GetCWfx(),
                               m_oVidComp.m_vInstalledCodecList ) ) 
      {
        m_oStateManager.SetAcmDriverID((HACMDRIVERID)nHadId);
        // Look for a codec with this FourCC
        bool bFound = false;
        for( int i=0; i<m_oVidComp.m_vInstalledCodecList.GetSize(); i++ )
        {
          if( m_oVidComp.m_vInstalledCodecList[i].fccHandler == nFourCC ) {
            m_oStateManager.SetVideoCodec( m_oVidComp.m_vInstalledCodecList[i] );
            bFound = true;
          }
        }
        if( !bFound )
          m_oStateManager.SetVideoUncompressed( FLO_VIDEO_RGB32 | FLO_VIDEO_YUY2 );
      }
      break;
  }
  return flo_ok;
}

flreturn_t COpenDMLAVIOutputApp::compile(compile_s *cs)
{
  videodef_s *vd=NULL;
  audiodef_s *ad=NULL;

 

  DBGOUTFMT(( buf, "compile: parsing defs\n"))
  // Setup the streams of the conversion
  for( ui32 i=0; i<cs->streamcnt; i++)
  {
    DBGOUTFMT(( buf, "compile: stream %d, type %d, streamdef %d", i, cs->streams[i]->streamtype, cs->streams[i]))
    switch(cs->streams[i]->streamtype)
    {
      case flo_video:
        vd = (videodef_s *)cs->streams[i];
        break;
      case flo_audio:
        ad = (audiodef_s *)cs->streams[i];
        break;
    }
  }

  DBGOUTFMT(( buf, "compile: streamcount=%d vd=%d ad=%d\n", cs->streamcnt, vd, ad))

  if( !ad && !vd )
    return flo_invalidpar;

  // Check that the codecs are in place
  if( !ConfigureCodecs( vd, ad ) )
    return flo_error;

  return DoCompile( cs, vd, ad );
}

bool COpenDMLAVIOutputApp::UpdateCodecNames()
{
  
  bool audioIsCompressed = m_oStateManager.GetWfxSize() > 0;
  bool videoIsCompressed = m_oStateManager.GetFccHandler() != 0;
 
  int idx;
  
  if( videoIsCompressed )
  {
    if(!m_oVidComp.CodecIsInstalled( m_oStateManager.GetFccHandler(), &idx ) )
      videoIsCompressed = false;
  }
  
  if( videoIsCompressed )
    sprintf( m_vVideoFormatName, "%S", m_oVidComp.m_vVideoCodecList[idx].szDescription ); 
  else
    strcpy( m_vVideoFormatName, "Uncompressed"); 
  
  int DriverIdx, formatIdx;
  if( audioIsCompressed )
  {
    if(!m_oAudComp.FormatIsInstalled( m_oStateManager.GetAcmDriverID(),
                                      m_oStateManager.GetWfx(),
                                      m_oStateManager.GetWfxSize(), 
                                      &DriverIdx, &formatIdx) )
      audioIsCompressed = false;
  }

  if( audioIsCompressed )
    strcpy( m_vAudioFormatName, m_oAudComp.m_vAudioCodecList[DriverIdx].szDriverName ); 
  else
    strcpy( m_vAudioFormatName, "Uncompressed"); 

  return true;
  
}
bool COpenDMLAVIOutputApp::ShowOptionsDlg()
{
  // Codecs are supposed to be enumerated already
  // Also a valid state for m_oStateManager is assumed. 
  // Minimal formats include uncompressed audio and video
  
  bool audioIsCompressed = m_oStateManager.GetWfxSize() > 0;
  bool videoIsCompressed = m_oStateManager.GetFccHandler() != 0;
  
  COptionsDlg optDlg;
  
  optDlg.m_vVideoCodecList.Copy( m_oVidComp.m_vVideoCodecList );
  optDlg.m_vAudioCodecList.Copy( m_oAudComp.m_vAudioCodecList );
  
  optDlg.m_sResult.status = 0; 
  int idx;
  
  if( videoIsCompressed )
  {
    if(m_oVidComp.CodecIsInstalled( m_oStateManager.GetFccHandler(), &idx ) )
      optDlg.m_sResult.videoSelected = idx;
    else
      optDlg.m_sResult.status |= VIDEO_UNCOMPRESSED;
  }
  else
    optDlg.m_sResult.status |= VIDEO_UNCOMPRESSED;
  
  
  int DriverIdx, formatIdx;
  if( audioIsCompressed )
  {
    if(m_oAudComp.FormatIsInstalled( m_oStateManager.GetAcmDriverID(),
      m_oStateManager.GetWfx(),
      m_oStateManager.GetWfxSize(), 
      &DriverIdx, &formatIdx) )
    {
      optDlg.m_sResult.audioDriverSelected = DriverIdx;
      optDlg.m_sResult.audioFormatSelected = formatIdx;
    }
    else
      optDlg.m_sResult.status |= AUDIO_UNCOMPRESSED;      
  }
  else
    optDlg.m_sResult.status |= AUDIO_UNCOMPRESSED;      
  
  
  optDlg.DoModal();
  

  // copy video codecs settings back

  CArray< TVideoCodecInfo , TVideoCodecInfo& > *src, *dst;
  src = &optDlg.m_vVideoCodecList;
  dst = &m_oVidComp.m_vInstalledCodecList;

  for( int j=0; j<dst->GetSize(); j++ )
    for( int i=0; i<src->GetSize(); i++ )
      if( (*src)[i].fccHandler == (*dst)[j].fccHandler )
        (*dst)[j] = (*src)[i];

  // check results
  // video
  if( optDlg.m_sResult.status & VIDEO_UNCOMPRESSED )
  {
    m_oStateManager.SetVideoUncompressed( FLO_VIDEO_YUY2|FLO_VIDEO_RGB32 );
  }
  else
  {
    m_oStateManager.SetVideoCodec( 
      m_oVidComp.m_vVideoCodecList[ optDlg.m_sResult.videoSelected ]); 
  }
  // audio
  if( optDlg.m_sResult.status & AUDIO_UNCOMPRESSED )
  {
    // This will remove the actual format in the state manager
    m_oStateManager.SetWfx(NULL, 0);
  }
  else
  {
    DriverIdx    = optDlg.m_sResult.audioDriverSelected;
    formatIdx = optDlg.m_sResult.audioFormatSelected;
    m_oStateManager.SetWfx(
      m_oAudComp.m_vAudioCodecList[DriverIdx].m_vWaveFormatExList[formatIdx].Get(),       // Get WAVEFORMATEX from the list
      m_oAudComp.m_vAudioCodecList[DriverIdx].m_vWaveFormatExList[formatIdx].GetSize() ); // Get Size of WAVEFORMATEX
    m_oStateManager.SetAcmDriverID( m_oAudComp.m_vAudioCodecList[DriverIdx].hadid );
    
  }
  return true;
}

bool COpenDMLAVIOutputApp::ConfigureCodecs(videodef_s *vd, audiodef_s *ad)
{
  bool bSuccess=true, doAudio = ad >0, doVideo = vd>0;
  bool bShowDialog = false;
  int idx;

  if( bSuccess && doVideo )
  {
    // Enumerate current installed codecs
    m_oVidComp.EnumerateVideoCodecs(  vd->width, vd->height );

    // Guess if the codecs selected are valid for the input format we have now. 
    // If the codec is not installed and the selected codec is a compressed one
    if( m_oStateManager.GetFccHandler() )
    { 
      if( !m_oVidComp.CodecIsInstalled( m_oStateManager.GetFccHandler(), &idx ) )
        bShowDialog = true;
      else
        m_oStateManager.SetVideoCodec( m_oVidComp.m_vVideoCodecList[ idx ] );
    }
  }

  if( bSuccess && doAudio )
  {
    m_oAudComp.EnumerateAudioCodecs(  ad->fmt.frequency, ad->fmt.channels, ad->fmt.depth );
    if( m_oStateManager.GetWfxSize() )
    {
      int formatDriverIdx, formatIdx;
      // For audio, change the sample frequency in the state manager to the one
      // we are going to use in the compile.
      // We'll check if that audio format is in fact available in the system. If its not
      // the codec option dialog will pop up. 
      m_oStateManager.GetWfx()->nSamplesPerSec = ad->fmt.frequency;
      m_oStateManager.GetWfx()->nChannels      = ad->fmt.channels;   
      
      if( m_oAudComp.FormatIsInstalled( m_oStateManager.GetAcmDriverID(), 
                                        m_oStateManager.GetWfx(),
                                        m_oStateManager.GetWfxSize(),
                                        &formatDriverIdx,
                                        &formatIdx ) )   {
        // The format stored in the preferences is installed.
        // retrieve hadid
        m_oStateManager.SetAcmDriverID( m_oAudComp.m_vAudioCodecList[formatDriverIdx].hadid );
      }
      else //audio format not installed. Present dialog
      {
        bShowDialog = true;
      }
    }    
  }

  if( bShowDialog )
  {
    bSuccess = ShowOptionsDlg(); // either video or audio codec were not installed.
  }

  return bSuccess;
}


bool COpenDMLAVIOutputApp::InitCompileFormats( videodef_s *vd, 
                                               audiodef_s *ad,
                                              DWORD dwInFccCode,
                                              int  nInBitsPerPel)
{
  bool audioIsCompressed = m_oStateManager.GetWfxSize() > 0;
  bool videoIsCompressed = m_oStateManager.GetFccHandler() != 0;  
  bool doAudio = ad >0, doVideo = vd>0;

  if( doVideo )
  {
    // VIDEO
    if( videoIsCompressed )
    {
      m_nBitmapInfoHdrSize = m_oVidComp.GetOutputFormatSize();
      m_pBitmapInfoHdr     = (BITMAPINFOHEADER *) new BYTE[m_nBitmapInfoHdrSize];
      memcpy( m_pBitmapInfoHdr, m_oVidComp.GetOutputFormat(), m_nBitmapInfoHdrSize );
    }
    else
    {
      // Generate uncompressed Bitmap Info structure
      BITMAPINFOHEADER bi;
    
      memset(&bi, 0, sizeof(BITMAPINFOHEADER));
      bi.biSize        = sizeof(BITMAPINFOHEADER);
      bi.biWidth       = vd->width;
      bi.biHeight      = vd->height;
      bi.biPlanes      = 1;
      bi.biBitCount    = nInBitsPerPel;
      bi.biCompression = dwInFccCode;
      bi.biSizeImage   = 0;

      m_nBitmapInfoHdrSize =  sizeof(BITMAPINFOHEADER);
      m_pBitmapInfoHdr     = (BITMAPINFOHEADER *) new BYTE[m_nBitmapInfoHdrSize];
      memcpy( m_pBitmapInfoHdr, &bi, sizeof BITMAPINFOHEADER );
    }
  }

  if( doAudio )
  {
    // AUDIO
    if( audioIsCompressed )
    {
      m_nWaveFormatExSize = m_oStateManager.GetWfxSize();
      m_pWaveFormatEx     = (WAVEFORMATEX *) new BYTE[m_nWaveFormatExSize];
      memcpy( m_pWaveFormatEx, m_oStateManager.GetWfx(), m_nWaveFormatExSize );
    }
    else
    {
      WAVEFORMATEX wfx;
      wfx.cbSize           = 0;
      wfx.nChannels        = ad->fmt.channels;
      wfx.nSamplesPerSec   = ad->fmt.frequency;
      wfx.wFormatTag       = WAVE_FORMAT_PCM;
      wfx.wBitsPerSample   = ad->fmt.depth;
      wfx.nBlockAlign      = wfx.nChannels * wfx.wBitsPerSample / 8;
      wfx.nAvgBytesPerSec  = wfx.nBlockAlign * wfx.nSamplesPerSec;

      m_nWaveFormatExSize = sizeof WAVEFORMATEX;
      m_pWaveFormatEx     = (WAVEFORMATEX *) new BYTE[m_nWaveFormatExSize];
      memcpy( m_pWaveFormatEx, &wfx, m_nWaveFormatExSize );
    }
  }
  return true;
}
bool COpenDMLAVIOutputApp::DeInitCompileFormats()
{
  if( m_pWaveFormatEx )
    delete []m_pWaveFormatEx;
  m_pWaveFormatEx = NULL;

  if( m_pBitmapInfoHdr )
    delete []m_pBitmapInfoHdr;
  m_pBitmapInfoHdr = NULL;
  
  return true;  
};               
                               
void COpenDMLAVIOutputApp::From32to24( PBYTE inImage, PBYTE outImage, long inSize )
{
 __asm
  {
      push	ebp
      push	edi
      push	esi
      push	edx
      push	ecx
      push	ebx
      push	eax
      
      mov	esi,inImage
      mov	edi,outImage

      mov ecx, inSize
      shr ecx, 4        ;our image has to be mod(4) pixels

main_loop:
      mov	eax,[esi]		  ;EAX = xxr0g0b0
      mov	ebx,[esi+4]		;EBX = xxr1g1b1
      mov	edx,ebx			  ;EDX = xxr1g1b1
      mov	ebp,[esi+8]		;EBP = xxr2g2b2
      shl	ebx,24			  ;EBX = b1000000
      and	eax,00ffffffh	;EAX = 00r0g0b0
      shr	edx,8			    ;EDX = 00xxr1g1
      or	eax,ebx			  ;EAX = b1r0g0b0
      mov	[edi+0],eax
      mov	ebx,ebp       ;EBX = xxr2g2b2
      shl	ebp,16			  ;EBP = g2b20000
      and	edx,0000ffffh ;EDX = 0000r1g1
      or	ebp,edx       ;EBP = g2b2r1g1
      mov	eax,[esi+12]  ;EAX = xxr3g3b3
      shr	ebx,16        ;EBX = 0000xxr2
      add	edi,12
      shl	eax,8         ;EAX = r3g3b300
      and	ebx,000000ffh ;EBX = 000000r2
      or	eax,ebx       ;EAX = r3g3b3r2
      mov	[edi+4-12],ebp
      add	esi,16
      mov	[edi+8-12],eax
      dec	ecx
      jge	main_loop
      
      pop	eax
      pop	ebx
      pop	ecx
      pop	edx
      pop	esi
      pop	edi
      pop	ebp
  }
}

flreturn_t COpenDMLAVIOutputApp::DoFirstPass( compile_s *cs, videodef_s *vd, DWORD dwInFcccode, long nBitsPerPixel )
{
  flreturn_t res;
  bool isKeyFrame;
  long frameSize;
  bool bSuccess = true;
  
  if( !vd )
    return flo_error;

  if( dwInFcccode == BI_RGB )
    return flo_error;

  if(!m_oVidComp.InitVideoCompressor( m_oStateManager.GetFccHandler(), m_oStateManager.GetHic(),
                                      vd->width, vd->height, 
                                      vd->framerate_num, vd->framerate_den,
                                      dwInFcccode, nBitsPerPixel,
                                      m_oStateManager.GetVideo()->GetSettings1st(),
                                      m_oStateManager.GetVideo()->GetSettingsSize() ))
  {
    MessageBox( NULL, "There was an error opening the video codec. \nMake sure the video codec selected supports the processing format", 
                "Error", MB_OK);
    bSuccess = false;
  }    


  getvideo_s        gv;
  getvideo_report_s gvr;

  gv.gs.cbsize = sizeof getvideo_s;
  gvr.gsr.cbsize = sizeof getvideo_report_s;

  if( bSuccess )
  {
    // Allocate buffer for rgb conversion
    PBYTE pUncompFrame = NULL;
    if( dwInFcccode == BI_RGB )
      pUncompFrame = new BYTE[vd->width*vd->height*4];

    while(  1  )
    {
      res = cs->getstream( cs->compileid, vd->strdef.streamid, (getstream_s *)&gv );

      if( res == flo_ok )
      {
        if( dwInFcccode == BI_RGB )
          // convert from 32 bits to 24 bits. Use the same buffer to output the stuff.
          From32to24( (PBYTE)gv.gs.buf, (PBYTE)m_pUncompFrame, vd->width*vd->height*4 );
        else
          m_pUncompFrame = (PBYTE)gv.gs.buf;        

         m_oVidComp.packFrame( m_pUncompFrame, &isKeyFrame, &frameSize );
      }
      else if( res==flo_endfirstpass || res==flo_stop ) {
        bSuccess = true;
        break;
      }
      else {
        bSuccess = false;
        break;
      }
    }

    delete []pUncompFrame;
  }
  m_oVidComp.DeInitVideoCompressor();

  return bSuccess ? res : flo_error;
}

flreturn_t COpenDMLAVIOutputApp::DoCompile( compile_s *cs, videodef_s *vd, audiodef_s *ad )
{

  bool bSuccess;
  DWORD dwInFccCode;
  int   nInBitsPerPel;
  bool doAudio = ad >0, doVideo = vd>0;
  bool videoIsCompressed, audioIsCompressed;
  flreturn_t res;

  DBGOUTFMT(( buf, "DoCompile: vd=%d ad=%d", vd, ad))

  // Determine input video format
  // Get input format if supported extensions
  if( doVideo  )
  {
    int nInFormat = vd->vidformat;
    switch( nInFormat )
    {
    case FLO_VIDEO_RGB32:
      dwInFccCode = BI_RGB;
      nInBitsPerPel = 24;
      break;
    case FLO_VIDEO_YUY2:
      dwInFccCode = '2YUY';
      nInBitsPerPel = 16;
      break;
    case FLO_VIDEO_YV12:
      dwInFccCode = '21VY';
      nInBitsPerPel = 12;
      break;
    default:
      return flo_invalidpar;
      break;
    }
    
    if( cs->istwopass ) 
    {
      res = DoFirstPass( cs, vd, dwInFccCode, nInBitsPerPel );
      switch( res )
      {
      case flo_endfirstpass:
          break;
        case flo_stop:
          return flo_stop;
        default:
          return flo_internalerror;
      }
    }

    videoIsCompressed = m_oStateManager.GetFccHandler() != 0;
    if( videoIsCompressed )
    {
      PBYTE settings = cs->istwopass ? m_oStateManager.GetVideo()->GetSettings2nd() :
                                       m_oStateManager.GetVideo()->GetSettings() ;
      long settingsize = m_oStateManager.GetVideo()->GetSettingsSize();

      if(!m_oVidComp.InitVideoCompressor( m_oStateManager.GetFccHandler(), m_oStateManager.GetHic(),
        vd->width, vd->height, 
        vd->framerate_num, vd->framerate_den,
        dwInFccCode, nInBitsPerPel,
        settings, settingsize ))
      {
        MessageBox( NULL, "There was an error opening the video codec. \nMake sure the video codec selected supports the processing format", 
                    "Error", MB_OK);
        return flo_internalerror;
      }    
    }
  }

  if( doAudio )
  { 
    audioIsCompressed = m_oStateManager.GetWfxSize() > 0;
    if( audioIsCompressed )
      if(!m_oAudComp.InitAudioCompressor( m_oStateManager.GetAcmDriverID(), 
                                          ad->fmt.frequency, 
                                          ad->fmt.channels,
                                          ad->fmt.depth, 
                                          m_oStateManager.GetWfx() ))
      {
        return flo_internalerror;
      }
  }

  InitCompileFormats( vd, ad, dwInFccCode, nInBitsPerPel );  


  // First of all, create the AVI writer object
  m_pAVIFile = new AVIOutputFile;

  // Initialize AVI output streams
  bSuccess = DoCompileInitAVIStreams( vd, ad ) == flo_ok;


  getvideo_s        gv;
  getvideo_report_s gvr;
  getaudio_s        ga;
  getaudio_report_s gar;

  getstream_streamreport_s gsr;

  gv.gs.cbsize = sizeof getvideo_s;
  gvr.gsr.cbsize = sizeof getvideo_report_s;
  ga.gs.cbsize = sizeof getaudio_s;
  gar.gsr.cbsize = sizeof getaudio_report_s;


  bool isKeyFrame;
  long  frameSize;
  BYTE *compFrame = NULL;

  unsigned char *uncompAudio;
  PBYTE compAudio ;
  unsigned long  compAudioSize, uncompAudioSize;
  unsigned long  nTotalCompAudioSize=0, nTotalAudioSamples=0;
  long  nChannels, nSampleRate, nSampleSize;
  unsigned long  nTotalFrames = 0;


  if(doAudio)
  {
    nChannels     = ad->fmt.channels;
    nSampleRate   = ad->fmt.frequency;
    nSampleSize   = ad->fmt.depth;
  }

  // Allocate buffer for rgb conversion
  m_pUncompFrame = NULL;
  if( dwInFccCode == BI_RGB )
    m_pUncompFrame = new BYTE[vd->width*vd->height*4];

  try
  {
    // Initialize AVI file
    m_pAVIFile->init(cs->filename, vd->width, vd->height, doVideo, doAudio, 100000, true);
    
    while(1)
    {
      if(doVideo)
      {
        if( cs->getstream( cs->compileid, vd->strdef.streamid, (getstream_s *)&gv ) != flo_ok )
          break;
        

        if( dwInFccCode == BI_RGB)
          // convert from 32 bits to 24 bits. Use the same buffer to output the stuff.
          From32to24( (PBYTE)gv.gs.buf, (PBYTE)m_pUncompFrame, vd->width*vd->height*4 );
        else
          m_pUncompFrame = (PBYTE)gv.gs.buf;        
        
        if( videoIsCompressed )
          compFrame = (BYTE *)m_oVidComp.packFrame( m_pUncompFrame, &isKeyFrame, &frameSize );
        else
        {
          compFrame = (BYTE *)m_pUncompFrame;
          frameSize = m_pBitmapInfoHdr->biBitCount/8 * 
                      m_pBitmapInfoHdr->biWidth      *
                      m_pBitmapInfoHdr->biHeight;
          isKeyFrame = true;
        }

        m_pAVIFile->videoOut->write( isKeyFrame ? AVIIF_KEYFRAME:0 , compFrame, frameSize, 1);

        gvr.bytes = frameSize;
        gvr.framenum = 0;
        gvr.recons = NULL;
        gvr.type = isKeyFrame;

        cs->getstream_report( cs->compileid, vd->strdef.streamid, (getstream_report_s *)&gvr );      
      }
      
      
      if(doAudio )
      { 
        unsigned long nTotalSamples;
        // interleave at this rate

        // calculate the number of total samples from the frame number
        if( vd )
          nTotalSamples= MulDiv( nTotalFrames, ad->fmt.frequency, vd->framerate_num )*vd->framerate_den;
        else
          nTotalSamples = ad->fmt.frequency;


        ga.samplecnt = nTotalSamples - nTotalAudioSamples;
        
        if( cs->getstream( cs->compileid, ad->strdef.streamid, (getstream_s *)&ga )!= flo_ok ) 
          break;

        uncompAudioSize = ga.gs.bufsize;
        uncompAudio = ga.gs.buf;
      
        nTotalAudioSamples += (uncompAudioSize / (nChannels*(nSampleSize>>3)));

        if( audioIsCompressed )
        {
          m_oAudComp.Compress( (PBYTE)uncompAudio, uncompAudioSize, compAudio, compAudioSize);
          nTotalCompAudioSize += compAudioSize;
        }
        else
        {
          compAudioSize = uncompAudioSize;
          compAudio     = (PBYTE)uncompAudio;
        }
        
        if( compAudioSize )
          // The number of samples is irrelevant when writing audio.
          // The info is self contained.
          m_pAVIFile->audioOut->write( 0, compAudio, compAudioSize, 0);
        
        gar.bytes = compAudioSize;
        cs->getstream_report( cs->compileid, ad->strdef.streamid, (getstream_report_s *)&gar );
      }

      // Report size written
      gsr.byteswritten = m_pAVIFile->get_fileposition();
      cs->getstream_report( cs->compileid, flo_streamreport, (getstream_report_s *)&gsr );

      nTotalFrames++;
    }// end while(1)
  }
  catch( MyError obError )
  {
    OutputDebugString(obError.gets());
    bSuccess = false;
  }


  // Free uncompressed buffer
  if( dwInFccCode == BI_RGB )
    delete []m_pUncompFrame;

  // Fix compressed audio info
  if(doAudio)
  {
    WAVEFORMATEX *pWfx = (WAVEFORMATEX *)m_pAVIFile->audioOut->getFormat();
    if(audioIsCompressed && pWfx->wFormatTag==WAVE_FORMAT_MPEGLAYER3)
    {
      pWfx->nAvgBytesPerSec = MulDiv( nTotalCompAudioSize, nSampleRate, nTotalAudioSamples );
      // Fix also, stream info
      m_pAVIFile->audioOut->streamInfo.dwRate = pWfx->nAvgBytesPerSec;

    }
  }

  // Try to finalize the file
  // Close AVI fil
  m_pAVIFile->finalize();
  
  DeInitCompileFormats();

  if(audioIsCompressed)
    m_oAudComp.DeInitAudioCompressor();

  if(videoIsCompressed)
    m_oVidComp.DeInitVideoCompressor();

  // delete writer object
  delete m_pAVIFile;

  return bSuccess ? flo_ok : flo_internalerror;
}

int COpenDMLAVIOutputApp::DoCompileInitAVIStreams(videodef_s *vd, audiodef_s *ad )
{
  bool bSuccess;
  bool doAudio = ad >0, doVideo = vd>0;
  bool videoIsCompressed = m_oStateManager.GetFccHandler() != 0;  
  // Create streams
  // Initialize output streams. This creates AVIOutputStreams objects. Audio and Video.
  bSuccess = m_pAVIFile->initOutputStreams() >0;
  if( bSuccess )
  {
    if(doVideo)
    {
      memset(&m_pAVIFile->videoOut->streamInfo, 0, sizeof(AVIStreamHeader_fixed));
      // Set information in AVI Video Output stream
      m_pAVIFile->videoOut->streamInfo.fccType    = streamtypeVIDEO;
      m_pAVIFile->videoOut->streamInfo.fccHandler = m_pBitmapInfoHdr->biCompression;
      m_pAVIFile->videoOut->streamInfo.dwFlags    = 0;
      m_pAVIFile->videoOut->streamInfo.dwScale    = vd->framerate_den;
      m_pAVIFile->videoOut->streamInfo.dwRate     = vd->framerate_num;
      m_pAVIFile->videoOut->streamInfo.dwStart    = 0;
      m_pAVIFile->videoOut->streamInfo.dwLength   = 1;
      m_pAVIFile->videoOut->streamInfo.dwInitialFrames = 0;
      m_pAVIFile->videoOut->streamInfo.dwSuggestedBufferSize = vd->width*vd->height*3;
      m_pAVIFile->videoOut->streamInfo.dwQuality  = -1;
      m_pAVIFile->videoOut->streamInfo.dwSampleSize = 0;
      m_pAVIFile->videoOut->streamInfo.rcFrame.left = 0;
      m_pAVIFile->videoOut->streamInfo.rcFrame.top = 0;
      m_pAVIFile->videoOut->streamInfo.rcFrame.right = vd->width;
      m_pAVIFile->videoOut->streamInfo.rcFrame.bottom = vd->height;
      
      m_pAVIFile->videoOut->setCompressed( videoIsCompressed );

      // Now set the format
      m_pAVIFile->videoOut->allocFormat( m_nBitmapInfoHdrSize );
      memcpy( m_pAVIFile->videoOut->getImageFormat(), 
              m_pBitmapInfoHdr,
              m_nBitmapInfoHdrSize );

    }
    if( doAudio )
    {
      memset(&m_pAVIFile->audioOut->streamInfo, 0, sizeof(AVIStreamHeader_fixed));
      // Set information in AVI Video Output stream
      m_pAVIFile->audioOut->streamInfo.fccType    = streamtypeAUDIO;
      m_pAVIFile->audioOut->streamInfo.fccHandler = 0;
      m_pAVIFile->audioOut->streamInfo.dwFlags    = 0;
      m_pAVIFile->audioOut->streamInfo.dwScale    = m_pWaveFormatEx->nBlockAlign;
      m_pAVIFile->audioOut->streamInfo.dwRate     = m_pWaveFormatEx->nAvgBytesPerSec;
      m_pAVIFile->audioOut->streamInfo.dwStart    = 0;
      m_pAVIFile->audioOut->streamInfo.dwLength   = 1;
      m_pAVIFile->audioOut->streamInfo.dwInitialFrames = 0;
      m_pAVIFile->audioOut->streamInfo.dwSuggestedBufferSize = 0;
      m_pAVIFile->audioOut->streamInfo.dwQuality  = 0;
      m_pAVIFile->audioOut->streamInfo.dwSampleSize = m_pWaveFormatEx->nBlockAlign;
      m_pAVIFile->audioOut->streamInfo.rcFrame.left = 0;
      m_pAVIFile->audioOut->streamInfo.rcFrame.top = 0;
      m_pAVIFile->audioOut->streamInfo.rcFrame.right = 0;
      m_pAVIFile->audioOut->streamInfo.rcFrame.bottom = 0;

      // Now set the format
      m_pAVIFile->audioOut->allocFormat( m_nWaveFormatExSize );
      memcpy( m_pAVIFile->audioOut->getWaveFormat(), 
              m_pWaveFormatEx,
              m_nWaveFormatExSize );
    }
  }
  return flo_ok;
}
