/* 
 *  AudioCompressor.cpp
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "stdafx.h"
#include "opendmlavioutput.h"
#include "AudioCompressor.h"
#include "debug.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


// Enable this to output debug messages
#if 1
  #if _DEBUG
    #define TRACK_ENUMERATION
  #endif
#endif

BOOL  CALLBACK   acmDriverEnumCallback( HACMDRIVERID hadid, DWORD dwInstance,   DWORD fdwSupport );
BOOL  CALLBACK   acmFormatEnumCallback( HACMDRIVERID hadid, 
                                            LPACMFORMATDETAILS pafd, 
                                            DWORD dwInstance,
                                            DWORD fdwSupport);
BOOL CALLBACK acmFormatTagEnumCallback(
                                          HACMDRIVERID hadid,           
                                          LPACMFORMATTAGDETAILS paftd,  
                                          DWORD dwInstance,             
                                          DWORD fdwSupport );

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CAudioCompressor::CAudioCompressor()
{
  pInAudio  = new BYTE[AUDIO_BUFFER_SIZE];
  pOutAudio = new BYTE[AUDIO_BUFFER_SIZE];
}

CAudioCompressor::~CAudioCompressor()
{
  if(pInAudio)
    delete []pInAudio;
  if(pOutAudio)
    delete []pOutAudio;

}

bool CAudioCompressor::FormatIsInstalled( HACMDRIVERID hadid, WAVEFORMATEX *wfx, int cbSize, int *dwFormatDriverIdx, int *FormatIdx )
{
  unsigned int wFormatTag, wBitsPerSample, nSamplesPerSec, nChannels, nAvgBytesPerSec;

  // Go through all the formats to see 
  if( wfx && cbSize)
  {
    for(int i=0; i<m_vAudioCodecList.GetSize(); i++)
    {
      if( m_vAudioCodecList[i].hadid != hadid )
        continue;

      for( int j=0; j<m_vAudioCodecList[i].m_vWaveFormatExList.GetSize(); j++)
      {
        CWaveFormatEx wfxTemp;
        wfxTemp = m_vAudioCodecList[i].m_vWaveFormatExList[j];
        if( wfxTemp.GetSize() == cbSize )
        {
          // if the stored wfx have the basic stuff
          wFormatTag      = wfxTemp.Get()->wFormatTag;
          wBitsPerSample  = wfxTemp.Get()->wBitsPerSample;
          nSamplesPerSec  = wfxTemp.Get()->nSamplesPerSec;
          nChannels       = wfxTemp.Get()->nChannels;
          nAvgBytesPerSec = wfxTemp.Get()->nAvgBytesPerSec;

          if( wFormatTag       == wfx->wFormatTag     &&
              wBitsPerSample   == wfx->wBitsPerSample &&
              nSamplesPerSec   == wfx->nSamplesPerSec &&
              nChannels        == wfx->nChannels      &&
              nAvgBytesPerSec  == wfx->nAvgBytesPerSec   )
          {
            *dwFormatDriverIdx = i;
            *FormatIdx = j;
            return true;
            // we found it
          }
        }
      }
    }
  }

    return false;
}
bool CAudioCompressor::EnumerateAudioCodecs(int sampleFreq, int channels, int bitsPerSample, bool bDeleteOld )
{

  m_nSampleFreq     = sampleFreq;
  m_nChannels       = channels;
  m_nBitsPerSample  = bitsPerSample;
  
  // Enumerate audio codecs
  if( bDeleteOld )
    m_vAudioCodecList.RemoveAll();

  acmDriverEnum( acmDriverEnumCallback, (DWORD)this, ACM_DRIVERENUMF_NOLOCAL );
  return true;
}

BOOL CALLBACK acmDriverEnumCallback( HACMDRIVERID hadid, DWORD dwInstance, DWORD fdwSupport )
{
  CAudioCompressor *AudioCompressor = (CAudioCompressor *)dwInstance;
  HACMDRIVER had=NULL;
  static int kk = 0;
  ACMFORMATTAGDETAILS aftd;
  ACMDRIVERDETAILS add;


  memset(&add, 0, sizeof ACMDRIVERDETAILS);
  add.cbStruct = sizeof( ACMDRIVERDETAILS );
  acmDriverDetails( hadid, &add, 0 );

#ifdef TRACK_ENUMERATION

  DBGOUTFMT((buf, "DRIVER: %s ", add.szLongName))
    
#endif 

  if( AudioCompressor )
  { 
    // Open the driver
    if(!acmDriverOpen( &had, hadid, 0 ))
    {
#ifdef TRACK_ENUMERATION
      DBGOUT("OPENED\n")
#endif


      // Enum the FormatTags for this driver
      AudioCompressor->m_hHad = had;
      AudioCompressor->m_hHadId = hadid;

      strcpy( AudioCompressor->szDriverName, add.szLongName );

      memset(&aftd, 0, sizeof aftd);
      aftd.cbStruct = sizeof( ACMFORMATTAGDETAILS );
      acmFormatTagEnum( had, &aftd, acmFormatTagEnumCallback, (DWORD)AudioCompressor, 0 );

      if(acmDriverClose( had, 0 ))
        DBGOUT("acmDriverEnumCallback - Couldnt close driver\n")

    }
    else
    {
#ifdef TRACK_ENUMERATION
      DBGOUT("FAILED\n")
#endif         
      OutputDebugString(" acmDriverEnumCallback - Error opening ACM driver ");
    } 
  }


  return TRUE;
}

BOOL CALLBACK acmFormatTagEnumCallback(   HACMDRIVERID hadid,           
                                       LPACMFORMATTAGDETAILS paftd,  
                                       DWORD dwInstance,             
                                       DWORD fdwSupport )
{
  CAudioCompressor *AudioCompressor = (CAudioCompressor *)dwInstance;
  ACMFORMATDETAILS afd;
  WAVEFORMATEX *wfx=NULL;
  // Get maximum WAVEFORMATEX size in the system
  DWORD size;
  acmMetrics( NULL, ACM_METRIC_MAX_SIZE_FORMAT, &size );
  
  
  wfx = (WAVEFORMATEX *)new char[size];
  memset(wfx, 0, sizeof WAVEFORMATEX);

  wfx->cbSize           = 0;
  wfx->nChannels        = AudioCompressor->m_nChannels;
  wfx->nSamplesPerSec   = AudioCompressor->m_nSampleFreq;
  wfx->wFormatTag       = (WORD)paftd->dwFormatTag;
  wfx->wBitsPerSample   = AudioCompressor->m_nBitsPerSample;
  wfx->nBlockAlign      = wfx->nChannels * wfx->wBitsPerSample / 8;
  wfx->nAvgBytesPerSec  = wfx->nBlockAlign * wfx->nSamplesPerSec;
  
  memset(&afd, 0, sizeof afd);
  afd.pwfx          = wfx;
  afd.dwFormatIndex = 0;
  afd.cbStruct      = sizeof( ACMFORMATDETAILS );
  afd.cbwfx         = size;
  afd.dwFormatTag   = paftd->dwFormatTag;
  afd.fdwSupport    = 0;

  // Enumerate tags
  acmFormatEnum( AudioCompressor->m_hHad, 
    &afd, 
    acmFormatEnumCallback, 
    (DWORD)AudioCompressor, 
    ACM_FORMATENUMF_WFORMATTAG  );
  
  delete [](char *)wfx;
  return TRUE;
}

BOOL CALLBACK acmFormatEnumCallback( HACMDRIVERID hadid, 
                                    LPACMFORMATDETAILS pafd, 
                                    DWORD dwInstance,
                                    DWORD fdwSupport)
{
  CAudioCompressor *ac = (CAudioCompressor *)dwInstance;
  
  WAVEFORMATEX     wfxSrc;
  HACMSTREAM       has;
  
  CAudioCodecInfo cnfo; 
  
  if( fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC )
  { 
    // Try to open a stream with our PCM input.
    // Our input
    wfxSrc.cbSize           = 0;
    wfxSrc.nChannels        = ac->m_nChannels;
    wfxSrc.nSamplesPerSec   = ac->m_nSampleFreq;
    wfxSrc.wFormatTag       = WAVE_FORMAT_PCM;
    wfxSrc.wBitsPerSample   = ac->m_nBitsPerSample;
    wfxSrc.nBlockAlign      = wfxSrc.nChannels * wfxSrc.wBitsPerSample / 8;
    wfxSrc.nAvgBytesPerSec  = wfxSrc.nBlockAlign * wfxSrc.nSamplesPerSec;
    
    // Don't add formats with different sampling rates. Add 44100 and 48000
    if( ac->m_nSampleFreq != pafd->pwfx->nSamplesPerSec &&
        ac->m_nSampleFreq != 44100 &&
        ac->m_nSampleFreq != 48000 )
      return TRUE;

    if(!acmStreamOpen( &has, 
      ac->m_hHad, 
      &wfxSrc, 
      pafd->pwfx, 
      NULL, 0, 0, 
      ACM_STREAMOPENF_QUERY ))
    {
      int idx = 0;
      // Yehaa! the conversion is supported.
      // Look for this driver
      bool AddNewDriver = true;

      for( int i=0; i<ac->m_vAudioCodecList.GetSize(); i++ )
        if( ac->m_vAudioCodecList[i].hadid == hadid ) {
          AddNewDriver = false;
          break;
        }

      if( AddNewDriver == false ) {
        idx = i;
      }
      else { //AddnewDriver == true
        cnfo.hadid       = hadid;
        strcpy( cnfo.szDriverName, ac->szDriverName );
        cnfo.m_vFormatDetailsList.RemoveAll();
        // Finally Add it
        ac->m_vAudioCodecList.InsertAt( 0, cnfo );
        idx = 0;
      }
      
      // Finally copy this stream format to this tag list
      
      // Copy WaveFormatEx structure
      CWaveFormatEx wfx;
      // Set size of WaveFormatEx
      wfx.SetSize( pafd->cbwfx );
      // Copy the actual structure
      memcpy( wfx.GetData(), pafd->pwfx, pafd->cbwfx );
      // We are storing, the WAVEFORMATEX for this format
      // and the ACMFORMATDETAILS structure for this format.
      ac->m_vAudioCodecList[idx].m_vWaveFormatExList.InsertAt( 0, wfx );
      ac->m_vAudioCodecList[idx].m_vFormatDetailsList.InsertAt( 0, *pafd );
      // Get FormatTag name
      ACMFORMATTAGDETAILS sAftd;
      memset( &sAftd, 0, sizeof ACMFORMATTAGDETAILS );
      sAftd.cbStruct = sizeof ACMFORMATTAGDETAILS;
      sAftd.dwFormatTag = pafd->dwFormatTag;

      acmFormatTagDetails( ac->m_hHad, &sAftd, ACM_FORMATTAGDETAILSF_FORMATTAG );

      ac->m_vAudioCodecList[idx].m_vFormatTagStrings.InsertAt( 0, CString(sAftd.szFormatTag) );
    }
  }
  return TRUE;
}

bool CAudioCompressor::DeInitAudioCompressor()
{
  acmStreamUnprepareHeader( m_hHas, &m_sStreamHdr, 0);
  acmStreamClose( m_hHas, 0 );
  acmDriverClose( m_hHad, 0 );
  return true;
}

bool CAudioCompressor::InitAudioCompressor(HACMDRIVERID hadid, int sampleFreq, int channels, int bitsPerSample, WAVEFORMATEX *wfxOut)
{
  int ret;
  if(ret = acmDriverOpen( &m_hHad, hadid, 0 ))
    return false;
  // Prepare input WAVEFORMATEX
  WAVEFORMATEX *wfx=NULL;
  // Get maximum WAVEFORMATEX size in the system
  DWORD size;
  acmMetrics( NULL, ACM_METRIC_MAX_SIZE_FORMAT, &size );
  
  
  WAVEFORMATEX inWfx;
  inWfx.cbSize           = 0;
  inWfx.nChannels        = channels;
  inWfx.nSamplesPerSec   = sampleFreq;
  inWfx.wFormatTag       = WAVE_FORMAT_PCM;
  inWfx.wBitsPerSample   = bitsPerSample;
  inWfx.nBlockAlign      = inWfx.nChannels * inWfx.wBitsPerSample / 8;
  inWfx.nAvgBytesPerSec  = inWfx.nBlockAlign * inWfx.nSamplesPerSec;

  // Try to create open the stream
  if( acmStreamOpen( &m_hHas, m_hHad, &inWfx, wfxOut, NULL, NULL, 0, ACM_STREAMOPENF_NONREALTIME ) )
    return false;

  memset( &m_sStreamHdr, 0, sizeof ACMSTREAMHEADER );
  m_sStreamHdr.cbStruct = sizeof ACMSTREAMHEADER;
  m_sStreamHdr.pbSrc       = pInAudio;
  m_sStreamHdr.cbSrcLength = AUDIO_BUFFER_SIZE;
  m_sStreamHdr.pbDst       = pOutAudio;
  m_sStreamHdr.cbDstLength = AUDIO_BUFFER_SIZE;

  if( acmStreamPrepareHeader( m_hHas, &m_sStreamHdr, 0) )
  {
    acmStreamClose( m_hHas, 0 );
    return false;
  }

  m_nBufferPtr = 0;
  return true;
}

bool CAudioCompressor::Compress(PBYTE inAudio, long inSize, PBYTE &outAudio, unsigned long  &outSize)
{
  outAudio = NULL;
  outSize  = 0;

  // copy source data to our stream header
  if( (m_nBufferPtr + inSize) > MIN_AUDIO_BUFFER )
  {
    // check that inSize is not too much
    if( (m_nBufferPtr+inSize)> AUDIO_BUFFER_SIZE )
      return false;
    // we have enough data to be compressed
    // copy the data
    memcpy( m_sStreamHdr.pbSrc+m_nBufferPtr, inAudio, inSize );
    m_nBufferPtr += inSize;
  }
  else
  {
    // copy data and exit
    memcpy( m_sStreamHdr.pbSrc+m_nBufferPtr, inAudio, inSize );
    m_nBufferPtr += inSize;
    return false;
  }

  m_sStreamHdr.cbSrcLength     = m_nBufferPtr;
  m_sStreamHdr.cbSrcLengthUsed = 0;
  m_sStreamHdr.cbDstLengthUsed = 0;
  // try to compress this buffer
  if( acmStreamConvert( m_hHas, &m_sStreamHdr, ACM_STREAMCONVERTF_BLOCKALIGN  ) )
    return false;
  if( !(m_sStreamHdr.fdwStatus & ACMSTREAMHEADER_STATUSF_DONE) )
    return false;

  // conversion was successful
  // copy remaining data after conversion in source buffer
  int newBufSize = 0;
  for( int j=0,i=m_sStreamHdr.cbSrcLengthUsed; i<m_nBufferPtr; i++,j++)
  {
    pInAudio[j] = pInAudio[i];
    newBufSize++;
  }
  m_nBufferPtr = newBufSize;

  // output the compressed data
  outAudio = m_sStreamHdr.pbDst;
  outSize  = m_sStreamHdr.cbDstLengthUsed;
  return true;
}