/* 
 *  options.cpp    -  Controls FlasKMPEG options
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

 
#include "flaskmpeg.h"
#include "resource.h"
#include "auxiliary.h"
#include <commctrl.h>
#include "error.h"
#include ".\Resizer\filter.h"
#include ".\Resizer\resizer.h"
#include ".\postprocessing.h"
#include ".\Video\Videowrapper.h"
#include ".\misc\ieee1180testdlg.h"
#include ".\runstate.h"
#include ".\YUVtoRGB.h"
#include ".\misc\selectordialog.h"
 
extern TConfig          o;
extern TRunState        rs;
extern HINSTANCE        hInst;

struct TOptions{
	int         frames_compile;
};

#define GET_FRAMERATE(x) ((double)x.scale/(double)x.sampleSize)

struct Trunoptions{
	// Temporal data
	ui8*      DisplayImage;
	ui8*      DecodedImage;
	VBitmap   in, out;
	TPPost              pp;
	TOptions            opt;
    TProfile            current_profile;
} opt_run;

HFONT myFont;


#define C_PAGES 5
typedef struct tag_dlghdr { 
   HWND hwndTab;       // tab control 
   HWND hwndDisplay;   // current child dialog box 
   RECT rcDisplay;     // display rectangle for the tab control 
   HWND DlgTabs[C_PAGES];
   /* these function pointers check for valid data in the given
    * tab (corresponding to DlgTabs); on valid data, true is returned;
	* on invalid data, an error is displayed, focus set on the
	* proper control and false is returned. 
	*
    * If this is NULL, then there are no checks and a true return
    * is assumed.
    */
   bool (*DlgTabCheck[C_PAGES])(HWND hDlg);
} DLGHDR; 
HWND video,audio,processing,files,general;
DLGHDR	*pHdr;

LRESULT CALLBACK OptionsProfileNameDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam);

////////////////////////////////////////////////////////
//             OUTPUT    PAD
////////////////////////////////////////////////////////
LRESULT CALLBACK DisplayProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam){

HGDIOBJ     originalObject;
HFONT       myFont;
BITMAPINFO  dibInfo;
HDC         hDC;
char        szTemp[256];
	switch (message)
	{
		case WM_INITDIALOG:
				return TRUE;
		case WM_PAINT:

			//BeginPaint(hWnd, &lp);
			ZeroMemory(&dibInfo, sizeof(BITMAPINFO));
			opt_run.out.MakeBitmapHeader(&dibInfo.bmiHeader);
			hDC=GetDC(hWnd);
			SetDIBitsToDevice(hDC ,0,0,opt_run.out.w,opt_run.out.h,
			         0,0,0,opt_run.out.h, (ui8 *)opt_run.out.Address(),&dibInfo,DIB_RGB_COLORS);
			if (opt_run.pp.crop) {
				SelectObject(hDC, GetStockObject(NULL_BRUSH));
				SelectObject(hDC, GetStockObject(WHITE_PEN));
				Rectangle(hDC, opt_run.pp.cropLeftOffset, opt_run.pp.cropTopOffset, 
						opt_run.pp.cropLeftOffset+opt_run.pp.cropWidth-1, 
						opt_run.pp.cropTopOffset+opt_run.pp.cropHeight-1);
			}
			// Paint image stats
			// Create Font
		    myFont = (HFONT)GetStockObject(ANSI_VAR_FONT); 
		    if (originalObject = SelectObject(hDC, myFont)) 
			{ 
				sprintf(szTemp, "Resize %d x %d -> %d x %d", opt_run.in.w, opt_run.in.h,
						opt_run.pp.resWidth, opt_run.pp.resHeight);
				//opt_run.out.w, opt_run.out.h);
		        TextOut(hDC, 3, 3, szTemp, strlen(szTemp));
				sprintf(szTemp, "AR input %.03f output %.03f", 1.0/opt_run.pp.iDAR,
						(double)opt_run.pp.resWidth / (double)opt_run.pp.resHeight);
				TextOut(hDC, 3, 20, szTemp, strlen(szTemp));
				if(opt_run.pp.crop){
					sprintf(szTemp, "Crop left/right %d/%d, top/bottom %d/%d -> %d x %d",
							opt_run.pp.cropLeftOffset, 
							opt_run.out.w-opt_run.pp.cropWidth-opt_run.pp.cropLeftOffset,
							opt_run.pp.cropTopOffset,
							opt_run.out.h-opt_run.pp.cropHeight-opt_run.pp.cropTopOffset,
							opt_run.pp.cropWidth, opt_run.pp.cropHeight);
				    TextOut(hDC, 3, 37, szTemp, strlen(szTemp)); 
					/*					
					sprintf(szTemp, "%s: %d", GS(OPT_OUTPAD_TOPOFFSET), opt_run.pp.cropTopOffset);
				    TextOut(hDC, 80, 3, szTemp, strlen(szTemp)); 
					sprintf(szTemp, "%s: %d", GS(OPT_OUTPAD_LEFTOFFSET), opt_run.pp.cropLeftOffset);
				    TextOut(hDC, 80, 20, szTemp, strlen(szTemp)); 
					*/

				}

				SelectObject(hDC, originalObject); 

		    } 
 


			ReleaseDC(hWnd, hDC);


			//EndPaint(hWnd, &lp);
			break;
		case WM_COMMAND:

			break;
	}
    return FALSE;
}


// Snap second window to the first
void Snap(HWND win1, HWND win2){
	RECT   rc;
		// Get rect win1
		GetWindowRect(win1, &rc);
		// Set position of second window
		WindowMove(win2, rc.left, rc.bottom );


}

// UpdateVBitmaps: update size from run state post processing options
//                 and from running input video stream
void UpdateVBitmaps(){
	if(opt_run.pp.crop)
		opt_run.out.init(opt_run.DisplayImage  , opt_run.pp.cropWidth, opt_run.pp.cropHeight    , 32);
	else
		opt_run.out.init(opt_run.DisplayImage  , opt_run.pp.resWidth , opt_run.pp.resHeight     , 32);
    opt_run.in.init(opt_run.DecodedImage  , rs.video->pictureWidth, rs.video->pictureHeight, 32);
}
// UpdateDisplay changes the size of the display window and writes the new image there
void UpdateDisplay(HWND hWnd){
	bool crop = opt_run.pp.crop > 0;
	opt_run.pp.crop = false;

	UpdateVBitmaps();
	// Resize client area
	WindowClientResize(hWnd, opt_run.out.w, opt_run.out.h);
	

	
	// PostProcess image
	PostProcessingStart(&opt_run.in , &opt_run.out, &opt_run.pp);
	PostProcess        (&opt_run.in , &opt_run.out, &opt_run.pp);
	PostProcessingStop (&opt_run.pp);

	opt_run.pp.crop = crop;

	// Force image update
	InvalidateRect(hWnd, NULL, false);


}


void EnableCropBoxes(HWND hDlg){
	DlgEnable(hDlg, IDC_CROP_TOPOFFSET);
	DlgEnable(hDlg, IDC_CROP_HEIGHT);
	DlgEnable(hDlg, IDC_CROP_LEFTOFFSET);
	DlgEnable(hDlg, IDC_CROP_WIDTH);
}

void DisableCropBoxes(HWND hDlg){
	DlgDisable(hDlg, IDC_CROP_TOPOFFSET);
	DlgDisable(hDlg, IDC_CROP_HEIGHT);
	DlgDisable(hDlg, IDC_CROP_LEFTOFFSET);
	DlgDisable(hDlg, IDC_CROP_WIDTH);
}
void EnableLetterboxBoxes(HWND hDlg){
					DlgEnable(hDlg, IDC_LETTERBOX_TOP);
					DlgEnable(hDlg, IDC_LETTERBOX_BOTTOM);
					DlgEnable(hDlg, IDC_LETTERBOX_RIGHT);
					DlgEnable(hDlg, IDC_LETTERBOX_LEFT);
}

void DisableLetterboxBoxes(HWND hDlg){
					DlgDisable(hDlg, IDC_LETTERBOX_TOP);
					DlgDisable(hDlg, IDC_LETTERBOX_BOTTOM);
					DlgDisable(hDlg, IDC_LETTERBOX_RIGHT);
					DlgDisable(hDlg, IDC_LETTERBOX_LEFT);
}
void SetCropBoxesText(HWND hDlg){
				DlgSetText(hDlg, IDC_CROP_TOPOFFSET, opt_run.pp.cropTopOffset);
				DlgSetText(hDlg, IDC_CROP_HEIGHT,    opt_run.pp.cropHeight);
				DlgSetText(hDlg, IDC_CROP_LEFTOFFSET,opt_run.pp.cropLeftOffset);
				DlgSetText(hDlg, IDC_CROP_WIDTH,     opt_run.pp.cropWidth);
}
void SetLetterboxBoxesText(HWND hDlg){
				DlgSetText(hDlg, IDC_LETTERBOX_TOP,       opt_run.pp.letterboxTop);
				DlgSetText(hDlg, IDC_LETTERBOX_BOTTOM,    opt_run.pp.letterboxBottom);
				DlgSetText(hDlg, IDC_LETTERBOX_RIGHT,     opt_run.pp.letterboxRight);
				DlgSetText(hDlg, IDC_LETTERBOX_LEFT,      opt_run.pp.letterboxLeft);
}
void GetCropBoxesText(HWND hDlg){
				opt_run.pp.cropTopOffset = DlgGetText(hDlg, IDC_CROP_TOPOFFSET);
				opt_run.pp.cropHeight    = DlgGetText(hDlg, IDC_CROP_HEIGHT);
				opt_run.pp.cropLeftOffset= DlgGetText(hDlg, IDC_CROP_LEFTOFFSET);
				opt_run.pp.cropWidth     = DlgGetText(hDlg, IDC_CROP_WIDTH);
}
void GetLetterboxBoxesText(HWND hDlg){
				opt_run.pp.letterboxTop    = DlgGetText(hDlg, IDC_LETTERBOX_TOP);
				opt_run.pp.letterboxBottom = DlgGetText(hDlg, IDC_LETTERBOX_BOTTOM);
				opt_run.pp.letterboxRight  = DlgGetText(hDlg, IDC_LETTERBOX_RIGHT);
				opt_run.pp.letterboxLeft   = DlgGetText(hDlg, IDC_LETTERBOX_LEFT);
}

void UpdateCropAndLetterboxBoxes(HWND hDlg){
	if(opt_run.pp.crop){
		DlgUnCheck(hDlg, IDC_NOCROP);
		EnableCropBoxes(hDlg);
	}
	else{
		DlgCheck(hDlg, IDC_NOCROP);
		DisableCropBoxes(hDlg);
	}
	if(opt_run.pp.letterbox){
		DlgUnCheck(hDlg, IDC_NOLETTERBOX);
		EnableLetterboxBoxes(hDlg);
	}
	else{
		DlgCheck(hDlg, IDC_NOLETTERBOX);
		DisableLetterboxBoxes(hDlg);
	}
		
	SetCropBoxesText(hDlg);
	SetLetterboxBoxesText(hDlg);
}

// EnableCrop : Enable crop buttons
void EnableCrop(HWND hDlg){
	DlgEnable(hDlg, CROP_TOPOFFSET_PLUS);
	DlgEnable(hDlg, CROP_TOPOFFSET_MINUS);
	DlgEnable(hDlg, CROP_HEIGHT_PLUS);
	DlgEnable(hDlg, CROP_HEIGHT_MINUS);
	DlgEnable(hDlg, CROP_LEFTOFFSET_PLUS);
	DlgEnable(hDlg, CROP_LEFTOFFSET_MINUS);
	DlgEnable(hDlg, CROP_WIDTH_PLUS);
	DlgEnable(hDlg, CROP_WIDTH_MINUS);
	DlgEnable(hDlg, WIDTH_MINUS1);
	DlgEnable(hDlg, WIDTH_PLUS1);
	DlgEnable(hDlg, HEIGHT_MINUS1);
	DlgEnable(hDlg, HEIGHT_PLUS1);
}
// DisableCrop : Disable crop buttons
void DisableCrop(HWND hDlg){
	DlgDisable(hDlg, CROP_TOPOFFSET_PLUS);
	DlgDisable(hDlg, CROP_TOPOFFSET_MINUS);
	DlgDisable(hDlg, CROP_HEIGHT_PLUS);
	DlgDisable(hDlg, CROP_HEIGHT_MINUS);
	DlgDisable(hDlg, CROP_LEFTOFFSET_PLUS);
	DlgDisable(hDlg, CROP_LEFTOFFSET_MINUS);
	DlgDisable(hDlg, CROP_WIDTH_PLUS);
	DlgDisable(hDlg, CROP_WIDTH_MINUS);
	DlgDisable(hDlg, WIDTH_MINUS1);
	DlgDisable(hDlg, WIDTH_PLUS1);
	DlgDisable(hDlg, HEIGHT_MINUS1);
	DlgDisable(hDlg, HEIGHT_PLUS1);
}
void UpdatePadButtons(HWND hDlg){
	if(opt_run.pp.crop){
		DlgCheck(hDlg, IDC_CROP);
		EnableCrop(hDlg);
	}
	else{
		DlgUnCheck(hDlg, IDC_CROP);
		DisableCrop(hDlg);
	}
	if(opt_run.pp.letterbox){
		DlgCheck(hDlg, IDC_LETTERBOX);
	}
	else{
		DlgUnCheck(hDlg, IDC_LETTERBOX);
	}

}
// Val: Validate post processing options and update display
int Val(HWND h_display){
	// Set miscelaneous parameters

	if(CheckVideoParameters(&opt_run.pp)){
		UpdateDisplay(h_display);
		return 1;
	}
	return 0;
}
//PAD
LRESULT CALLBACK DlgPad(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	presInfo PresInfo;
	TVideoOptions video_opt;
	static HWND h_display;
	switch (message)
	{
		case WM_INITDIALOG:
			    // Initialize strings
				SetText(hDlg, GS(OUTPUT_PAD_TITLE));
				// Crop
				DlgSetText(hDlg, IDC_CROP, GS(OUTPUT_PAD_CROP) );
				DlgSetText(hDlg, OPT_OUTPAD_TOPOFFSET, GS(OPT_OUTPAD_TOPOFFSET));
				DlgSetText(hDlg, OPT_OUTPAD_CROPHEIGHT, GS(OPT_OUTPAD_CROPHEIGHT));
				DlgSetText(hDlg, OPT_OUTPAD_LEFTOFFSET, GS(OPT_OUTPAD_LEFTOFFSET));
				DlgSetText(hDlg, OPT_OUTPAD_CROPWIDTH, GS(OPT_OUTPAD_CROPWIDTH));
				// Output Size
				DlgSetText(hDlg, R_OUTPUT_PAD_OUTPUTSIZE, GS(OUTPUT_PAD_OUTPUTSIZE));
				DlgSetText(hDlg, OPT_OUTPAD_WIDTH, GS(OPT_OUTPAD_WIDTH));
				DlgSetText(hDlg, OPT_OUTPAD_HEIGHT, GS(OPT_OUTPAD_HEIGHT));
				DlgSetText(hDlg, IDOK , GS(OUTPUT_PAD_HIDE));

				DlgSetText(hDlg, IDC_LETTERBOX, GS(OUTPUT_PAD_LETTERBOX));
				DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_TOP, GS(OPT_OUTPAD_LETTERBOX_TOP));
				DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_BOTTOM, GS(OPT_OUTPAD_LETTERBOX_BOTTOM));
				DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_LEFT, GS(OPT_OUTPAD_LETTERBOX_LEFT));
				DlgSetText(hDlg, OPT_OUTPAD_LETTERBOX_RIGHT, GS(OPT_OUTPAD_LETTERBOX_RIGHT));
				//Reset
				DlgSetText(hDlg, IDC_RESET, GS(OUTPUT_PAD_RESET));

				if(rs.prof.keepAspect) 
					DlgCheck(hDlg, ASPECT_CHECK);
				else 
					DlgUnCheck(hDlg, ASPECT_CHECK);

				SendDlgItemMessage(hDlg, SLIDER, TBM_SETRANGE, (WPARAM) TRUE,(LPARAM) MAKELONG(0, 1000)); 
				SendDlgItemMessage(hDlg, SLIDER, TBM_SETPOS, (WPARAM) TRUE, 500);   
   
			    // Allocate buffer memory
				if(!(opt_run.DisplayImage = (ui8 *)malloc(MAX_IMAGE_MEM)))
					return false;
				if(!(opt_run.DecodedImage = (ui8 *)malloc(MAX_IMAGE_MEM)))
					return false;

				//Set crop checkbox
				UpdatePadButtons(hDlg);

				UpdateVBitmaps();

			    // Create display window
			    h_display = CreateNormalWindow( DisplayProc, hInst, hDlg);
				WindowHide( h_display );

				YUVImageTag *yuv;
				// Check for video object and get a frame
				if(rs.video){
					video_opt.idctIndex = rs.prof.idctIndex;
					video_opt.recons_progressive = rs.prof.recons_progressive;
          video_opt.bStartInSync         = false;
					//Seek the video to the middle
					rs.video->SetStreamPos( rs.video->GetStreamSize()>>1 );
					rs.video->Start(&video_opt);
					rs.video->GetFrame(&PresInfo, &yuv);
					rs.video->Stop();
					//Yuv Convert it
					YUVtoRGB(yuv, (unsigned char *)opt_run.in.data);	
					// update iDAR
					opt_run.pp.iDAR = rs.video->DAR;
					UpdateDisplay(h_display);

				}
				else 
					return FALSE;

				WindowShow( h_display );
				Snap(hDlg, h_display);

				return TRUE;
		case WM_MOVE:
				Snap(hDlg, h_display);
			break;
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				DestroyWindow(h_display);
				if(opt_run.DisplayImage){
					free(opt_run.DisplayImage);
					opt_run.DisplayImage=NULL;
				}
				if(opt_run.DecodedImage){
					free(opt_run.DecodedImage);
					opt_run.DecodedImage=NULL;
				}
				//Set video res
				DlgSetText(video, IDC_WIDTH, opt_run.pp.resWidth);
				DlgSetText(video, IDC_HEIGHT, opt_run.pp.resHeight);
				
				//Set crop and letterbox
				SetCropBoxesText(processing);
				SetLetterboxBoxesText(processing);
				
				UpdateCropAndLetterboxBoxes(processing);

				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}

			switch (LOWORD(wParam)){
				case IDC_RESET:
						opt_run.pp.crop            = 0;
						opt_run.pp.cropWidth      = rs.video->pictureWidth;
						opt_run.pp.cropHeight     = rs.video->pictureHeight;
						opt_run.pp.cropLeftOffset = 0;
						opt_run.pp.cropTopOffset  = 0;

						opt_run.pp.letterbox       = 0;
						opt_run.pp.letterboxBottom = 0;
						opt_run.pp.letterboxLeft   = 0;
						opt_run.pp.letterboxRight  = 0;
						opt_run.pp.letterboxTop    = 0;

						opt_run.pp.resWidth        = rs.video->pictureWidth;
						opt_run.pp.resHeight       = rs.video->pictureHeight;
						UpdateDisplay(h_display);
						UpdatePadButtons(hDlg);
					break;
				case IDC_CROP:
					opt_run.pp.crop=DlgIsChecked(hDlg, IDC_CROP);
					if(opt_run.pp.crop){
						opt_run.pp.cropWidth      = opt_run.pp.resWidth;
						opt_run.pp.cropHeight     = opt_run.pp.resHeight;
						opt_run.pp.cropLeftOffset = 0;
						opt_run.pp.cropTopOffset  = 0;
					}
					else {
						// if crop is turned off, we must set resWidth/Height to
						// multiples of 16
						opt_run.pp.resHeight = 16 * ( opt_run.pp.resHeight / 16 );
						opt_run.pp.resWidth = 16 * (opt_run.pp.resWidth / 16 );
					}
					UpdateDisplay(h_display);
					UpdatePadButtons(hDlg);
					break;
				
				case IDC_LETTERBOX:
					opt_run.pp.letterbox=DlgIsChecked(hDlg, IDC_LETTERBOX);
					if(opt_run.pp.letterboxBottom<=0 &&
				       opt_run.pp.letterboxLeft  <=0 && 
					   opt_run.pp.letterboxRight <=0 &&
					   opt_run.pp.letterboxTop   <=0){
						    opt_run.pp.letterboxBottom =0;
					        opt_run.pp.letterboxLeft   =0;
					        opt_run.pp.letterboxRight  =0;
					        opt_run.pp.letterboxTop    =0;
					}
					UpdateDisplay(h_display);
					UpdatePadButtons(hDlg);
					break;
				case ASPECT_CHECK:
					/* o.options.keepAspect = */
					opt_run.pp.doAR = DlgIsChecked(hDlg, ASPECT_CHECK);
					UpdateDisplay(h_display);
					break;

				case WIDTH_PLUS:
					opt_run.pp.resWidth+=16;
					if( !Val(h_display) ) opt_run.pp.resWidth-=16;
				break;
				case WIDTH_MINUS:
					opt_run.pp.resWidth-=16;
					if( !Val(h_display) ) opt_run.pp.resWidth+=16;
				break;
				case WIDTH_PLUS1:
					opt_run.pp.resWidth+=2;
					if( !Val(h_display) ) opt_run.pp.resWidth-=2;
				break;
				case WIDTH_MINUS1:
					opt_run.pp.resWidth-=2;
					if( !Val(h_display) ) opt_run.pp.resWidth+=2;
				break;
				case HEIGHT_PLUS:
					opt_run.pp.resHeight+=16;
					if( !Val(h_display) ) opt_run.pp.resHeight-=16;
				break;
				case HEIGHT_MINUS:
					opt_run.pp.resHeight-=16;
					if( !Val(h_display) ) opt_run.pp.resHeight+=16;
				break;
				case HEIGHT_PLUS1:
					opt_run.pp.resHeight+=1;
					if( !Val(h_display) ) opt_run.pp.resHeight-=1;
				break;
				case HEIGHT_MINUS1:
					opt_run.pp.resHeight-=1;
					if( !Val(h_display) ) opt_run.pp.resHeight+=1;
				break;
				case CROP_TOPOFFSET_PLUS:
					opt_run.pp.cropTopOffset++;
					if( !Val(h_display) ) opt_run.pp.cropTopOffset--;
				break;
				case CROP_TOPOFFSET_MINUS:
					opt_run.pp.cropTopOffset--;
					if( !Val(h_display) ) opt_run.pp.cropTopOffset++;
				break;
				case CROP_HEIGHT_PLUS:
					opt_run.pp.cropHeight+=16;
					if( !Val(h_display) ) opt_run.pp.cropHeight-=16;
				break;
				case CROP_HEIGHT_MINUS:
					opt_run.pp.cropHeight-=16;
					if( !Val(h_display) ) opt_run.pp.cropHeight+=16;
				break;
				case CROP_LEFTOFFSET_PLUS:
					opt_run.pp.cropLeftOffset++;
					if( !Val(h_display) ) opt_run.pp.cropLeftOffset--;
				break;
				case CROP_LEFTOFFSET_MINUS:
					opt_run.pp.cropLeftOffset--;
					if( !Val(h_display) ) opt_run.pp.cropLeftOffset++;
				break;
				case CROP_WIDTH_PLUS:
					opt_run.pp.cropWidth+=16;
					if( !Val(h_display) ) opt_run.pp.cropWidth-=16;
				break;
				case CROP_WIDTH_MINUS:
					opt_run.pp.cropWidth-=16;
					if( !Val(h_display) ) opt_run.pp.cropWidth+=16;
				break;
				case LETTERBOX_TOP_PLUS:
					opt_run.pp.letterboxTop++;
					if( !Val(h_display) ) opt_run.pp.letterboxTop--;
				break;
				case LETTERBOX_TOP_MINUS:
					opt_run.pp.letterboxTop--;
					if( !Val(h_display) ) opt_run.pp.letterboxTop++;
				break;
				case LETTERBOX_BOTTOM_PLUS:
					opt_run.pp.letterboxBottom++;
					if( !Val(h_display) ) opt_run.pp.letterboxBottom--;
				break;
				case LETTERBOX_BOTTOM_MINUS:
					opt_run.pp.letterboxBottom--;
					if( !Val(h_display) ) opt_run.pp.letterboxBottom++;
				break;
				case LETTERBOX_LEFT_PLUS:
					opt_run.pp.letterboxLeft++;
					if( !Val(h_display) ) opt_run.pp.letterboxLeft--;
				break;
				case LETTERBOX_LEFT_MINUS:
					opt_run.pp.letterboxLeft--;
					if( !Val(h_display) ) opt_run.pp.letterboxLeft++;
				break;
				case LETTERBOX_RIGHT_PLUS:
					opt_run.pp.letterboxRight++;
					if( !Val(h_display) ) opt_run.pp.letterboxRight--;
				break;
				case LETTERBOX_RIGHT_MINUS:
					opt_run.pp.letterboxRight--;
					if( !Val(h_display) ) opt_run.pp.letterboxRight++;
				break;
			}
			break;
		case WM_HSCROLL:
			int nPos;
			switch(LOWORD(wParam)){
				case TB_THUMBTRACK:
					break;
				case TB_THUMBPOSITION :
					nPos = SendDlgItemMessage(hDlg, SLIDER, TBM_GETPOS, 0, 0);   
					// Check for video object and get a frame
					if(! rs.video)
						break;
					video_opt.idctIndex = rs.prof.idctIndex;
					video_opt.recons_progressive = rs.prof.recons_progressive;
          video_opt.bStartInSync         = false;
					//Seek the video to the middle
					rs.video->SetStreamPos( ((i64)rs.video->GetStreamSize() * (nPos / 1000.0)) );
					rs.video->Start(&video_opt);
					rs.video->GetFrame(&PresInfo, &yuv);
					rs.video->Stop();
					//Yuv Convert it
					YUVtoRGB(yuv, (unsigned char *)opt_run.in.data);	
					// update iDAR
					opt_run.pp.iDAR = rs.video->DAR;
					UpdateDisplay(h_display);
			}
			break;
	}
    return FALSE;
}








///////////////////////////////////////////////////////////
//             GLOBAL    OPTIONS
///////////////////////////////////////////////////////////

// Retrieve timebase from video options tab
void GetFrameRate(HWND hDlg, compTimebaseRec *timebase){

	if(DlgIsChecked(hDlg,IDC_TB23)){ //23.976
		//HACK: LSX ENCODER DOESNT LIKE RIGHT TIMEBASE
		timebase->scale=		23976;
		timebase->sampleSize=	1000;
	}else if(DlgIsChecked(hDlg,IDC_TB24)){
		timebase->scale=		24;
		timebase->sampleSize=	1;
	}else if(DlgIsChecked(hDlg,IDC_TB29)){
		timebase->scale=		2997;
		timebase->sampleSize=	100;
	}else if(DlgIsChecked(hDlg,IDC_TB30)){
		timebase->scale=		30;
		timebase->sampleSize=	1;
	}else {
		timebase->scale=		25;
		timebase->sampleSize=	1;
	}
}

	// OnChildDialogInit - Positions the child dialog box to fall 
	//     within the display area of the tab control. 
 
	VOID WINAPI OnChildDialogInit(HWND hwndDlg) 
	{ 
    HWND hwndParent = GetParent(hwndDlg); 
    DLGHDR *pHdr = (DLGHDR *) GetWindowLong( 
        hwndParent, GWL_USERDATA); 
    SetWindowPos(hwndDlg, HWND_TOP, 
        pHdr->rcDisplay.left, pHdr->rcDisplay.top, 
        0, 0, SWP_NOSIZE); 
	} 


void GetFilter(HWND hDlg){
	// Check filter settings
	if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_NEAREST))
		opt_run.pp.filterMode=FILTER_NONE;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_BILINEAR))
		opt_run.pp.filterMode=FILTER_BILINEAR;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_BICUBIC))
		opt_run.pp.filterMode=FILTER_BICUBIC;
	else if(DlgIsChecked(hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC))
		opt_run.pp.filterMode=FILTER_TABLEBICUBIC;
	else 
		opt_run.pp.filterMode=FILTER_TABLEBICUBIC;
}

void DlgProcessingStartUp( HWND hDlg )
{
    FromConfigToPPost(&opt_run.current_profile, &opt_run.pp); 
    //CROP
	 UpdateCropAndLetterboxBoxes(hDlg);

    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_NEAREST );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_BILINEAR );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_BICUBIC );
    DlgUnCheck( hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC );
    switch(opt_run.current_profile.filter)
	{
	    case FILTER_NONE:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_NEAREST);
			break;
		case FILTER_BILINEAR:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_BILINEAR);
			break;
		case FILTER_BICUBIC:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_BICUBIC);
			break;
		case FILTER_TABLEBICUBIC:
			DlgCheck( hDlg, R_OPT_PPOST_RESIZING_HQBICUBIC);
			break;
	}
	if(opt_run.current_profile.keepAspect) 
		DlgCheck(hDlg, R_OPT_PPOST_KEEPASPECT);
	else 
	    DlgUnCheck(hDlg, R_OPT_PPOST_KEEPASPECT);
    return;
}

LRESULT CALLBACK DlgProcessing(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
				OnChildDialogInit(hDlg);
                DlgProcessingStartUp(hDlg);
				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			
			switch(LOWORD(wParam)){
				case R_OPT_PPOST_KEEPASPECT:
					opt_run.pp.doAR = DlgIsChecked(processing, R_OPT_PPOST_KEEPASPECT);
					break;
				case IDC_SHOWPAD:
					DialogBox(hInst, MAKEINTRESOURCE(DLG_OUTPUT_PAD), hDlg, (DLGPROC)DlgPad);
					break;

				case IDC_NOCROP:
					opt_run.pp.crop=!DlgIsChecked(hDlg, IDC_NOCROP);
					if(opt_run.pp.crop)
						EnableCropBoxes(hDlg);
					else
						DisableCropBoxes(hDlg);
					break;
				case IDC_NOLETTERBOX:
					opt_run.pp.letterbox = !DlgIsChecked(hDlg, IDC_NOLETTERBOX);
					if(opt_run.pp.letterbox )
						EnableLetterboxBoxes(hDlg);
					else
						DisableLetterboxBoxes(hDlg);
					break;
				case R_OPT_PPOST_RESIZING_NEAREST:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_BILINEAR:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_BICUBIC:
					GetFilter(hDlg); break;
				case R_OPT_PPOST_RESIZING_HQBICUBIC:
					GetFilter(hDlg); break;

				break;
			}
			break;
	}
    return FALSE;

}


/* Update settings from hDlg's video section.  If all settings are valid, return
 * true; if any setting is incorrect, print a warning, set focus on the erroneous
 * entry and return false.
 *
 * The allowed sizes depend on wether we're cropping or not. If we use crop,
 * the demand that sizes are a multiple of 16 goes for the cropping window,
 * with some more freedom for the resizing dimensions.
 */
static bool DlgVideo_Check(HWND hDlg)
{
	// width must be smaller than 1024 always,
	// a multiple of 16 if we're not cropping, a multiple of 2 if crop is on.
	if(DlgGetText(hDlg, IDC_WIDTH)<=0 || DlgGetText(hDlg, IDC_WIDTH)>=1024){
		PrintError(SIZE_OOR,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	if( (!opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_WIDTH)%16)!=0)){
		PrintError(SIZE_NOT16,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	if ( (opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_WIDTH)%2)!=0)){
		PrintError(INCORRECT_SETTING,(int)hDlg,0); // FIXME: should be size not /2
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	// height must be smaller than 1024 always,
	// a multiple of 16 if we're not cropping, free otherwise.
	if(DlgGetText(hDlg, IDC_HEIGHT)<=0 || DlgGetText(hDlg, IDC_HEIGHT)>=1024){
		PrintError(SIZE_OOR,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_HEIGHT);
		return false;
	}
	if( (!opt_run.pp.crop) && ((DlgGetText(hDlg, IDC_HEIGHT)%16)!=0)){
		PrintError(SIZE_NOT16,(int)hDlg,0);
		DlgSetFocus(hDlg, IDC_WIDTH);
		return false;
	}
	opt_run.pp.resWidth = DlgGetText(hDlg, IDC_WIDTH);
	opt_run.pp.resHeight = DlgGetText(hDlg, IDC_HEIGHT);
	return true;
}

static bool DlgProcessing_Check(HWND hDlg)
{
	GetLetterboxBoxesText(hDlg);
	GetCropBoxesText(hDlg);

	if(!CheckVideoParameters(&opt_run.pp)){
		PrintError(INCORRECT_SETTING, (int)hDlg,0);
		/* FIXME: we need to know where to set focus, but what's a
		 * clean way to do this without duplicating CheckVideoParameters() code? */
		// DlgSetFocus(hDlg, LOWORD(wParam));
		return false;
	}

	return true;
}
static bool DlgGeneral_Check(HWND hDlg)
{
	char szTemp[256];

	if(DlgIsChecked(hDlg,IDC_COMPILE_WHOLE)) return true; /* don't bother */

	GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
	if(!atof(szTemp)){
		PrintError(TIME_IS_NOT_POSITIVE, (int)hDlg,0);
		DlgSetFocus(hDlg, IDC_COMPILE_TIME);
		return false;
	}

	if(opt_run.current_profile.displayTimeUnits == 0) {
		/* it's in seconds; switch to frames */
		compTimebaseRec timebase;
		GetFrameRate(video, &timebase);

		GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
		opt_run.opt.frames_compile = (int)(atof(szTemp) * GET_FRAMERATE(timebase));
	} else {
		/* it's in frames; copy it directly */
		GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
		opt_run.opt.frames_compile = atoi(szTemp);
	} 

	return true;
}

void DlgVideoStartUp( HWND hDlg )
{
  FromConfigToPPost(&opt_run.current_profile, &opt_run.pp); 
  
		DlgSetText(hDlg, IDC_WIDTH, opt_run.pp.resWidth);
    DlgSetText(hDlg, IDC_HEIGHT, opt_run.pp.resHeight);
    
    // Uncheck all checkboxes
    DlgUnCheck( hDlg, IDC_TB23 );
    DlgUnCheck( hDlg, IDC_TB24 );
    DlgUnCheck( hDlg, IDC_TB29 );
    DlgUnCheck( hDlg, IDC_TB30 );
    DlgUnCheck( hDlg, IDC_TB25 );
    switch( opt_run.current_profile.timeBase.scale )
    {
    case 23976: //23.976
      DlgCheck(hDlg, IDC_TB23);
      break;
    case 24:    //24
      DlgCheck(hDlg, IDC_TB24);
      break; 
    case 2997: //29.97
      DlgCheck(hDlg, IDC_TB29);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.current_profile.recons_progressive = 0;
      break;
    case 30:    //30
      DlgCheck(hDlg, IDC_TB30);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.current_profile.recons_progressive = 0;
      break;
    default:	//if not, set 25
      DlgCheck(hDlg, IDC_TB25);
      DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
      opt_run.current_profile.recons_progressive = 0;
      break;
    }

    DlgCheckBoxState( hDlg, R_OPT_IDCT_AUTOSELECT, opt_run.current_profile.idctAutoSelect );
    if( DlgIsChecked(hDlg, R_OPT_IDCT_AUTOSELECT) )
      DlgDisable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
    else
      DlgEnable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
    
    DlgSetText(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD, opt_run.current_profile.threshold);
    DlgCheckBoxState( hDlg, R_OPT_TIMEBASE_RECONS, opt_run.current_profile.recons_progressive);
    DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK, opt_run.current_profile.deinterlace);
    DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND, opt_run.current_profile.blend);
    if(opt_run.current_profile.deinterlace)
    {
      DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
      DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
    }
    else
    {
      DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
      DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
    }
    return;
}

LRESULT CALLBACK DlgVideo(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  int i;
  switch (message)
  {
		case WM_INITDIALOG:
      OnChildDialogInit(hDlg);
      DlgVideoStartUp( hDlg );
      
      return TRUE;
      
    case WM_COMMAND:
      if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
      {
        EndDialog(hDlg, LOWORD(wParam));
        return TRUE;
      }
      TSelectorDialog selDlgInfo;
      // framerate
      switch(LOWORD(wParam))
      {
      case R_OPT_IDCT_PERFORMIEEE1180:
        {
          CIEEE1180Dlg obTest(hDlg, rs.hInst);
          // Initialize idct
          void (* pInitIdct)() = rs.video->GetIdctInfo(opt_run.current_profile.idctIndex)->InitIdct;
          void (* pDeInit)() = rs.video->GetIdctInfo(opt_run.current_profile.idctIndex)->DeInitIdct;

          if(pInitIdct)
            pInitIdct();
          obTest.IsIEEE1180(rs.video->GetIdctInfo(opt_run.current_profile.idctIndex)->Idct,
                            rs.video->GetIdctInfo(opt_run.current_profile.idctIndex)->sDescription);
          if(pDeInit)
            pDeInit();
          
        }
        break;
      case R_OPT_IDCT_SELECTIONBUTTON:
        selDlgInfo.tittle = "iDCT Selection";
        selDlgInfo.lateral_text = "Select iDCT";
        selDlgInfo.button_text  = "Ok";
        selDlgInfo.section_count = 1;
        selDlgInfo.sections_titles[0] = "Available iDCTs";

        LPFMpegIdctInfo idctInfo;
        selDlgInfo.strings[0].SetArraySize(rs.video->GetIdctCount());
        selDlgInfo.helpstrings[0].SetArraySize(rs.video->GetIdctCount());
        for(i=0; i<rs.video->GetIdctCount(); i++)
        {
          idctInfo = rs.video->GetIdctInfo(i);
          //selDlgInfo.strings[0][i] =  (char *)(idctInfo->sDescription);
          selDlgInfo.strings[0][i] =  (char *)(idctInfo->sName);
          selDlgInfo.helpstrings[0][i] =  (char *)(idctInfo->sDescription);
          selDlgInfo.section_mode[0] = MUST_SELECT | SINGLE_SELECT;
        }
        ResetSelections(&selDlgInfo);
        // Select first one
        selDlgInfo.selected[0][opt_run.current_profile.idctIndex] = 1;
        OpenSelectorDialog( hDlg, hInst, &selDlgInfo );
        // Get result of selection
        opt_run.current_profile.idctIndex = GetFirstSelected( &selDlgInfo, 0 );

        break;
      case R_OPT_IDCT_AUTOSELECT:
        if( DlgIsChecked(hDlg, R_OPT_IDCT_AUTOSELECT) )
          DlgDisable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
        else
          DlgEnable(hDlg, R_OPT_IDCT_SELECTIONBUTTON );
        break;
      case R_OPT_VIDEO_DEINTERLACE_CHECK:
        if(DlgIsChecked(hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK) )
        {
          DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
          DlgEnable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
        }
        else
        {
          DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND);
          DlgDisable(hDlg, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
        }
        break;
      case IDC_TB23: 
      case IDC_TB24:
        DlgEnable(hDlg, R_OPT_TIMEBASE_RECONS);
        break; 
      case IDC_TB25:
      case IDC_TB29:
      case IDC_TB30:
        DlgDisable(hDlg, R_OPT_TIMEBASE_RECONS);
        opt_run.current_profile.recons_progressive = 0;
        break;
      }
  }
  return FALSE;
}


void DlgAudioStartUp( HWND hDlg )
{
			//Setting init options
                DlgUnCheck( hDlg, R_OPT_AUDIO_DOAUDIO );
                DlgUnCheck( hDlg, R_OPT_AUDIO_DSC );
                DlgUnCheck( hDlg, R_OPT_AUDIO_NOAUDIO );

				int audiomode = -1;
				if(rs.audio)
        {
					if(rs.audio->GetFormat()!=MpegAudio  &&  rs.audio->GetFormat()!=Ac3 )
          {
						DlgDisable(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(hDlg, R_OPT_AUDIO_DSC);
						audiomode = NO_AUDIO;
					}
				}
				else{
						DlgDisable(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(hDlg, R_OPT_AUDIO_DSC);
						audiomode = NO_AUDIO;
				}

				if (audiomode < 0)
					audiomode = opt_run.current_profile.audioMode;

                DlgUnCheck( hDlg, IDC_44100 );
                DlgUnCheck( hDlg, IDC_48000 );
				if(opt_run.current_profile.outSampleFrequency==44100)
					DlgCheck(hDlg, IDC_44100);
				else
					DlgCheck(hDlg, IDC_48000);

				switch( audiomode ){
					case DO_AUDIO:
						DlgCheck(hDlg, R_OPT_AUDIO_DOAUDIO);
						DlgDisable(files, IDC_OUTPUT_FILE2);
						DlgDisable(files, IDC_FILEBUTTON2);
						if(opt_run.current_profile.sampleFreqSameAsInput){
							DlgDisable(hDlg, IDC_48000);
							DlgDisable(hDlg, IDC_44100);
							DlgCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						}
						else{
							DlgEnable(hDlg, IDC_48000);
							DlgEnable(hDlg, IDC_44100);
							DlgUnCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						}
						DlgEnable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
					case DSC:
						DlgCheck(hDlg, R_OPT_AUDIO_DSC);
						DlgEnable(files, IDC_OUTPUT_FILE2);
						DlgEnable(files, IDC_FILEBUTTON2);
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
					case NO_AUDIO:
						DlgCheck(hDlg, R_OPT_AUDIO_NOAUDIO);
						DlgDisable(files, IDC_OUTPUT_FILE2);
						DlgDisable(files, IDC_FILEBUTTON2);
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
						break;
				}
    return;
}

LRESULT CALLBACK DlgAudio(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:
				OnChildDialogInit(hDlg);
                DlgAudioStartUp( hDlg );
				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case R_OPT_AUDIO_SAMPLING_SAMEASINPUT:
					if(DlgIsChecked(hDlg,R_OPT_AUDIO_SAMPLING_SAMEASINPUT))
					{
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
					}
					else{
						DlgEnable(hDlg, IDC_48000);
						DlgEnable(hDlg, IDC_44100);
					}
					break;
				case R_OPT_AUDIO_DOAUDIO:
					DlgDisable(files, IDC_OUTPUT_FILE2);
					DlgDisable(files, IDC_FILEBUTTON2);
					if(opt_run.current_profile.sampleFreqSameAsInput){
						DlgDisable(hDlg, IDC_48000);
						DlgDisable(hDlg, IDC_44100);
						DlgCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					}
					else{
						DlgEnable(hDlg, IDC_48000);
						DlgEnable(hDlg, IDC_44100);
						DlgUnCheck(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					}
					DlgEnable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
				case R_OPT_AUDIO_DSC:
					DlgEnable(files, IDC_OUTPUT_FILE2);
					DlgEnable(files, IDC_FILEBUTTON2);
					DlgDisable(hDlg, IDC_48000);
					DlgDisable(hDlg, IDC_44100);
					DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
				case R_OPT_AUDIO_NOAUDIO:
					DlgDisable(files, IDC_OUTPUT_FILE2);
					DlgDisable(files, IDC_FILEBUTTON2);
					DlgDisable(hDlg, IDC_48000);
					DlgDisable(hDlg, IDC_44100);
					DlgDisable(hDlg, R_OPT_AUDIO_SAMPLING_SAMEASINPUT);
					break;
			}
			break;
	}
    return FALSE;
}


void DlgGeneralStartUp( HWND hDlg )
{
    	if(opt_run.current_profile.compileWhole){
    		// whole file
	    	DlgCheck( hDlg, IDC_COMPILE_WHOLE);
		    // disable edit box and unit selection
    		DlgDisable( hDlg, IDC_COMPILE_TIME);
	    	DlgDisable( hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
		    DlgDisable( hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
		}
		else
		{
            DlgUnCheck( hDlg, IDC_COMPILE_WHOLE);
			DlgEnable(hDlg, IDC_COMPILE_TIME);
			DlgEnable(hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
			DlgEnable(hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
		}

		opt_run.opt.frames_compile = opt_run.current_profile.framesToCompile;
				
		/* make sure we have a sane frame count, and not our magic '300000' */
		if(opt_run.current_profile.compileWhole) opt_run.opt.frames_compile = 100;

		if(opt_run.current_profile.displayTimeUnits == 0) {
    		/* secs */
	    	compTimebaseRec timebase;
		    GetFrameRate(video, &timebase);

    		DlgSetDouble( hDlg, IDC_COMPILE_TIME, opt_run.current_profile.framesToCompile / GET_FRAMERATE(timebase));
	    	DlgCheck(hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
		} else if(opt_run.current_profile.displayTimeUnits == 1) {
			/* frames */
			DlgSetText( hDlg, IDC_COMPILE_TIME, opt_run.current_profile.framesToCompile);
			DlgCheck(hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
		}

		DlgSetText(hDlg, IDC_LURKSIZE, opt_run.current_profile.lurk_size );

		if(opt_run.current_profile.shutdown)
			DlgCheck(hDlg, R_GENERAL_MISC_SHUTDOWN);
		else
			DlgUnCheck(hDlg, R_GENERAL_MISC_SHUTDOWN);
    return;
}

LRESULT CALLBACK DlgGeneral(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	char szTemp[1024];
	switch (message)
	{
		case WM_INITDIALOG:
			OnChildDialogInit(hDlg);
            DlgGeneralStartUp(hDlg);

				return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case IDC_COMPILE_WHOLE:
					opt_run.current_profile.compileWhole = DlgIsChecked(hDlg,IDC_COMPILE_WHOLE);
					if(opt_run.current_profile.compileWhole){
						// whole file
						DlgCheck( hDlg, IDC_COMPILE_WHOLE);
						// disable edit box and unit selection
						DlgDisable( hDlg, IDC_COMPILE_TIME);
						DlgDisable( hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
						DlgDisable( hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
					}
					else
					{
						DlgEnable(hDlg, IDC_COMPILE_TIME);
						DlgEnable(hDlg, R_GENERAL_COMPTIME_SECONDSCOMPILE);
						DlgEnable(hDlg, R_GENERAL_COMPTIME_FRAMESCOMPILE);
					}

					break;
				case R_GENERAL_COMPTIME_SECONDSCOMPILE:
					if(HIWORD(wParam) == BN_CLICKED && opt_run.current_profile.displayTimeUnits == 1) {
						/* switch IDC_COMPILE_TIME to seconds */						
						compTimebaseRec timebase;
						GetFrameRate(video, &timebase);

						GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
						DlgSetDouble( hDlg, IDC_COMPILE_TIME, atoi(szTemp) / GET_FRAMERATE(timebase));
						opt_run.current_profile.displayTimeUnits = 0;
					}
					

					break;
				case R_GENERAL_COMPTIME_FRAMESCOMPILE:
					if(HIWORD(wParam) == BN_CLICKED && opt_run.current_profile.displayTimeUnits == 0) {
						/* switch IDC_COMPILE_TIME to frames */
						compTimebaseRec timebase;
						GetFrameRate(video, &timebase);

						GetDlgItemText( hDlg, IDC_COMPILE_TIME, szTemp, 1024);
						DlgSetText( hDlg, IDC_COMPILE_TIME, (int) (atof(szTemp) * GET_FRAMERATE(timebase)));
						opt_run.current_profile.displayTimeUnits = 1;
					}

					break;
			}
			break;
	}
    return FALSE;
}

void DlgFilesStartUp( HWND hDlg )
{
	DlgSetText(hDlg, IDC_OUTPUT_FILE1, opt_run.current_profile.outputFile);
	DlgSetText(hDlg, IDC_OUTPUT_FILE2, opt_run.current_profile.audioOutFile);	
	if(opt_run.current_profile.warn_overwrite)
		DlgCheck(hDlg, R_FILES_WARN);
	else
    	DlgUnCheck(hDlg, R_FILES_WARN);
    return;
}

LRESULT CALLBACK DlgFiles(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	char szTemp[MAXFILENAME];

	switch (message)
	{
		case WM_INITDIALOG:
			OnChildDialogInit(hDlg);
            DlgFilesStartUp( hDlg );
			return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
			}
			switch(LOWORD(wParam)){
				case IDC_FILEBUTTON1:
					strcpy( szTemp, opt_run.current_profile.outputFile);
					if(GetOutputFileName(hDlg, szTemp, opt_run.current_profile.outputDir, "FlasK MPEG - Select output video file")){
						DlgSetText(hDlg, IDC_OUTPUT_FILE1, szTemp);
					}
					break;
				case IDC_FILEBUTTON2:
					strcpy( szTemp, opt_run.current_profile.audioOutFile);
					if(GetOutputFileName(hDlg, szTemp, opt_run.current_profile.outputDir, "FlasK MPEG - Select output audio file")){
						DlgSetText(hDlg, IDC_OUTPUT_FILE2, szTemp);
					}
					break;
			}
			break;
	}
    return FALSE;
}
// DoLockDlgRes - loads and locks a dialog template resource. 
// Returns the address of the locked resource. 
// lpszResName - name of the resource 
DLGTEMPLATE * WINAPI DoLockDlgRes(LPCSTR lpszResName) 
{ 
    HRSRC hrsrc = FindResource(NULL, lpszResName, RT_DIALOG); 
    HGLOBAL hglb = LoadResource(hInst, hrsrc); 
    return (DLGTEMPLATE *) LockResource(hglb); 
} 


// OnSelChanged - processes the TCN_SELCHANGE notification. 
// hwndDlg - handle to the parent dialog box. 
VOID WINAPI OnSelChanged(HWND hwndDlg) 
{ 
	    DLGHDR *pHdr = (DLGHDR *) GetWindowLong( 
        hwndDlg, GWL_USERDATA); 
    int iSel =opt_run.current_profile.selectedTab= TabCtrl_GetCurSel(pHdr->hwndTab); 
 
    // Hide the current child dialog box, if any. 
    if (pHdr->hwndDisplay != NULL) 
        ShowWindow(pHdr->hwndDisplay,SW_HIDE); 
 
    // Show. 
    pHdr->hwndDisplay = pHdr->DlgTabs[iSel];
        ShowWindow(pHdr->hwndDisplay,SW_SHOW);
} 


VOID WINAPI TabbedDialogInit(HWND hwndDlg) 
{ 
    pHdr = (DLGHDR *) LocalAlloc(LPTR, sizeof(DLGHDR)); 
    DWORD dwDlgBase = GetDialogBaseUnits(); 
    int cxMargin = LOWORD(dwDlgBase) / 4; 
    int cyMargin = HIWORD(dwDlgBase) / 8; 
    TCITEM tie; 
    RECT rcTab; 


    // Save a pointer to the DLGHDR structure. 
    SetWindowLong(hwndDlg, GWL_USERDATA, (LONG) pHdr); 
 
    // Create the tab control. 
    InitCommonControls(); 
    pHdr->hwndTab = CreateWindow( 
        WC_TABCONTROL, "", 
        WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE, 
        0, 0, 100, 100, 
        hwndDlg, NULL, hInst, NULL 
        ); 
    if (pHdr->hwndTab == NULL) {
        return ;
    }
	// Set font
	SendMessage(pHdr->hwndTab, WM_SETFONT, (WPARAM)myFont, MAKELPARAM(TRUE, 0));    
 
    // Add a tab for each of the three child dialog boxes. 
    tie.mask = TCIF_TEXT | TCIF_IMAGE; 
    tie.iImage = -1; 
    tie.pszText = "Video"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 0, &tie); 
    tie.pszText = "Audio"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 1, &tie); 
    tie.pszText = "Post Processing"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 2, &tie); 
     tie.pszText = "Files"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 3, &tie); 
     tie.pszText = "General"; 
    TabCtrl_InsertItem(pHdr->hwndTab, 4, &tie); 
 
 
    SetRectEmpty(&rcTab); 
    rcTab.right = (292 * LOWORD(dwDlgBase)) / 4; 
    rcTab.bottom = (130 * HIWORD(dwDlgBase)) / 8; 
 
    // Calculate how large to make the tab control, so 
    // the display area can accommodate all the child dialog boxes. 
    TabCtrl_AdjustRect(pHdr->hwndTab, TRUE, &rcTab); 
    OffsetRect(&rcTab, cxMargin - rcTab.left, 
            cyMargin - rcTab.top); 
 
    // Calculate the display rectangle. 
    CopyRect(&pHdr->rcDisplay, &rcTab); 
    TabCtrl_AdjustRect(pHdr->hwndTab, FALSE, &pHdr->rcDisplay); 
 
    // Set the size and position of the tab control, buttons, 
    // and dialog box. 
    SetWindowPos(pHdr->hwndTab, NULL, rcTab.left, rcTab.top, 
            rcTab.right - rcTab.left, rcTab.bottom - rcTab.top, 
            SWP_NOZORDER); 
 
  
	//Create dialogs
	// files tab has dependences from all the others (audio)
	files=		pHdr->DlgTabs[3]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_FILES), hwndDlg, (DLGPROC)DlgFiles);

	video=		pHdr->DlgTabs[0]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_VIDEO_OPTIONS), hwndDlg, (DLGPROC)DlgVideo);
	audio=		pHdr->DlgTabs[1]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_AUDIO_OPTIONS), hwndDlg, (DLGPROC)DlgAudio);
	processing=	pHdr->DlgTabs[2]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_PROCESSING_OPTIONS), hwndDlg, (DLGPROC)DlgProcessing);
	general=	pHdr->DlgTabs[4]=CreateDialog(hInst, MAKEINTRESOURCE(DLG_GENERAL), hwndDlg, (DLGPROC)DlgGeneral);

	pHdr->DlgTabCheck[0] = DlgVideo_Check;
	pHdr->DlgTabCheck[1] = pHdr->DlgTabCheck[3] = NULL;
	pHdr->DlgTabCheck[2] = DlgProcessing_Check;
	pHdr->DlgTabCheck[4] = DlgGeneral_Check;

	// Select Tab from options
	TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.current_profile.selectedTab);
	OnSelChanged(hwndDlg); 
    
} 
 

// RefreshTabsFromProfile()
//    Refreses the current tabs from a given profile
void RefreshTabsFromProfile(HWND hDlg)
{
    //Select Tab from profile
    TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.current_profile.selectedTab);
    OnSelChanged(hDlg);

	//Get post processing options from config
	FromConfigToPPost(&opt_run.current_profile, &opt_run.pp);
    
    DlgProcessingStartUp( processing );
    DlgVideoStartUp( video );
    DlgAudioStartUp( audio );
    DlgGeneralStartUp( general );
    DlgFilesStartUp( files );

    return;
}

void RefreshListBox( HWND hDlg )
{
    int i;
    //Language Selection
    //DlgSetText(hDlg, R_WELCOME_LANG, GS(MENU_LANGUAGE));
    ListClean(GetDlgItem(hDlg, IDC_PROFILES_LIST));
	for(i=0; i<rs.profiler->GetCount() ; i++){
		ListAddText(GetDlgItem(hDlg, IDC_PROFILES_LIST), (rs.profiler->Get(i))->profile_name );
	}
	ListSetCur(GetDlgItem(hDlg, IDC_PROFILES_LIST), rs.profiler->GetSelectedIndex() );
    return;
}

void RefreshOptionsDialog( HWND hDlg )
{
    // Get selected profile
    memcpy( &opt_run.current_profile, rs.profiler->GetSelected(), sizeof(TProfile) );

	//Get post processing options from config
	FromConfigToPPost(&opt_run.current_profile, &opt_run.pp);

    RefreshListBox(hDlg);
    RefreshTabsFromProfile(hDlg);
    return;
}

void RetrieveProfile( HWND hDlg )
{


    	//Video options
        opt_run.pp.resWidth  = DlgGetText( video, IDC_WIDTH);
        opt_run.pp.resHeight = DlgGetText( video, IDC_HEIGHT);

    //iDCT options
    if( opt_run.current_profile.idctAutoSelect = DlgIsChecked( video, R_OPT_IDCT_AUTOSELECT ) )
      opt_run.current_profile.idctIndex = rs.video->SelectIdctDefault();


		DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_CHECK, opt_run.current_profile.deinterlace);
		DlgCheckBoxState( hDlg, R_OPT_VIDEO_DEINTERLACE_BLEND, opt_run.current_profile.blend);

        //Retrieve timebase
		GetFrameRate( video, &opt_run.current_profile.timeBase );
		// opt_run.current_profile.secondsToCompile = opt_run.opt.seconds;
		opt_run.current_profile.framesToCompile  = opt_run.opt.frames_compile;
		opt_run.current_profile.recons_progressive = DlgIsChecked(video, R_OPT_TIMEBASE_RECONS);

		opt_run.pp.deinterlace   = DlgIsChecked(video, R_OPT_VIDEO_DEINTERLACE_CHECK);
		opt_run.pp.blend_fields  = DlgIsChecked(video, R_OPT_VIDEO_DEINTERLACE_BLEND);
		opt_run.pp.threshold     = DlgGetText(video, R_OPT_VIDEO_DEINTERLACE_THRESHOLD);
		//Audio options
		    if(DlgIsChecked(audio, R_OPT_AUDIO_DSC))
				opt_run.current_profile.audioMode=DSC;
			else if(DlgIsChecked(audio, R_OPT_AUDIO_DOAUDIO)){
				    opt_run.current_profile.audioMode=DO_AUDIO;
        			if(DlgIsChecked(audio, IDC_48000)) opt_run.current_profile.outSampleFrequency=48000;
		        	else opt_run.current_profile.outSampleFrequency=44100;
        			if(DlgIsChecked(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT)) opt_run.current_profile.sampleFreqSameAsInput=true;
		        	else opt_run.current_profile.sampleFreqSameAsInput=false;
			    }
				else 
                    opt_run.current_profile.audioMode=NO_AUDIO;

        //Processing
		//Files
		DlgGetText( files, IDC_OUTPUT_FILE1, opt_run.current_profile.outputFile);	
		DlgGetText( files, IDC_OUTPUT_FILE2, opt_run.current_profile.audioOutFile);
		opt_run.current_profile.warn_overwrite = DlgIsChecked(files, R_FILES_WARN);
		
		//General
		opt_run.current_profile.shutdown = DlgIsChecked(general, R_GENERAL_MISC_SHUTDOWN);
		if(DlgIsChecked(general,IDC_COMPILE_WHOLE)){ //If we don't have the CompileWhole
			opt_run.current_profile.framesToCompile = 3000000; // at least 25hrs at 30fps
		}
		else
		{
			opt_run.current_profile.framesToCompile =  opt_run.opt.frames_compile;
		}
    opt_run.current_profile.lurk_size = DlgGetText( general, IDC_LURKSIZE );

        // Post Processing
	    FromPPostToConfig(&opt_run.current_profile, &opt_run.pp);
        // Right now, all the data from the current options is in opt_run.current_profile
        return;
}

// Mesage handler for output options
LRESULT CALLBACK OutputOptionsDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	
	static LOGFONT logFont;
    char szTemp[MAX_PATH];

	switch (message)
	{
		case WM_INITDIALOG:
      // Get selected profile
      memcpy( &opt_run.current_profile, rs.profiler->GetSelected(), sizeof(TProfile) );

			//Get post processing options from config
			FromConfigToPPost(&opt_run.current_profile, &opt_run.pp);

			strcpy(logFont.lfFaceName,"MS Sans Serif");
			logFont.lfWeight=FW_NORMAL;
			logFont.lfHeight=8; 
			myFont=CreateFontIndirect(&logFont);
			TabbedDialogInit(hDlg);
			

      RefreshListBox(hDlg);

			// SET LANGUAGE
			DlgSetText(hDlg, IDOK,GS(OPT_OK));
			DlgSetText(hDlg, IDCANCEL,GS(OPT_CANCEL));
			DlgSetText(hDlg, IDC_SHOWPAD ,GS(OPT_SHOWPAD));
			DlgSetText(hDlg, IDC_1, GS(OPT_SHOWPAD_EXP));
			SetText(hDlg, GS(OPT_TITLE));
			// Setting Tabs
			TabSetText(pHdr->hwndTab, 0, GS(OPT_VIDEO) );
			TabSetText(pHdr->hwndTab, 1, GS(OPT_AUDIO) );
			TabSetText(pHdr->hwndTab, 2, GS(OPT_PPOST) );
			TabSetText(pHdr->hwndTab, 3, GS(OPT_FILES) );
			TabSetText(pHdr->hwndTab, 4, GS(OPT_GENERAL) );
			// Video
			DlgSetText(video, R_OPT_FRAMESIZE, GS(OPT_FRAMESIZE));
			DlgSetText(video, R_OPT_WIDTH, GS(OPT_WIDTH));
			DlgSetText(video, R_OPT_HEIGHT, GS(OPT_HEIGHT));
			DlgSetText(video, R_OPT_FRAMESIZE_EXP, GS(OPT_FRAMESIZE_EXP));
			DlgSetText(video, R_OPT_OUTPAD_EXP, GS(OPT_OUTPAD_EXP));
			DlgSetText(video, R_OPT_TIMEBASE, GS(OPT_TIMEBASE));
			DlgSetText(video, R_OPT_TIMEBASE_RECONS, GS(R_OPT_TIMEBASE_RECONS) );
			DlgSetText(video, R_OPT_TIMEBASE_EXP, GS(OPT_TIMEBASE_EXP));
			DlgSetText(video, R_OPT_IDCT_TITLE, GS(OPT_IDCT_TITLE));
			DlgSetText(video, R_OPT_IDCT_EXP, GS(OPT_IDCT_EXP));
			DlgSetText(video, R_OPT_VIDEO_NOTE, GS(OPT_VIDEO_NOTE));
			
			DlgSetText(video, R_OPT_VIDEO_VIDEOSTRUCTURE, GS(R_OPT_VIDEO_VIDEOSTRUCTURE));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_CHECK, GS(R_OPT_VIDEO_DEINTERLACE_CHECK));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_BLEND, GS(R_OPT_VIDEO_DEINTERLACE_BLEND));
			DlgSetText(video, R_OPT_VIDEO_DEINTERLACE_THRESHOLD_CAPTION, GS(R_OPT_VIDEO_DEINTERLACE_THRESHOLD_CAPTION));
			
			// Audio
			DlgSetText(audio, R_OPT_AUDIO_AUDIOMODE_EXP, GS(OPT_AUDIO_AUDIOMODE_EXP));
			DlgSetText(audio, R_OPT_AUDIO_AUDIOMODE, GS(OPT_AUDIO_AUDIOMODE));
			DlgSetText(audio, R_OPT_AUDIO_DSC, GS(OPT_AUDIO_DSC));
			DlgSetText(audio, R_OPT_AUDIO_DOAUDIO, GS(OPT_AUDIO_DOAUDIO));
			DlgSetText(audio, R_OPT_AUDIO_NOAUDIO, GS(OPT_AUDIO_NOAUDIO));
			DlgSetText(audio, R_OPT_AUDIO_DSC_EXP, GS(OPT_AUDIO_DSC_EXP));
			DlgSetText(audio, R_OPT_AUDIO_DOAUDIO_EXP, GS(OPT_AUDIO_DOAUDIO_EXP));
			DlgSetText(audio, R_OPT_AUDIO_NOAUDIO_EXP, GS(OPT_AUDIO_NOAUDIO_EXP));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_EXP, GS(OPT_AUDIO_SAMPLING_EXP));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_TITLE, GS(OPT_AUDIO_SAMPLING_TITLE));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT, GS(OPT_AUDIO_SAMPLING_SAMEASINPUT));
			DlgSetText(audio, R_OPT_AUDIO_SAMPLING_SAMEASINPUT_EXP, GS(OPT_AUDIO_SAMPLING_SAMEASINPUT_EXP));
			// Post Processing
			DlgSetText(processing, R_OPT_PPOST_RESIZING_TITLE  , GS(R_OPT_PPOST_RESIZING_TITLE ));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_NEAREST, GS(R_OPT_PPOST_RESIZING_NEAREST));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_BILINEAR, GS(R_OPT_PPOST_RESIZING_BILINEAR));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_BICUBIC, GS(R_OPT_PPOST_RESIZING_BICUBIC));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_HQBICUBIC, GS(R_OPT_PPOST_RESIZING_HQBICUBIC));
			DlgSetText(processing, R_OPT_PPOST_RESIZING_NOTE, GS(R_OPT_PPOST_RESIZING_NOTE));
			DlgSetText(processing, R_OPT_PPOST_CROP_TITLE, GS(R_OPT_PPOST_CROP_TITLE));
			DlgSetText(processing, IDC_NOCROP  , GS(IDC_NOCROP ));
			DlgSetText(processing, IDC_NOLETTERBOX, GS(IDC_NOLETTERBOX));
			DlgSetText(processing, R_OPT_PPOST_CROP_TOPOFFSET, GS(R_OPT_PPOST_CROP_TOPOFFSET));
			DlgSetText(processing, R_OPT_PPOST_CROP_HEIGHT, GS(R_OPT_PPOST_CROP_HEIGHT));
			DlgSetText(processing, R_OPT_PPOST_CROP_LEFTOFFSET, GS(R_OPT_PPOST_CROP_LEFTOFFSET));
			DlgSetText(processing, R_OPT_PPOST_CROP_WIDTH, GS(R_OPT_PPOST_CROP_WIDTH));
			DlgSetText(processing, R_OPT_PPOST_CROP_TOP  , GS(R_OPT_PPOST_CROP_TOP ));
			DlgSetText(processing, R_OPT_PPOST_CROP_BOTTOM, GS(R_OPT_PPOST_CROP_BOTTOM));
			DlgSetText(processing, R_OPT_PPOST_CROP_LEFT, GS(R_OPT_PPOST_CROP_LEFT));
			DlgSetText(processing, R_FILES_AUDIOBOX_TITLE, GS(R_FILES_AUDIOBOX_TITLE));
			DlgSetText(processing, R_OPT_PPOST_CROP_RIGHT, GS(R_OPT_PPOST_CROP_RIGHT));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT, GS(R_OPT_PPOST_KEEPASPECT));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT_EXP, GS(R_OPT_PPOST_KEEPASPECT_EXP));
			DlgSetText(processing, R_OPT_PPOST_SHOWPAD_EXP, GS(R_OPT_PPOST_SHOWPAD_EXP));
			DlgSetText(processing, R_OPT_PPOST_CROP_SETTINGS_EXP, GS(R_OPT_PPOST_CROP_SETTINGS_EXP));
			DlgSetText(processing, R_OPT_PPOST_CROP_LETTERBOX_EXP, GS(R_OPT_PPOST_CROP_LETTERBOX_EXP));
			DlgSetText(processing, R_OPT_PPOST_KEEPASPECT, GS(R_OPT_PPOST_KEEPASPECT));
			DlgSetText(processing, IDC_SHOWPAD, GS(OPT_SHOWPAD));
			// Files
			DlgSetText(files, R_FILES_OUTFILES_TITLE  , GS(R_FILES_OUTFILES_TITLE ));
			DlgSetText(files, R_FILES_VIDEOBOX_TITLE, GS(R_FILES_VIDEOBOX_TITLE));
			DlgSetText(files, R_FILES_OUTVIDEO_SEX, GS(R_FILES_OUTVIDEO_SEX));
			DlgSetText(files, R_FILES_AUDIOBOX_TITLE, GS(R_FILES_AUDIOBOX_TITLE));
			DlgSetText(files, R_FILES_AUDIOBOX_EXP, GS(R_FILES_AUDIOBOX_EXP));
			DlgSetText(files, R_FILES_WARN, GS(R_FILES_WARN));
			DlgSetText(files, IDC_FILEBUTTON1, GS(1113));
			DlgSetText(files, IDC_FILEBUTTON2, GS(1113));
			// General
			DlgSetText(general, R_GENERAL_COMPTIME_TITLE  , GS(R_GENERAL_COMPTIME_TITLE ));
			DlgSetText(general, R_GENERAL_COMPTIME_SECONDSCOMPILE, GS(R_GENERAL_COMPTIME_SECONDSCOMPILE));
			DlgSetText(general, R_GENERAL_COMPTIME_FRAMESCOMPILE, GS(R_GENERAL_COMPTIME_FRAMESCOMPILE));
			DlgSetText(general, IDC_COMPILE_WHOLE  , GS(IDC_COMPILE_WHOLE ));
			DlgSetText(general, R_GENERAL_COMPTIME_COMPTIME_EXP1, GS(R_GENERAL_COMPTIME_COMPTIME_EXP1));

			// DlgSetText(general, R_GENERAL_COMPTIME_COMPTIME_EXP2, GS(R_GENERAL_COMPTIME_COMPTIME_EXP2));
			DlgSetText(general, R_GENERAL_SEARCHSIZE_TITLE, GS(R_GENERAL_SEARCHSIZE_TITLE));			
			DlgSetText(general, R_GENERAL_SEARCHSIZE_BOX, GS(R_GENERAL_SEARCHSIZE_BOX));
			DlgSetText(general, R_GENERAL_SEARCHSIZE_EXP, GS(R_GENERAL_SEARCHSIZE_EXP));
			DlgSetText(general, R_GENERAL_MISC_TITLEIDC_STATIC, GS(R_GENERAL_MISC_TITLEIDC_STATIC));
			DlgSetText(general, R_GENERAL_MISC_SHUTDOWN, GS(R_GENERAL_MISC_SHUTDOWN));			


			//Load from config
			//Audio output
			//If its not an AC3 disable DO_AUDIO
/*			if(!audio->isAC3){
				DlgDisable(hDlg, IDC_DOAUDIO);
				if(opt_run.current_profile.audioMode==DO_AUDIO) opt_run.current_profile.audioMode=DSC;
			}
			else{
				DlgEnable(hDlg, IDC_DOAUDIO);
			}*/


			return TRUE;
		case WM_NOTIFY:
			switch(((LPNMHDR)lParam)->code){
				case TCN_SELCHANGE:
					/* is all data valid? */
					if(pHdr->DlgTabCheck[opt_run.current_profile.selectedTab] &&
						!pHdr->DlgTabCheck[opt_run.current_profile.selectedTab](pHdr->hwndDisplay)) {
						/* there's a problem; set the focused tab back */
						TabCtrl_SetCurSel(pHdr->hwndTab, opt_run.current_profile.selectedTab);
					    return FALSE;
					}

					/* we're OK */
					OnSelChanged(hDlg);

					break;
			}
			break;
		case WM_COMMAND:
			//Notifications
			switch(LOWORD(wParam))
			{
				case IDOK:
					/* is all data valid? */
					if(pHdr->DlgTabCheck[opt_run.current_profile.selectedTab] &&
						!pHdr->DlgTabCheck[opt_run.current_profile.selectedTab](pHdr->hwndDisplay)) {
						/* there's a problem */
					    return FALSE;
					}

          RetrieveProfile(hDlg);

          // If the current profile is different from the selected one
          if (memcmp( &opt_run.current_profile, rs.profiler->GetSelected(), sizeof(TProfile) ) != 0)
          {
              // Change the name of the profile to "Default"
              strcpy( opt_run.current_profile.profile_name, "Default");
              // Store it as the default
              int nDefaultProfileIdx = rs.profiler->GetDefaultIndex();

              rs.profiler->DeleteProfile(nDefaultProfileIdx);
              rs.profiler->AddProfile(&opt_run.current_profile);
              // And select it
              rs.profiler->SelectDefault();
          }
          
          // Set Global Run State profile from selected one
          rs.prof = *rs.profiler->GetSelected();
          o.options.selected_profile = rs.profiler->GetSelectedIndex();

					DeleteObject(myFont);
					EndDialog(hDlg, LOWORD(wParam));
					return TRUE;
				break;
                case R_OPT_SAVECURRENT_BUTTON:
                    if (DialogBoxParam( hInst, 
                                        MAKEINTRESOURCE(IDD_OUTPUT_OPTIONS_PROFILE_NAME),
                                        hDlg, 
                                        (DLGPROC)OptionsProfileNameDlg,
                                        (LPARAM)szTemp) == IDOK)
                    {
                        //The string is in szTemp
                        RetrieveProfile(hDlg);
                        strcpy( opt_run.current_profile.profile_name, szTemp);
                        rs.profiler->AddProfile( &opt_run.current_profile );
                        RefreshOptionsDialog(hDlg);
                    }
                    break;
                case R_OPT_DELETEPROFILE_BUTTON:
                        //Delete selected profile
                        rs.profiler->DeleteProfile( ListGetCur(GetDlgItem(hDlg, IDC_PROFILES_LIST)) );
                        RefreshOptionsDialog(hDlg);
                    break;
				case IDC_SHOWPAD:
						DialogBox(hInst, MAKEINTRESOURCE(DLG_OUTPUT_PAD), hDlg, (DLGPROC)DlgPad);
					break;

				case IDCANCEL:
					EndDialog(hDlg, LOWORD(wParam));
					return TRUE;
				break;
				
			}
            if(HIWORD(wParam)==CBN_SELCHANGE){
                rs.profiler->Select( ListGetCur( GetDlgItem(hDlg, IDC_PROFILES_LIST)) );
				RefreshOptionsDialog( hDlg );

				return TRUE;
			}
			break;
	}
    return FALSE;
}

LRESULT CALLBACK OptionsProfileNameDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    static char  *string_buffer=NULL;

	switch (message)
	{
		case WM_INITDIALOG:
                string_buffer = (char *) lParam;
				return TRUE;

		case WM_COMMAND:

            DlgGetText(hDlg, IDC_EDITBOX, string_buffer );

			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
				EndDialog(hDlg, LOWORD(wParam));
				return TRUE;
            }
			break;
	}
    return FALSE;
}
