/* 
 *  FlasKMPEG Mpeg Input Stream Module API 1.0
 *  
 *  This API defines a module to input mpeg streams to FlasKMPEG.
 *
 *	Copyright (C) Alberto Vigata - December 2000  ultraflask@yahoo.com
 *    
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

/*   This file defines MISM for FlasKMPEG where MISM stands for Mpeg Input Stream Module.
   The definition of this module is an attempt to standarize all kinds of MPEG sources
   to FlasKMPEG. That includes for example Transport Streams, Program Streams, 
   System Streams, Elementary Streams ...
     Version 1.0 of this API only accepts Program Streams as a valid output. That means that
   any source must be converted to a valid MPEG1 or MPEG2 program stream to be able to 
   feed the stream to FlasKMPEG. Future versions will support elementary streams as well as PES
   streams. 
     MISM modules must be reentrant.
*/

#ifndef MISMAPI_H
#define MISMAPI_H

#include "flasktypes.h"

#define	FM_MISM_VERSION		0x00000100
#define	FM_MISM_OK	  		0x00000004
#define	FM_MISM_ERROR			0x00000005

// Features flags
#define	FM_MISM_READFILES	  0x00000001 // The input source of this module are files

// Output stream types
#define FM_MISM_PROGRAMSTREAM   0X00000001  // Only this type is supported as version 1.0 of MISM
#define FM_MISM_TRANSPORTSTREAM 0X00000002
#define FM_MISM_AUDIOES         0X00000004
#define FM_MISM_VIDEOES         0X00000008

#define FM_MISM_MERIT_DONOTUSE  0
#define FM_MISM_MERIT_NORMAL    FM_MISM_MERIT_DONOTUSE + 5 

struct fmStreamId
{
  BYTE    streamId;   // Main stream id of an mpeg stream
  BYTE subStreamId;   // substream id for mpeg private streams
  char  streamName[256]; // Identifier name for this stream
};

#define FM_MISM_MAXSTREAMS 16
struct fmStreamIds
{
  fmStreamId vIds[FM_MISM_MAXSTREAMS];  // vector of stream ids that we are interested
  int       idAudioDef; // index of the default audio track
  int       idCount;    // number of ids contained inside vIds. Up to 16.
};

typedef struct
{
  char *(* GetString)(int x); // FlasK localization strings
  int  lurkSize;    // This parameter can be used as a guide to know how many bytes
                    //   the module can seek from the beginning of the file in search of streams.
  HWND      hWnd;
  HINSTANCE hInst;
  
} FMMismStdParms,*LPFMMismStdParms;

typedef struct tagMismProperties
{
  DWORD dwSize;   // Size of this structure
  DWORD dwMismId; // Identifier for this Mism
} FMMismProperties, *LPFMMismProperties;

typedef long fmHandle;

typedef	struct {
  // basic information
  DWORD	dwSize;	    	      // Size of this structure
  DWORD dwVersion;          // Must be FM_MISM_VERSION
  DWORD dwOutputTypes;      // Output types that this modules can output. Must 
                            //  support FM_MISM_PROGRAMSTREAM for version 1.0 of MISM
  DWORD dwReserved;         // Not used as MISM 1.0. Set it to 0.
  LPFMMismProperties pProperties; // Properties of this Mism.
  char  sDescription[256];  // Description of the module. Brief explanation of what this module does.
  char  sExtensions[256];   // a list of file patterns (*.vob;*.ifo) sep. by ; For example
                            // if your module accepts program streams as input one valid string could be:
                            // *.m2v;*.mpg;*.vob;*.m1v;*.m2v;*.mpe;*.mpeg;*.mv2;*.mpv
                            // Note: the file extensions defined here don't preclude the opening of the stream
                            //       See CanOpenFile() below.
  // File IO ptrs
  bool (*CanOpenFile)(char *file, DWORD *merit); // A file path is passed through file. If the module has
                                   //   the ability to parse that file it return true.

  fmHandle (*OpenStream)(char *file); 
  // Returns a handle to the opened stream, if failure returns NULL.
  //  Call GetError() to get a description of the error
  
  int (*ReadStream)(fmHandle strHandle, ui8 *buf, unsigned int size);  
  // Reads size program stream bytes for the given opened stream.  
  //  Returns the actual number of bytes read.

  int (*SetStreamPos)(fmHandle strHandle, ui64 pos); 
  // Set the position of the stream if the module supports FM_MISM_READFILES feature
  // Return FM_MISM_ERROR otherwise.

  int (*GetStreamPos)(fmHandle strHandle, ui64 *pos);  
  // Get the position of the stream if the module supports FM_MISM_READFILES feature
  // Return FM_MISM_ERROR otherwise.

  int (*GetStreams)(fmStreamIds *ids, char *fileName);
  // This function is called by FlasKMPEG to know what streams should be used.
  // Right now only one video stream and one audio stream are allowed. Fill TStreamIds accordingly.
  // fileName is the file from we want to retrieve the streams from

  int (*CloseStream)( fmHandle strHandle ); // Close stream

  char* (*GetLastError)(void); // Call this to get a description of Last error
  char* (*GetFileName)( fmHandle strHandle ); // Gets the name of the opened file. Returns NULL if no file is opened.

  // Stats info ptrs
  char* (*GetStreamStatus)( fmHandle strHandle ); // Gets a textual description of the status of the module
  ui64 (*GetStreamSize)( fmHandle strHandle );    // total size of the stream if the module supports FM_MISM_READFILES

  void (*Init)( LPFMMismStdParms parms );
  void (*DeInit)( void );

} FMMismInfo,*LPFMMismInfo;

// The library must export this function
// void fmGetMism(LPFMMismInfo mismInfo);
typedef void (*fmGetMismPtr)(LPFMMismInfo pMismInfo);

#endif