/* 
 *  AudioProperties.cpp 
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "FlasKMPEG.h"
#include "auxiliary.h"
#include "resource.h"
#include "Player.h"
#include "AudioProperties.h"
#include "./RunState.h"
#include "./Audio/AudioPlayback.h"
#include "./Misc/StatsStrings.h"
#include <string>

// Extern globals
extern TConfig          o;
extern TRunState        rs;
extern HINSTANCE        hInst;
extern HWND				hMainWnd;

// Local variables
static HWND                 hPlayerDlg;
static TAudioProperties ap;

const int nTrackBarRange = 1000;


// Create worker thread for the player
class CAudioThread : public CFlThread
{
public:
    //Constructor
    CAudioThread::CAudioThread(){
      m_nMaxSample = 0;
    };

    void Normalize()
    {
      SendCommand(normalize, 0);
    }
    void Seek(ui64 nSeekPos)
    {
      SendCommand(seek, nSeekPos);
    }
    void Play()
    {
      SendCommand(play, 0);
    }
    void TrackChange(TAudioTrack *pTrack)
    {
      SendCommand(trackChange, (ui64)pTrack);
    }
    void Stop()
    {
      SendCommand(stop, 0);
    }
    ui32 GetState(){ return m_nState; }
    ui32 GetNormalizeValue()
    {
      if( m_nMaxSample==0 )
        return 100;

      ui32 ret = (32767*100)/m_nMaxSample;
      // If we get a 100% of normalization value
      // we're finished
      if(ret<=100)
        SendCommand(stop, 0);
      return ret;
    }
    ui16 FindMax(i16 *data, int nDataSize)
    {
      i16 mono;
      ui32 nSamples = nDataSize>>2;
      ui16 max=0;
      for(int i=0; i<nSamples; i++)
      {
        mono = (data[2*i] + data[2*i+1])/2;
        if( ABS(mono) >= max )
          max = ABS(mono);
      }
      return max;
    }
    enum Commands { seek=THREAD_COMMAND_COUNT, play, stop, normalize, trackChange};
    enum State { stStopped=0, stPlaying, stNormalizing };
private: 
    ui16 m_nMaxSample;
    State m_nState;
    // Actual running thread
    DWORD ThreadProc();


};

// Actual running thread
DWORD CAudioThread::ThreadProc()
{
  m_nState = stStopped;
  m_nCommand = play;


  TAudioPlayback sAudioPlayback;
  unsigned char *data;
  sAudioPlayback.n_channels = 2;
  if (rs.prof.sampleFreqSameAsInput)
    sAudioPlayback.sample_freq = rs.audio->sampleRate;
  else
    sAudioPlayback.sample_freq = rs.prof.outSampleFrequency; 
  sAudioPlayback.chunk_size = 4096*6;
  AudioPlaybackStart(&sAudioPlayback);
  
  while(1)
  {
    
    // open scope to lock command
    {
      CFlAutoLock commandLock(&m_csCommand);
      switch( m_nCommand )
      {
      case noCommand:
        break;
      case seek:
        if(m_nState!=stStopped)
          rs.audio->Stop();
        rs.audio->SetStreamPos(m_nParam);
        rs.audio->Start(sAudioPlayback.sample_freq, DO_AUDIO|AUDIO_PREVIEW, &ap, &rs.sAudioTrack);
        m_nState = stPlaying;
        break;
      case play:
        if(m_nState!=stStopped)
          rs.audio->Stop();
        rs.audio->Start(sAudioPlayback.sample_freq, DO_AUDIO|AUDIO_PREVIEW, &ap, &rs.sAudioTrack);
        m_nState = stPlaying;
        break;
      case trackChange:
        {
          TAudioTrack *pTrack = (TAudioTrack *)m_nParam;
          rs.audio->SetTrack(pTrack);
        }
        break;
      case stop:
        if(m_nState!=stStopped)
          rs.audio->Stop();
        m_nState = stStopped;        
        break;
      case normalize:
        if(m_nState!=stStopped)
          rs.audio->Stop();
        rs.audio->SetStreamPos(0);
        rs.audio->Start(sAudioPlayback.sample_freq, DO_AUDIO|AUDIO_PREVIEW|AUDIO_DONT_NORMALIZE, &ap, &rs.sAudioTrack );
        m_nState = stNormalizing;
        break;
      case exit:
        if(m_nState!=stStopped)
          rs.audio->Stop();
        goto end;
        break;
      }
    }
    EndCommand();

    // Perform action according the state of the playe
    
    
    switch(m_nState)
    {
    case stStopped:
      break;    
    case stPlaying:
      rs.audio->GetSamples(0, (short **)&data, (sAudioPlayback.chunk_size)>>2);
      AudioPlaybackWrite(&sAudioPlayback, data);
      break;      
    case stNormalizing:
      rs.audio->GetSamples(0, (short **)&data, (sAudioPlayback.chunk_size)>>2);
      ui16 max = FindMax((i16*)data, sAudioPlayback.chunk_size);
      if(  max >= m_nMaxSample )
        m_nMaxSample = max;
      break;
    }
    
    // If we are stopped, wait for a command
    if( m_nState == stStopped )
      WaitCommand();
  }

end:
  
  AudioPlaybackStop(&sAudioPlayback);
  SignalExit();
  
  // Exit
  return 0;
}


void RenderMCSlider(HWND hDlg, int sliderID, int sliderDB, int value)
{
  char szTemp[256];

  SendDlgItemMessage(hDlg, sliderID, TBM_SETRANGE, 
    (WPARAM) TRUE, (LPARAM) MAKELONG(0, TAUDIOPROPERTIES_MCHANNEL_MAX_RANGE*2));


  sprintf(szTemp, "%s%1.2f dB", value>0?"+":"", ((float)(value)/(float)100));
  DlgSetText( hDlg, sliderDB, szTemp );

  value += TAUDIOPROPERTIES_MCHANNEL_MAX_RANGE;
  value = TAUDIOPROPERTIES_MCHANNEL_MAX_RANGE*2 - value;
  // Set the track position
  SendDlgItemMessage(hDlg, sliderID, TBM_SETPOS, (WPARAM) (BOOL) true,(LPARAM) (LONG) value);
}

// Activate, de-activate sliders
void RenderSliders(HWND hDlg, TAudioProperties *pAudProps)
{
  DlgCheck(hDlg, IDC_DRCCHECK, pAudProps->drc );
  DlgEnable(hDlg, IDC_DRCSLIDER, pAudProps->drc);

  DlgCheck(hDlg, IDC_NORMCHECK, pAudProps->normalize );
  DlgEnable(hDlg, IDC_NORMSEARCH, pAudProps->normalize);
  DlgEnable(hDlg, IDC_NORMEDIT, pAudProps->normalize);
  DlgSetText(hDlg, IDC_NORMEDIT, pAudProps->normalize_value );
  
  DlgCheck(hDlg, IDC_DOLBYDOWNMIX, pAudProps->dolby_surround_downmix );
  DlgCheck(hDlg, IDC_MCCHECK, pAudProps->multichannel_volume );
  DlgEnable(hDlg, IDC_MCCENTER, pAudProps->multichannel_volume);
  DlgEnable(hDlg, IDC_MCFRONT, pAudProps->multichannel_volume);
  DlgEnable(hDlg, IDC_MCREAR, pAudProps->multichannel_volume);
  DlgEnable(hDlg, IDC_MCRESET, pAudProps->multichannel_volume);
  // Set range of mc sliders
  RenderMCSlider(hDlg, IDC_MCFRONT, IDC_FRONTDB, pAudProps->front);
  RenderMCSlider(hDlg, IDC_MCCENTER, IDC_CENTERDB, pAudProps->center);
  RenderMCSlider(hDlg, IDC_MCREAR, IDC_REARDB, pAudProps->rear);
  // Now set the value

  // DRC
  SendDlgItemMessage(hDlg, IDC_DRCSLIDER, TBM_SETRANGE, 
    (WPARAM) TRUE, (LPARAM) MAKELONG(0, COMPRESSOR_STEPS));

  SendDlgItemMessage(hDlg, IDC_DRCSLIDER, TBM_SETPOS, 
    (WPARAM) (BOOL) true,
    (LPARAM) (LONG) COMPRESSOR_STEPS - pAudProps->drc_value);
  // Finally set the values to the audio renderer
  rs.audio->SetAudioProperties(pAudProps);
}

void EndAudioProperties( HWND hDlg, CAudioThread *pAudioThread )
{
  KillTimer(hDlg, 2);
  pAudioThread->Exit();
  delete pAudioThread;
  return;
}
LRESULT CALLBACK AudioPropertiesDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  static CAudioThread *pAudioThread;
  static DWORD lPosition;
  static ui64 nFilePos;
  static bool userIsTracking;
  static int  nFirstAudioId;
  TStatsStrings st;
  int i;

	switch (message) 
	{
		case WM_TIMER:
      {
        if( pAudioThread->GetState()==CAudioThread::stNormalizing )
          DlgSetText( hDlg, IDC_NORMEDIT, pAudioThread->GetNormalizeValue() );

        lPosition= (int)(((double)(i64)(rs.audio->GetStreamPos())/(double)(i64)(rs.audio->GetStreamSize()))*nTrackBarRange);
        if( !userIsTracking  && pAudioThread->GetState()!=CAudioThread::stNormalizing)
          SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_SETPOS, (WPARAM) (BOOL) true,(LPARAM) (LONG) lPosition); 

        if( pAudioThread->GetState()==CAudioThread::stNormalizing )
          SendDlgItemMessage(hDlg, IDC_NORMPROGRESS, PBM_SETPOS, (WPARAM) (LONG) lPosition, 0 ); 
        else
          SendDlgItemMessage(hDlg, IDC_NORMPROGRESS, PBM_SETPOS, (WPARAM)  0, 0); 

        // Put text in audio properties
        RenderStatsStrings(&st);
        DlgSetText( hDlg, IDC_PROPSFORMAT, st.InMedia.audio_format );
        DlgSetText( hDlg, IDC_PROPSBITRATE, st.InMedia.audio_bit_rate );
        DlgSetText( hDlg, IDC_PROPSMODE, st.InMedia.audio_channels );
        DlgSetText( hDlg, IDC_PROPSSAMPLEFREQ, st.InMedia.audio_sample_freq );
      }
			break;
		case WM_INITDIALOG:
      // First of all check to see that in fact there is audio here
      if(!rs.audio)
        return FALSE;

      ap = rs.prof.sAudioProperties;
      RenderSliders(hDlg, &ap);

      // Range
      SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_SETRANGE, (WPARAM) TRUE,(LPARAM) MAKELONG(0, nTrackBarRange)); 
      SendDlgItemMessage(hDlg, IDC_NORMPROGRESS, PBM_SETRANGE, (WPARAM) TRUE,(LPARAM) MAKELONG(0, nTrackBarRange)); 


      // Fill tracks list from working mism
      fmStreamId strId;
      GetFirstAudioId(&rs.sStreamIds, &strId, &nFirstAudioId);
      for(i=0; i<rs.sStreamIds.idCount; i++)
      {
        if( MismIsAudioId(&rs.sStreamIds.vIds[i]) )
          ListAddText(GetDlgItem(hDlg, IDC_TRACKS), rs.sStreamIds.vIds[i].streamName);
      }
      // select the default one
      ListSetCur(GetDlgItem(hDlg, IDC_TRACKS), rs.sStreamIds.idAudioDef - nFirstAudioId);
      

      // Create timer
      SetTimer(hDlg, 2, 500,  NULL );

      // Create thread
      pAudioThread = new CAudioThread;
      pAudioThread->Create();
      pAudioThread->Play();
		  return TRUE;
    case WM_HSCROLL:
      int nPos;
      switch(LOWORD(wParam)){
      case TB_THUMBTRACK:
        // Get position
        // No need to serialize this.
        userIsTracking = true;        
        break;
      case TB_THUMBPOSITION :
        nPos = SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_GETPOS, 0, 0);   
        
        nFilePos=(i64)(((double)nPos/1000.0)*(double)(i64)(rs.audio->GetStreamSize()));
        
        // Seek the player to that position
        pAudioThread->Seek( nFilePos );
        userIsTracking = false;
        break;
      }
      break;
    case WM_VSCROLL:
      {
        int range = TAUDIOPROPERTIES_MCHANNEL_MAX_RANGE;
        ap.rear   = -(SendDlgItemMessage(hDlg, IDC_MCREAR, TBM_GETPOS, 0, 0 ) - range);
        ap.front  = -(SendDlgItemMessage(hDlg, IDC_MCFRONT, TBM_GETPOS, 0, 0 ) - range);
        ap.center = -(SendDlgItemMessage(hDlg, IDC_MCCENTER, TBM_GETPOS, 0, 0 ) - range);

        ap.drc_value = COMPRESSOR_STEPS - SendDlgItemMessage(hDlg, IDC_DRCSLIDER, TBM_GETPOS, 0, 0 );

        RenderSliders(hDlg, &ap);
      }
      break;
		case WM_COMMAND:
      switch( HIWORD(wParam))
      {
      case EN_UPDATE:
        ap.normalize_value = DlgGetText(hDlg, IDC_NORMEDIT);
        rs.audio->SetAudioProperties( &ap );
        break;
      case CBN_SELCHANGE:
        {
          int nSelected = ListGetCur(GetDlgItem(hDlg, IDC_TRACKS));
          rs.sAudioTrack.nStreamId = rs.sStreamIds.vIds[nSelected+nFirstAudioId].streamId;
          rs.sAudioTrack.nSubStreamId = rs.sStreamIds.vIds[nSelected+nFirstAudioId].subStreamId;
          pAudioThread->TrackChange(&rs.sAudioTrack);
        }
        break;
      }
			switch( LOWORD(wParam))
      {
      case IDOK:
        EndAudioProperties( hDlg, pAudioThread );
        rs.prof.sAudioProperties = ap;
        EndDialog(hDlg, LOWORD(wParam));
        break;
      case IDCANCEL:
        EndAudioProperties( hDlg, pAudioThread );
        EndDialog(hDlg, LOWORD(wParam));
        break;

      case IDC_DRCCHECK:
        ap.drc = DlgIsChecked(hDlg, IDC_DRCCHECK);
        RenderSliders(hDlg, &ap);
        break;
      case IDC_NORMCHECK:
        ap.normalize = DlgIsChecked(hDlg, IDC_NORMCHECK);
        RenderSliders(hDlg, &ap);
        break;
      case IDC_MCCHECK:
        ap.multichannel_volume = DlgIsChecked(hDlg, IDC_MCCHECK);
        RenderSliders(hDlg, &ap);
        break;
      case IDC_MCRESET:
        ap.front = ap.center = ap.rear = 0;
        RenderSliders(hDlg, &ap);
        break;
      case IDC_DOLBYDOWNMIX:
        ap.dolby_surround_downmix = DlgIsChecked(hDlg, IDC_DOLBYDOWNMIX);
        RenderSliders(hDlg, &ap);
        break;
      case IDC_NORMSEARCH:
        pAudioThread->Normalize();
        break;
      case IDC_SLIDER:
        break;
      case IDC_PLAY:
        pAudioThread->Play();
        break;
      case IDC_STOP:
        pAudioThread->Stop();
        break;
      }
    break;
	}
  return FALSE;
}

void ShowAudioProperties()
{
    DialogBox(hInst,MAKEINTRESOURCE(IDD_AUDIOPROPERTIES), hMainWnd, (DLGPROC) AudioPropertiesDlg);
}
