/* 
 *  VideoRenderer.cpp
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#define INITGUID
#include "VideoRenderer.h"
#include <math.h>
#include <stdlib.h>

#define PI 3.141592

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CVideoRenderer::CVideoRenderer(HWND hRenderWnd): m_oBackground(NULL), m_oPausedFrame(NULL)
{
  m_hRenderWnd = hRenderWnd;
  m_nWidth = 720;
  m_nHeiht = 576;
  fTime = 0;
  m_bDDrawPresent = OpenDirectDraw();
  m_bDDCanUseYuvOverlay = IsYUVOverlayAvailable();
  m_bDDCanUseRgbOverlay = IsRGBOverlayAvailable();
  StopPlaying();
}

CVideoRenderer::~CVideoRenderer()
{
  CloseOverlay();
  CloseDirectDraw();
}

bool CVideoRenderer::OpenDirectDraw()
{
  bool bSuccess = false;
  if (DirectDrawCreate(NULL, &m_pDD, NULL)==DD_OK)
  {
    // Ask for interface IID_IDirectDraw2
    if (m_pDD->QueryInterface(IID_IDirectDraw2, (LPVOID*)&m_pDD2)==DD_OK)
    {
      if (m_pDD2->SetCooperativeLevel(m_hRenderWnd, DDSCL_NORMAL)==DD_OK)
        bSuccess = true;
      else
      {
        m_pDD2->Release();
        m_pDD->Release();        
      }
    }
    else
      m_pDD->Release();
  }
  return bSuccess;
}

void CVideoRenderer::CloseDirectDraw()
{
  if(m_pDD) SafeRelease(m_pDD);
  if(m_pDD2) SafeRelease(m_pDD2);
  if(m_pDDPrimary) SafeRelease(m_pDDPrimary);
  if(m_pDDOverlay) SafeRelease(m_pDDOverlay);
}

bool CVideoRenderer::IsYUVOverlayAvailable()
{
  bool bSuccess = OpenOverlay(YuvMode);
  CloseOverlay();
  return bSuccess;
}

bool CVideoRenderer::IsRGBOverlayAvailable()
{
  bool bSuccess = OpenOverlay(RgbMode);
  CloseOverlay();
  return bSuccess;
}

bool CVideoRenderer::OpenOverlay(OverlayMode eOverlayMode)
{
  bool bSuccess = false;

  if(m_bDDrawPresent)
  {
    // Create primary surface
    ZeroMemory(&m_sDDPrimaryDesc, sizeof(DDSURFACEDESC));
    m_sDDPrimaryDesc.dwSize = sizeof(DDSURFACEDESC);
    m_sDDPrimaryDesc.dwFlags = DDSD_CAPS;
    m_sDDPrimaryDesc.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_VIDEOMEMORY;
    
    if (m_pDD2->CreateSurface(&m_sDDPrimaryDesc, &m_pDDPrimary, NULL)==DD_OK)
    {
      // Now check for overlay support
      ZeroMemory(&m_sDDHalCaps, sizeof(DDCAPS));
      m_sDDHalCaps.dwSize = sizeof(DDCAPS);
      
      if (m_pDD2->GetCaps(&m_sDDHalCaps, NULL)==DD_OK)
      {
        if (m_sDDHalCaps.dwCaps & DDCAPS_OVERLAY)
        {
          // Try to create overlay
          DDPIXELFORMAT ddPixelFormat;
          ZeroMemory( &ddPixelFormat, sizeof(DDPIXELFORMAT) );
          ddPixelFormat.dwSize        = sizeof(DDPIXELFORMAT);
          switch(eOverlayMode)
          {
          case YuvMode:
            ddPixelFormat.dwFlags       = DDPF_FOURCC;
            ddPixelFormat.dwFourCC      = mmioFOURCC('Y','U','Y','2');
            ddPixelFormat.dwYUVBitCount = 16;            
            break;
          case RgbMode:
            ddPixelFormat.dwFlags       = DDPF_RGB;
            ddPixelFormat.dwRGBBitCount = 32;
            ddPixelFormat.dwRBitMask  = 0x00FF0000;
            ddPixelFormat.dwGBitMask  = 0x0000FF00;
            ddPixelFormat.dwBBitMask  = 0x000000FF;
            break;
          }
          
          ZeroMemory(&m_sDDOverlayDesc, sizeof(DDSURFACEDESC));
          m_sDDOverlayDesc.dwSize = sizeof(DDSURFACEDESC);
          m_sDDOverlayDesc.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT;
          m_sDDOverlayDesc.ddsCaps.dwCaps = DDSCAPS_OVERLAY | DDSCAPS_VIDEOMEMORY;
          m_sDDOverlayDesc.dwWidth = m_nWidth;
          m_sDDOverlayDesc.dwHeight = m_nHeiht;
          
          memcpy(&(m_sDDOverlayDesc.ddpfPixelFormat), &ddPixelFormat, sizeof(DDPIXELFORMAT));
          
          if (m_pDD2->CreateSurface(&m_sDDOverlayDesc, &m_pDDOverlay, NULL)==DD_OK)
          {
            bSuccess = true;
            /*
            ZeroMemory(&ddofx, sizeof(DDOVERLAYFX));
            ddofx.dwSize = sizeof(DDOVERLAYFX);
            
            ddofx.dckDestColorkey.dwColorSpaceLowValue = DDColorMatch(lpPrimary, MASKCOLOR);
            ddofx.dckDestColorkey.dwColorSpaceHighValue = ddofx.dckDestColorkey.dwColorSpaceLowValue;
            */
          }
        }
      }// get caps
    }// create primary
  }

  if(!bSuccess)
  {
    if(m_pDDOverlay) SafeRelease(m_pDDOverlay);
    if(m_pDDPrimary) SafeRelease(m_pDDPrimary);
  }
  return bSuccess;
}
void CVideoRenderer::CloseOverlay()
{
  if(m_pDDOverlay) SafeRelease(m_pDDOverlay);
  if(m_pDDPrimary) SafeRelease(m_pDDPrimary);
}

void CVideoRenderer::StepBackground()
{
  int nWidth = m_oBackground.GetWidth();
  int nHeight = m_oBackground.GetHeigth();
  ui32 *pBuffer = (ui32*)m_oBackground.GetBuffer();

  // Only 32bpp!
  ui32  valueR=0xE5, valueG=0xBD, valueB=0x45;
  fTime += 0.001;
  double fIntime = fTime;
  // Randomize phases
  fPhase1 = 1;
  fPhase2 = 1;
  // Make one line
  for(int i=0; i<nWidth; i++)
  {//326DA8
    valueR = (double)0x32 * (cos(2*PI*0.5*fIntime) + 1)/2;
    valueG = (double)0x6D * (cos(2*PI*0.5*fIntime) + 1)/2;
    valueB = (double)0xA8 * (cos(2*PI*0.5*fIntime) + 1)/2;
    fIntime += 0.01;
    pBuffer[i] = valueR<<16 | valueG<< 8 | valueB<<0;
  }

  // copy to the other lines
  for( i=1; i<nHeight; i++)
    for( int j=0; j<nWidth; j++ )
      pBuffer[i*nWidth + j] = pBuffer[j];
    
}

void CVideoRenderer::SetBackground(CFrame *pFrame)
{

  if(!pFrame)
    return; 
  if(!pFrame->IsValid())
    return;
  m_oBackground = *pFrame;
}

void CVideoRenderer::Update()
{
  switch(m_nState)
  {
  case Stopped:
    Draw(&m_oBackground);
    break;
  case Paused:
    Draw(&m_oPausedFrame);
    break;
  }
}
void CVideoRenderer::Draw(CFrame *pFrame)
{
  HDC hDC;

  if(!pFrame)
    return;

  if(!pFrame->IsValid())
    return;

  switch( pFrame->GetFormat() )
  {
    case FRAME_RGB:
      hDC = GetDC(m_hRenderWnd);
      SetDIBitsToDevice((HDC) hDC,0,0,pFrame->GetWidth(),pFrame->GetHeigth(),
        0,0,0,pFrame->GetHeigth(), pFrame->GetBuffer(), pFrame->GetBmpInfo(), DIB_RGB_COLORS);
      ReleaseDC( m_hRenderWnd, (HDC)hDC);
      break;
  }
}