/* 
 *  VideoWrapper.cpp 
 *
 *	Copyright (C) Alberto Vigata - January 2000  ultraflask@yahoo.com
 *                 
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef VIDEOWRAPPER_H
#define VIDEOWRAPPER_H


// Standard includes
#include "..\Misc\CArray.h"
#include "..\Demux\Demux.h"
#include "..\thread.h"
#include "..\framesource.h"
#include "..\YUVtoRGB.h"
#include "..\windebug.h"
#include "idctapi.h"


#define DECODER_READ_BUFFER	2048
#define SEQ_FINISHED        234
#define SEQ_STARTED         235
#define SEQ_FIRSTTIME		236

//Errors returned in GetError()
#define END_OF_STREAM    1
#define PLAYER_STOPPED   2
				
// Static size for frame buffers
#define MEM_Y  1024*1024
#define MEM_C  1024*1024

#define FREE         0
#define FULL         3

#define MINUS        4
#define EQUAL        5
#define PLUS         6


// Buffer used to
// reconstruct field
// information
struct TRecImage 
{
	YUVImageTag im;
	int      state;
	i64      PTS;
};

struct TDoubleBuffer
{
	TRecImage *rec[2];
};


// Timecodes
typedef struct timeCodeTag
{
	int hour;
	int minute;
	int second;
	int frame;
} TTimeCode;

struct presInfo
{
	i64  imagePTS;
	i64  imageSCR;
};

// Main picture information
// used in the reconstruction
// process
struct TFrameInfo
{
	i64 PTS;
	i64 SCR;
  int picture_coding_type;
	int progressive_frame;
	int repeat_first_field;
	int picture_structure;
	int top_field_first;
};

// This structure stores
// the timestamps for the
// images in the buffers.
struct presTimes
{
	TFrameInfo actual;
	TFrameInfo image;
	TFrameInfo forward;
	TFrameInfo backward;
};

// Init structure passed to Init()
struct TVideoInit
{
  LPTWorkingMism pMismInfo;
  int nStreamId;
  int nSubStreamId;	
  int nSubpicStreamId;
  int nSubpicSubstreamId;
  unsigned char (*clut)[16][4];
  char *ProgramDirectory;
};

// TVideo Info
// stores basic information of the current stream
struct TMPGVideoInfo
{
    int isMPEG2;
    int width;
    int height;
    int aspect_ratio_information;
    int progressive_frame;      // structure of current frame
    int progressive_sequence;
    int bit_rate_value;
    int frame_rate_code;
    int detected_frame_rate_code;
};

// Options for the decoder. Passed through Start().
struct TVideoOptions
{
	int   idctIndex;
	bool  recons_progressive;
  ui64  nSyncPoint;
  ui64  nEndPoint;
  bool  bStartInSync;
  CFrameBuffer *pFrameBuffer; // Frame buffer from where to grab data when using GetFrame(CFrame **)
};

// Search defines
#define SEARCH_I          0x01
#define SEARCH_B          0x02
#define SEARCH_P          0x04
#define SEARCH_BACKWARDS  0x08
#define SEARCH_FORWARD    0x10
#define SEARCH_ALIGN_FIRST_FIELD 0x20

#define GetSearchType(x) (x>>16)
#define SetSearchType(x, type) {x &= 0x0000FFFF; x |= type<<16;}

// DecodePicture defines
// Params
#define DECODING_PICTURE    0x01
#define SKIP_REF_SEARCH     0x02
#define DONT_UPDATE_REF     0x04
#define CONTINOUS_READING   0x08
// Decex
#define DECEX_NEXT_PICTURE  0x01
#define DECEX_PREV_PICTURE  0x02
#define DECEX_JUST_IPICTURE 0x04
#define DECEX_FORCEDECODING 0x08

// Return values
#define DEC_OK              0x00
#define DEC_EOS             0x01
#define DEC_NOTENOUGHREF    0x02 //not enough references to decode this picture.


// The VideoWrapper is derived from CDemux
// from which it gets demuxing capabilites from 
// mpeg program streams.
class VideoWrapper: public CDemux, public CFrameSource
{
public:
    //-------------------------------------------------------------------
    // Public methods
    //-------------------------------------------------------------------
    
    // Constructor & desctructor.
    VideoWrapper();
    ~VideoWrapper();
    
    // Retrieves video stream information
    TMPGVideoInfo * GetVideoInfo();

    // Gets an error code when the GetFrame returns 0.
    int GetError();
    
    // After constructing the object, this method must be called
    // It initializes the object from the given stream.
    int Init(TVideoInit *pInitParams);
    
    // These two functions control the state of the decoder.
    // Anytime an operation has to be performed the following
    // sequence of calls must be performed.
    // Start() => Command() => Stop()
    // A Command() is those methods that change the state of the decoder like GetFrame()
    // SetStreamPos() and so on.
    int		Start(TVideoOptions *opt);
    int		Stop();
    
    // These two are overrides of the methods from CDemux.
    // They are able to seek to a given specified byte even
    // in multiplexed streams.
    bool SetStreamPos(ui64 pos);
    ui64 GetStreamPos();
    ui64 GetLastDecodedFrameStart(){ return m_nLastDecodedPictureStart; };
    
    // Retrieves a frame of video from the current position of the stream.
    // Call Start() before this. pInfo contains the timestamp of when this
    // frame has to be displayed. All frames contain presentation information.
    int	VideoWrapper::GetFrame(presInfo *pInfo, YUVImageTag **frame);
    bool VideoWrapper::GetFrame(CFrame **pFrame);

    bool GetNextFrame(YUVImageTag **ppFrame);
    bool GetNextFrame(CFrame **pFrame);

    bool GetPrevFrame(YUVImageTag **ppFrame);    
    bool GetPrevFrame(CFrame **pFrame);

    bool GetNextIFrame(YUVImageTag **ppFrame);
    bool GetNextIFrame(CFrame **pFrame);

    bool GetPrevIFrame(YUVImageTag **ppFrame);
    bool GetPrevIFrame(CFrame **pFrame);
    
    // Stub for video decoder
    // ppBuffer will store the position of the buffer read
    // pBufferSize the size of the buffer at ppBuffer
    bool ReadVideoData( ui8** ppBuffer, ui32* pBufferSize );
    
    // Searchs a picture header.
    bool SearchPictureHeader( int &nParams );
    
    // IDCT routines. These are initialized from the iDCT module.
    void (*Idct)(short *block);
    void (*InitIdct)(void);
    void (*DeInitIdct)(void);
    // Retrieve number of iDCTs
    int GetIdctCount(){ return m_vIdct.GetCount(); };
    // Return a pointer to iDCT info
    LPFMpegIdctInfo GetIdctInfo( int idx ){ 
        if(idx<0 || idx>=m_vIdct.GetCount()) 
            return NULL;
        return &m_vIdct[idx];
    };
    int GetIdctSelectedIndex(){ return m_nSelectedIdct; }
    // Select the default idct. Returns iDCT index.
    int SelectIdctDefault()
    {
      IdctSpeedTest();
      return m_nSelectedIdct;
    }
    // Select and iDCT
    int SelectIdct( int idx )
    { 
        if(idx<0 || idx>=m_vIdct.GetCount()) 
        {
          // This iDCT doesnt exist
          // Do the speed test and select the fastest one
          IdctSpeedTest();
          idx = m_nSelectedIdct;
        }

        Idct       = m_vIdct[idx].Idct;
        InitIdct   = m_vIdct[idx].InitIdct;
        DeInitIdct = m_vIdct[idx].DeInitIdct;

        return m_nSelectedIdct = idx;
    };
    
    //-------------------------------------------------------------------
    // Public variables
    //-------------------------------------------------------------------
    
    // Store the streamID and subStreamID of the video
    int  subStreamID, streamID;
    // Store the stream ids of a subpicture stream
    int  subpic_substreamID, subpic_streamID; /* -1 = off */
    
    
    // Used in detectFrameRate to flag the parsing of Gop
    bool gopFlag;
    
    // presTimes stores all the presentation information of all
    // the reference frames and present images that have not 
    // been output yet
    presTimes	p;
    
    
    // Basic settings that should
    // be private. 
    double frameDelay;
    double frameRate;
    double DAR;
    int	 pictureWidth;
    int	 pictureHeight;
    int  detectedFrameRateCode;

    // We use this variable to store the
    // output of the decoder before reconstruction
    YUVimage YUV;
    
    // Flag this variable to make the decoder
    // stop.
    bool    stopDecoding;
    
    // Timecode of the present video
    TTimeCode		timeCode;
    
    // information of the PES currently being read
    PESinfo	myPES;
    
    // Tracks the time embedded in the stream in miliseconds.
    ui32 time;

    // This variable is flagged when the end of the stream is reached.
    bool m_bReadError;

private:

    //-------------------------------------------------------------------
    // Private methods and variables
    //-------------------------------------------------------------------
    
    bool GetBackByte(ui8 *byte);
    bool GetFordByte(ui8 *byte);
    bool GetFordDWord(ui32 *dword);
    bool ExitCoreContext();
    bool EnterCoreContext();
    int  DecodePicture(int nDecParams=0);
    int  DecodePictureEx( int nDecExParams );
    void FlushBuffer();
    //
    ui64 m_nEndPoint;
    CFrameBuffer *m_pFrameBuffer;
    CFrame *m_pFrontFrame, *m_pBackFrame;
    CFrame  m_oUtilFrame, m_oYuvFrame;
    TYUVImage sYuvFrame;
    // Temporal frame buffers for reconstruction
    unsigned char *m_pU422;
    unsigned char *m_pV422;
    unsigned char *m_pU444;
    unsigned char *m_pV444;
    bool m_bFronFrameIsReady;
    void ConvertTo444();
    void AllocateReconBuffers()
    {
      m_pU422 = new BYTE[pictureWidth*pictureHeight];
      m_pV422 = new BYTE[pictureWidth*pictureHeight];
      m_pU444 = new BYTE[pictureWidth*pictureHeight];
      m_pV444 = new BYTE[pictureWidth*pictureHeight];
    }
    void DeAllocateReconBuffers()
    {
      delete []m_pU422;
      delete []m_pV422;
      delete []m_pU444;
      delete []m_pV444;
    }
    

    // Used in ReadVideoData.
    bool m_bReadSuccess;      
    __int64			 internalPTS;
    
    
    
    // 
    ui64 m_nLastDecodedPictureStart;
    bool m_bSyncedDecoding;
    bool m_bSyncedNotEnoughReferences;

    bool m_bPreviousDirWasForward;
    bool m_bFirstPictureDecoded;
    bool m_bSequenceInitialized;

    // This flag keeps track of the polarity of the output video
    bool m_bTopFieldExpected;

    i64 get_time(int field);
    void swap_db();
    TRecImage  reca;
    TRecImage  recb;
    int        rec_first;
    TDoubleBuffer dbuf;
    
    TMPGVideoInfo video_info;
    // Clut
    unsigned char (*pClut)[16][4];

    // IDCT function array
    CArr<FMpegIdctInfo> m_vIdct;
    void LoadIdctModule( char *szTemp );
    bool LoadIDCTs();
    bool IdctSpeedTest();
    void IdctSelectDefault();
    int  m_nSelectedIdct;


    char program_directory[MAX_PATH];
    HINSTANCE m_hModule;
    
    //Pull down stuff
    bool     get_frame();
    int     guess_32pulldown();
    void    adjust_pulldown_timing();
    i64     interlaced_field_delay;
    i64     progressive_frame_delay;
    int     pulldown_state;
    bool    pullDownDetected;
    bool    recons_progressive;
    
    double  detectedFrameDelay;
    
    bool Is24Progressive();
    int detectFrameRate();
    double detectedFrameRate;
    
    void updateVideoStats();
    int error;
    
    bool			StartingDecoder;
    bool			firstReadTime;
    bool			stopDecoder;
    bool		    decoderStopped;
    int Return_Value;
    int				Bitstream_Framenum;
    int				Sequence_Framenum;
    int				sequencePos;
    char			startBuffer[DECODER_READ_BUFFER];
    
    // variables for video buffer control
    // Indicates where the pointer of the file remains.
    // Either at the end of the PES, at the beginning or nowhere
    enum { PesBegin = 0,
        PesEnd,
        PesNotDefined } m_nPesPosState;
    
    
    // Only valid when m_bTempVideoBufferReadBack is true.
    // Indicates the end position of Pes that was read backwards.
    ui64  m_nEndPesPosition;
    ui8  *m_pTempVideoBuffer;
    i32  m_nTempVideoBufferPtr;
    ui32  m_nTempVideoBufferSize;
    
    // Critical section to synchronize access from different threads
    CCritSec m_csLockSection;

};


#endif