/* 
 *  VideoWrapper.cpp 
 *
 *	Copyright (C) Alberto Vigata - January 2000  ultraflask@yahoo.com
 *                 
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "VideoWrapper.h"
#include "..\Subpic\Subpic.h"
#include "..\cpusupport.h"


extern "C"
{ 
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>
#include <assert.h>

#define GLOBAL
#include "config.h"
#include "global.h"
//#include "getbits.h"
}


#include "misc_tables.h"


#define ABS(x) ((x>0)? x:-x)

const i64 nJumpBackwards = 102400;

//Global variables for syncing with decoding thread
VideoWrapper *myVideo;
int			 gopPOS;
bool		 closedGOP;
bool		 synced=false;


// This function returns the position of the 
// stream in offset bytes from the buffer
// Defined in getbits.cpp
extern          int Get_Buffer_Pos();
extern          void Reset_Bits_Buffer();
extern unsigned int Set_Buffer_State( ui8 *buffer, ui32 offset, ui32 buffersize );
extern          void picture_data();
extern void Write_Frame(unsigned char *src[], int frame);

/* private prototypes */
static int  video_sequence _ANSI_ARGS_((int *framenum));
static int Decode_Bitstream _ANSI_ARGS_((void));
static void        Initialize_Sequence _ANSI_ARGS_((void));
       void Deinitialize_Sequence _ANSI_ARGS_((void));
static void Process_Options _ANSI_ARGS_((int argc, char *argv[]));
static void InitClip();
static void DeInitClip();

bool NextSequenceHeader(){

  ui32 nStartCode=Show_Bits(32);
  while (nStartCode!=0x000001B3  && nStartCode!=0x000001B7)
  { 
    Flush_Buffer(8);
    nStartCode = Show_Bits(32);
  }
  if(nStartCode!=0x000001B3)
    return false;
  else
    return true;
}


int init_recons(TRecImage *image, int xsize, int ysize)
{
	if(image){
		image->im.Y = (YUVPixel *)malloc(xsize*ysize);
		image->im.U = (YUVPixel *)malloc((xsize*ysize)>>2);
		image->im.V = (YUVPixel *)malloc((xsize*ysize)>>2);
		image->im.Yxsize = xsize;
		image->im.Yysize = ysize;
		image->im.Cxsize = xsize>>1;
		image->im.Cysize = ysize>>1;
		image->state = FREE;
		return 1;
	}
	else
		return 0;

}

void deinit_recons(TRecImage *image)
{
	if( image ){
		free(image->im.Y);
			image->im.Y = NULL;
		free(image->im.U);
			image->im.U = NULL;
		free(image->im.V);
			image->im.V = NULL;
	}
}

VideoWrapper::VideoWrapper(): m_oUtilFrame(NULL),m_oYuvFrame(NULL)
{
	time          = 0;
	error         = 0;
	pictureWidth  = 0;
	pictureHeight = 0;
	myVideo       = this;
	stopDecoding  = false;
  m_nLastDecodedPictureStart = 0;


  
  m_bSequenceInitialized = false;
  
  // Reset reconstruction structs
  memset( &reca, 0, sizeof reca );
  memset( &recb, 0, sizeof recb );
  
  Idct          = NULL;
  m_hModule     = NULL;

}

VideoWrapper::~VideoWrapper()
{

    if(m_bSequenceInitialized)
    {
        // Deallocating memory
        deinit_recons(&reca);
        deinit_recons(&recb);
        
        DeInitClip();
        Deinitialize_Sequence();
        DeAllocateReconBuffers();
    }
    if(m_hModule)
        FreeLibrary( m_hModule );
}

void VideoWrapper::LoadIdctModule( char *szTemp )
{
  FMpegIdct_PluginFunc iDctPlugFunc;
  FMpegIdctInfo        iDctInfo;
  fmGetPluginFuncPtr      getPluginFunc;

  long nSupportedExt = CPUGetSupportedExtensions();
  long nSupportedExtIdct = 0;

  if( m_hModule = LoadLibrary( szTemp ) )
  {
    if( getPluginFunc = (fmGetPluginFuncPtr)GetProcAddress( m_hModule, "fmGetPluginFunc" ) )
    {
      getPluginFunc( &iDctPlugFunc );
      if (iDctPlugFunc.GetIdct && iDctPlugFunc.GetIdctCount)
      {
        for( int i=0; i< iDctPlugFunc.GetIdctCount(); i++ )
        {
          iDctPlugFunc.GetIdct( i, &iDctInfo );
          // Check that the CPU supports this iDCT
          nSupportedExtIdct = 0;
          // Adapt flags from CPU support to iDCT api conventions
          nSupportedExtIdct |= nSupportedExt&CPU_SUPPORTS_MMX ? SUPPORTS_MMX : 0;
          nSupportedExtIdct |= nSupportedExt&CPU_SUPPORTS_SSE ? SUPPORTS_SSE : 0;
          nSupportedExtIdct |= nSupportedExt&CPU_SUPPORTS_SSE2 ? SUPPORTS_SSE2 : 0;
          nSupportedExtIdct |= nSupportedExt&CPU_SUPPORTS_3DNOW ? SUPPORTS_3DNOW : 0;
          nSupportedExtIdct |= nSupportedExt&CPU_SUPPORTS_3DNOW_EXT ? SUPPORTS_3DNOW_EXT : 0;
          nSupportedExtIdct &= PROCESSOR_SUPPORT_MASK;

          // If all the CPU features are supported
          if( (iDctInfo.dwFlags&PROCESSOR_SUPPORT_MASK) & nSupportedExtIdct)
            m_vIdct.AddItem( &iDctInfo );
          // If no special CPU features are required add it also
          else if( !(iDctInfo.dwFlags&PROCESSOR_SUPPORT_MASK) )
            m_vIdct.AddItem( &iDctInfo );
        }
      }
    }
  }

}

bool VideoWrapper::IdctSpeedTest()
{
  FMpegIdctInfo        iDctInfo;
  i64   nStartTime, nEndTime;
  i64   nMinTime = MAXLONGLONG;
  bool  bTimerError = false;

  for(int i=0; i<m_vIdct.GetCount(); i++)
  {
    iDctInfo = m_vIdct[i];

    void (* idct)(short *block) = iDctInfo.Idct;
    
    // Perform speed test
    if(iDctInfo.InitIdct)
      iDctInfo.InitIdct();
    
    CFrame frTestFrame(NULL);
    // This will allocate a chunk of aligned memory 
    // to 16bytes boundaries.
    frTestFrame.Set( 8, 8, FRAME_RGB, 32, 0);
    // Get the buffer
    short *block = (short *)frTestFrame.GetBuffer();

    QueryPerformanceCounter( (LARGE_INTEGER *)&nStartTime );
    for(int j=0; j<2500; j++)
      idct(block);
    QueryPerformanceCounter( (LARGE_INTEGER *)&nEndTime );

    if(iDctInfo.DeInitIdct)
      iDctInfo.DeInitIdct();    
    
    if(nEndTime==0)
    {
      bTimerError = true;
      break;
    }


    i64 nTestTime = nEndTime - nStartTime;
    if( nTestTime < nMinTime )
    {
      nMinTime = nTestTime;
      SelectIdct(i);
    }

  }
  // If we couldnt use QueryPerformanceCounter
  if(bTimerError)
  {
    // Look for one that supports MMX
    for(int i=0; i<m_vIdct.GetCount(); i++)
    {
      iDctInfo = m_vIdct[i];
      if(iDctInfo.dwFlags&SUPPORTS_MMX)
      {
        SelectIdct(i);
        break;
      }
    }
    if( i>=m_vIdct.GetCount() )
      SelectIdct(0);
  }
  return true;
}

bool VideoWrapper::LoadIDCTs()
{
  WIN32_FIND_DATA find_data;
  char     directory[MAX_PATH], szTemp[MAX_PATH];
  
  HANDLE   search_handle;
  int      i;
  
  
  sprintf(directory, "%s\\*.idct.flask", program_directory );
  
  m_vIdct.EmptyArray();
  
  i=0;
  search_handle = FindFirstFile(directory, &find_data);
  if(search_handle==INVALID_HANDLE_VALUE){
    m_vIdct.EmptyArray();
  }
  else{
    sprintf(szTemp,"%s\\%s", program_directory, find_data.cFileName );
    LoadIdctModule( szTemp );
    
    while( FindNextFile(search_handle, &find_data ) ){
      sprintf(szTemp,"%s\\%s", program_directory, find_data.cFileName );
      LoadIdctModule( szTemp );
    }
    FindClose(search_handle);
  }
  
  IdctSpeedTest();

  return m_vIdct.GetCount() > 0;
}

int VideoWrapper::Init(TVideoInit *pInitParams)
{
	int ret;

  strcpy( this->program_directory, pInitParams->ProgramDirectory );
  // Try to load iDCTs from files
  if(!LoadIDCTs())
    return 0;
  // testing

	VideoWrapper::streamID    = pInitParams->nStreamId;
	VideoWrapper::subStreamID = pInitParams->nSubStreamId;

	VideoWrapper::subpic_streamID    = pInitParams->nSubpicStreamId;
	VideoWrapper::subpic_substreamID = pInitParams->nSubpicSubstreamId;



	
	//Start demuxer
	if(!SetInput(pInitParams->pMismInfo))
		return 0;

	InitClip();               /* Creating clipping vector */

  //MSSG decoder Initialization
  ld               = &base; /* select base context */
  Frame_Store_Flag = 1;     /* store full frames */
  Output_Type=2;            /* YUV format output */
  System_Stream_Flag = 0;
  
  FlushBuffer();

	m_pTempVideoBuffer     = NULL;

	//GET  SEQUENCE PROPERTIES
	CDemux::SetStreamPos(0);
	StartReadLPES();
	Initialize_Buffer(); 
  if(!NextSequenceHeader())
    return 0;
  
	/* Headers returns when end of sequence (0) or picture
       header has been parsed (1) */
	ret = Get_Hdr();
	if(ret==1)
	{
	   //VIDEO SEQUENCE....
  	Sequence_Framenum=0;
		Initialize_Sequence();
		pictureWidth = Coded_Picture_Width;
		pictureHeight= Coded_Picture_Height;
		// Create output video buffer
		//   given that we already know
		//   the size of our video
		init_recons(&reca, pictureWidth, pictureHeight);
		init_recons(&recb, pictureWidth, pictureHeight);
    AllocateReconBuffers();
    m_bSequenceInitialized = true;
	}
  else
    return 0;

  // Now, guess if this is 24fps 
  detectedFrameRateCode = Is24Progressive() ? 1 : frame_rate_code;
  updateVideoStats();

  // FIXME: This should be changed from FRAME_RGB
  m_oUtilFrame.Set( pictureWidth, pictureHeight, FRAME_RGB, 32, 0);


	// Rewind stream
	CDemux::SetStreamPos(0);
	StartReadLPES();
	Initialize_Buffer(); 
	synced=true;
  
  // Store clut
  pClut = pInitParams->clut;

	return 1;
}

void VideoWrapper::FlushBuffer()
{
  m_pTempVideoBuffer     = NULL;
  m_nTempVideoBufferPtr  = 0;
  m_nTempVideoBufferSize = 0;
  m_nPesPosState         = PesNotDefined;
  Reset_Bits_Buffer();
}

int VideoWrapper::Start(TVideoOptions *opt)
{
  

   m_bFronFrameIsReady = false;
   m_pFrontFrame = m_pBackFrame = NULL;
   m_pFrameBuffer = opt->pFrameBuffer;

   m_oYuvFrame.SetFormat(FRAME_YUV444);
   m_oYuvFrame.SetBuffer( (ui8*)&sYuvFrame );
   m_oYuvFrame.SetSize( pictureWidth, pictureHeight );

   m_bSyncedNotEnoughReferences = false;
   m_bFirstPictureDecoded   = false;
   m_bPreviousDirWasForward = false;

	time=0;
	SelectIdct( opt->idctIndex ); 
  /* IDCT */
  if( myVideo->InitIdct )
    myVideo->InitIdct();

	m_bTopFieldExpected = true;

	//Decoder Configuration
	timeCode.frame=0;
	timeCode.hour=0;
	timeCode.minute=0;
	timeCode.second=0;
	stopDecoding=false;

  // Only allow recons_progressive flag if the frame rate is 29.97 or 30
  if( frame_rate_code == 4 || frame_rate_code == 5 )
    recons_progressive     = opt->recons_progressive;
  else
    recons_progressive = false;
  
  memset(&p, 0, sizeof(presTimes));
	internalPTS=0;

	// Pull down stuff
	pullDownDetected=false;
	pulldown_state  = 3;
	interlaced_field_delay = (__int64)(MPEG2_CLK_REF*(frameDelay))>>1;
  progressive_frame_delay = (__int64)(MPEG2_CLK_REF*(1.25*frameDelay));

	Bitstream_Framenum = 0;
	sequencePos=SEQ_FIRSTTIME;

	// Double Buffer
	reca.state = FREE;
	recb.state = FREE;
	dbuf.rec[0] = &reca;
	dbuf.rec[1] = &recb;
	image_zero(&reca.im);
	image_zero(&recb.im);

  YUV.Y = NULL;


  if( opt->bStartInSync )
  {

    // During this start-stop sequence, only calls to GetFrame, can be performed
    // Timestamps will be valid since the sync point.
    // In the start part, we retrieve any possible references needed for the decoding
    // of the first frame.
    // We first set our position to the sync position
    CDemux::SetStreamPos( opt->nSyncPoint );
    // Flush the buffer
    FlushBuffer();
    // Now we decode the previous picture.
    // This will retrieve any references needed.
    // If there are not enough references available, inform about it
    // (likely we're at the beginning of the file)
    // First align the stream with the first picture to be decoded.
    int nParams = SEARCH_P | SEARCH_I | SEARCH_B | SEARCH_FORWARD  | SEARCH_ALIGN_FIRST_FIELD;
    // This shoudlnt fail
    SearchPictureHeader( nParams );
    // Now call DecodePictureEx that will decode the previous
    // picture retrieving any references needed.
    // Parse the start code so
    ui32 dword;
    GetFordDWord(&dword);
    // Decode without forcing to retrieve the first availabe picture
    DecodePictureEx(DECEX_PREV_PICTURE|DECEX_FORCEDECODING);

    // Reset the current timestamps because they are not valid at all.
    p.forward.PTS = p.backward.PTS = 0;
    p.forward.SCR = p.backward.SCR = 0;

    memset(&myPES, 0, sizeof(PESinfo));
    // Now set the position back again to the sync point
    CDemux::SetStreamPos( opt->nSyncPoint );
    // Flag sync Decoding operation. i.e. only calls to getFrame can be performed.
    m_bSyncedDecoding = true;
    // Flush the buffer
    FlushBuffer();
    
    //Read Linear PES init
    StartReadLPES();
    // Store the end point
    m_nEndPoint = opt->nEndPoint;
    if(m_nEndPoint <= opt->nSyncPoint)
      m_nEndPoint = GetStreamSize();

    // That should do it!
  }
  else
  {
    m_bSyncedDecoding = false;
  }
	// Subpic init
  if( subpic_streamID!=-1 )
	  subpic_init(pClut);

	return 0;
}

#define IMAGE_COPY(x)   image_copy(&dbuf.rec[x]->im, &YUV)
#define FIELD_COPY(x,y) field_copy(&dbuf.rec[x]->im, &YUV, y)
#define FIELD_COPY_SWAP(x,y) field_copy_swap(&dbuf.rec[x]->im, &YUV, y)
#define GET_DB(x)  (&dbuf.rec[x]->im)
#define GET_PTS(x)  (dbuf.rec[x]->PTS)
#define SET_PTS(x,y)  (dbuf.rec[x]->PTS=y)
#define SET_FREE(x)	(dbuf.rec[x]->state  =  FREE)
#define SET_FULL(x) (dbuf.rec[x]->state  =  FULL)
#define DB_0 0
#define DB_1 1
void VideoWrapper::swap_db()
{
		TRecImage *temp;
		temp        = dbuf.rec[0];
		dbuf.rec[0] = dbuf.rec[1];
		dbuf.rec[1] = temp;	

}
i64 inline VideoWrapper::get_time(int adjust_clock)
{
	i64    present;
	i64    delay = (__int64)(MPEG2_CLK_REF*(frameDelay));
	ui32  half_delay = (int)delay>>1;


	if(p.image.PTS){
#if DEBUG
    if( ABS(internalPTS - p.image.PTS) > 900000 ) 
      DBG_STR((str, "VideoWrapper::get_time() - Timestamp deviates from expected value by %I64d clock units or %I64d ms\n", internalPTS - p.image.PTS, (internalPTS - p.image.PTS)/27000));
#endif
		internalPTS = p.image.PTS;
		switch( adjust_clock )
		{
			case EQUAL:
				present = internalPTS;
				break;
			case PLUS:
				present = internalPTS = internalPTS + half_delay;
				break;
			case MINUS:
				present = internalPTS = internalPTS - half_delay;
				break;
		}

    		internalPTS += i64(recons_progressive ? (1.25*(double)delay) : delay);	

	}
	else{
			present = internalPTS;
			internalPTS += i64(recons_progressive ? (1.25*(double)delay) : delay);	
	}
	return present;
}

void inline VideoWrapper::ConvertTo444()
{
  m_oUtilFrame.From420to422(YUV.U   , m_pU422, YUV.Yxsize, YUV.Yysize, p.image.progressive_frame );
  m_oUtilFrame.From420to422(YUV.V   , m_pV422, YUV.Yxsize, YUV.Yysize, p.image.progressive_frame );
  m_oUtilFrame.From422to444(m_pU422, m_pU444, YUV.Yxsize, YUV.Yysize);
  m_oUtilFrame.From422to444(m_pV422, m_pV444, YUV.Yxsize, YUV.Yysize);
  
  sYuvFrame.Y = YUV.Y;
  sYuvFrame.U = m_pU444;
  sYuvFrame.V = m_pV444;
}

// CFrame GetFrame
// Frame source definition
bool VideoWrapper::GetFrame(CFrame **pFrame)
{
  int nVal;
  *pFrame = NULL;

  if( m_nLastDecodedPictureStart >= m_nEndPoint &&
      m_bSyncedDecoding)
    return false;  

	ui64  frame_delay = (__int64)(MPEG2_CLK_REF*(frameDelay));

  if(!m_pFrontFrame)
  {
    m_pFrontFrame = m_pFrameBuffer->GetFreeFrame();
    if(!m_pFrontFrame)
      return false;
  }

  if(!m_pBackFrame)
  {
    m_pBackFrame  = m_pFrameBuffer->GetFreeFrame();
    if(!m_pBackFrame)
      return false;
  }

  
  // If we already have a frame
  // return it
  if(m_bFronFrameIsReady)
  {
    *pFrame       = m_pFrontFrame;
    m_pFrontFrame = m_pBackFrame;
    m_pBackFrame  = NULL;
    m_bFronFrameIsReady = false;
    goto EndGetFrame;
  }

  // Retrieve a frame
  nVal = get_frame();
  if(!nVal)
    return false;

  // Convert to 444
  ConvertTo444();
  
  // Parse frame structure
	if(progressive_sequence)
  {
		//Well, here MPEG2 standard defines 
		// to output 1, 2 or 3 consecutive frames
		// for 60 fps progressive output
		// let's output just one
		// This is the case of MPEG1 too
    m_pFrontFrame->SetFrame(&m_oYuvFrame);
    m_pFrontFrame->SetPresTime( get_time(EQUAL) );
    m_pFrontFrame->SetFlags( FRAME_PROGRESSIVE );

    *pFrame       = m_pFrontFrame;
    m_pFrontFrame = m_pBackFrame;
    m_pBackFrame  = NULL;
	}
  else
  {
		if(p.image.progressive_frame)
		{
			if(recons_progressive)
      {
        m_pFrontFrame->SetFrame(&m_oYuvFrame);
        m_pFrontFrame->SetPresTime( get_time(EQUAL) );
        m_pFrontFrame->SetFlags( FRAME_PROGRESSIVE );

        *pFrame       = m_pFrontFrame;
        m_pFrontFrame = m_pBackFrame;
        m_pBackFrame  = NULL;        
        m_bTopFieldExpected = true;
			}
			else if(p.image.top_field_first)
      {
        m_pFrontFrame->SetFrame(&m_oYuvFrame);
        m_pFrontFrame->SetPresTime( get_time(EQUAL) );
        m_pFrontFrame->SetFlags( FRAME_INTERLACED );

				if(p.image.repeat_first_field )
        {
          m_pBackFrame->SetField(&m_oYuvFrame, true);
          m_pBackFrame->SetFlags( FRAME_TOPFIELD );
          m_bTopFieldExpected = false;
				}
        else
          m_bTopFieldExpected = true;

        *pFrame       = m_pFrontFrame;
        m_pFrontFrame = m_pBackFrame;
        m_pBackFrame  = NULL;        
      }
			else //Bottom field first
      {

        m_pFrontFrame->SetField(&m_oYuvFrame, false);
        m_pFrontFrame->SetPresTime( get_time(MINUS) );
        m_pFrontFrame->SetFlags( FRAME_INTERLACED );

				if(m_bTopFieldExpected)
          m_pFrontFrame->SetField(&m_oYuvFrame, true);

        m_pBackFrame->SetField(&m_oYuvFrame, true);
        if(p.image.repeat_first_field )
        {
          m_pBackFrame->SetField(&m_oYuvFrame, false);
          m_pBackFrame->SetPresTime( get_time(PLUS) );
          m_bFronFrameIsReady = true;
          m_bTopFieldExpected = true;
        }
        else
          m_bTopFieldExpected = false;

        *pFrame       = m_pFrontFrame;
        m_pFrontFrame = m_pBackFrame;
        m_pBackFrame  = NULL;    
			}
    } 
		else // Frame is interlaced
    {
			if( p.image.picture_structure == FRAME_PICTURE )
      {
				if(p.image.top_field_first)
        {
          m_pFrontFrame->SetFrame(&m_oYuvFrame);
          m_pFrontFrame->SetPresTime( get_time(EQUAL) );
          m_pFrontFrame->SetFlags( FRAME_INTERLACED );

          *pFrame       = m_pFrontFrame;
          m_pFrontFrame = m_pBackFrame;
          m_pBackFrame  = NULL; 
          m_bTopFieldExpected = true;
				}
				else //bottom field first
        {
          m_pFrontFrame->SetField(&m_oYuvFrame, false);
          m_pFrontFrame->SetPresTime( get_time(MINUS) );
          m_pFrontFrame->SetFlags( FRAME_INTERLACED );

          if(m_bTopFieldExpected)
            m_pFrontFrame->SetField(&m_oYuvFrame, true);
          
          m_pBackFrame->SetField(&m_oYuvFrame, true);

          *pFrame       = m_pFrontFrame;
          m_pFrontFrame = m_pBackFrame;
          m_pBackFrame  = NULL; 
          m_bTopFieldExpected = false;
				}
			}
			else // frames is reconstructed from field pictures
      {
        if(p.image.top_field_first)
        {
          m_pFrontFrame->SetFrame(&m_oYuvFrame);
          m_pFrontFrame->SetPresTime( get_time(EQUAL) );
          m_pFrontFrame->SetFlags( FRAME_INTERLACED );
          
          *pFrame       = m_pFrontFrame;
          m_pFrontFrame = m_pBackFrame;
          m_pBackFrame  = NULL;
          m_bTopFieldExpected = true;
        }
        else //bottom field first
        {
          m_pFrontFrame->SetField(&m_oYuvFrame, false);
          m_pFrontFrame->SetPresTime( get_time(MINUS) );
          m_pFrontFrame->SetFlags( FRAME_INTERLACED );
          
          if(m_bTopFieldExpected)
            m_pFrontFrame->SetField(&m_oYuvFrame, true);
          
          m_pBackFrame->SetField(&m_oYuvFrame, true);
          
          *pFrame       = m_pFrontFrame;
          m_pFrontFrame = m_pBackFrame;
          m_pBackFrame  = NULL; 
          m_bTopFieldExpected = false;
        }
			}	
		}
	}

EndGetFrame:
  // Apply subpictures
  if( subpic_streamID!=-1 )
    subpic_apply(*pFrame, (double)(*pFrame)->GetPresTime()/27000.0);

;
  return true;
}
// Legacy GetFrame
int VideoWrapper::GetFrame(presInfo *pInfo, YUVImageTag **frame)
{

	int val;
	ui64  frame_delay = (__int64)(MPEG2_CLK_REF*(frameDelay));

	*frame = NULL;

   // if there are full frames in the double buffer
	//    output them
	if(dbuf.rec[0]->state == FULL){
		//Return frame 0
		//Swap buffers and signal as FREE
		dbuf.rec[0]->state = FREE;
		*frame = &dbuf.rec[0]->im;
		pInfo->imagePTS = GET_PTS(0);
		swap_db();
		return 1;
	}
	val = get_frame();
	if(!val)
		return 0;


	if(progressive_sequence){
		//Well, here MPEG2 standard defines 
		// to output 1, 2 or 3 consecutive frames
		// for 60 fps progressive output
		// let's output just one
		// This is the case of MPEG1 too
		IMAGE_COPY(0);
		SET_PTS(0, get_time(EQUAL) );
		dbuf.rec[0]->state  =  FREE;
		*frame = GET_DB(0);
		pInfo->imagePTS = GET_PTS(0);

		goto get_frame_end;
	}
	else{
		if(p.image.progressive_frame)
		{
			if(recons_progressive){

				IMAGE_COPY(0);
				SET_PTS(0, get_time(EQUAL));
				dbuf.rec[0]->state  =  FREE;

				*frame = GET_DB(0);
				pInfo->imagePTS = GET_PTS(0);
				goto get_frame_end;
			}

			if(p.image.top_field_first){

				IMAGE_COPY(0);
				SET_PTS(0, get_time(EQUAL));
				dbuf.rec[0]->state  =  FREE;
				if(p.image.repeat_first_field ){
					FIELD_COPY(1, TOP_FIELD);
					dbuf.rec[1]->state  =  TOP_FIELD;
				}
				*frame = GET_DB(0);
				pInfo->imagePTS = GET_PTS(0);

				swap_db();
				goto get_frame_end;
			}
			else{//Bottom field first

				FIELD_COPY(0, BOTTOM_FIELD);
				SET_PTS(0, get_time(MINUS) );
				SET_FREE(0);
				if(m_bTopFieldExpected)
					FIELD_COPY_SWAP(0, BOTTOM_FIELD);

				FIELD_COPY(1, TOP_FIELD);
				dbuf.rec[1]->state = TOP_FIELD;
				if(p.image.repeat_first_field ){
					FIELD_COPY(1, BOTTOM_FIELD);
					// Reset PTS for the repeat_first_field if any
					p.image.PTS = 0;
					SET_PTS(1, get_time(EQUAL) );
					SET_FULL(1);
				}
				*frame = GET_DB(0);
				pInfo->imagePTS = GET_PTS(0);

				swap_db();
				goto get_frame_end;
			}
		}
		else{
			if( p.image.picture_structure == FRAME_PICTURE ){
				if(p.image.top_field_first){
					IMAGE_COPY(0);
					SET_PTS(0, get_time(EQUAL));
					SET_FREE(0);
					*frame = GET_DB(0);
					pInfo->imagePTS = GET_PTS(0);

					goto get_frame_end;
				}
				else{//bottom field first
					FIELD_COPY(0, BOTTOM_FIELD);
					SET_PTS(0, get_time(MINUS) );
					SET_FREE(0);
					if(m_bTopFieldExpected)
						FIELD_COPY_SWAP(0, BOTTOM_FIELD);


					FIELD_COPY(1, TOP_FIELD);
					dbuf.rec[1]->state  =  TOP_FIELD;
					*frame = GET_DB(0);
					pInfo->imagePTS = GET_PTS(0);

					swap_db();
					goto get_frame_end;
				}
			}
			else{
				if(p.image.picture_structure == TOP_FIELD){
					IMAGE_COPY(DB_0);
					SET_PTS(0, get_time(EQUAL) );
					SET_FREE(0);
					*frame = GET_DB(0);
					pInfo->imagePTS = GET_PTS(0);
	
					goto get_frame_end;
				}
				else{//field picture. BOTTOM FIELD
					FIELD_COPY(DB_0, BOTTOM_FIELD);
					SET_PTS(0, get_time(MINUS) );
					SET_FREE(0);
					if(m_bTopFieldExpected)
						FIELD_COPY_SWAP(0, BOTTOM_FIELD);

					FIELD_COPY(DB_1, TOP_FIELD);
					dbuf.rec[1]->state  =  TOP_FIELD;
					*frame = GET_DB(0);
					pInfo->imagePTS = GET_PTS(0);
	
					swap_db();
					goto get_frame_end;
				}
			}	
		}
	}
get_frame_end:
	m_bTopFieldExpected = false;
	return val;	
}


bool VideoWrapper::GetBackByte(ui8 *byte)
{
	bool bSuccess = true;
  PESinfo dummyPES;

	// The offset has to be 
	if( m_nTempVideoBufferPtr <= (i32)m_nTempVideoBufferSize )
	{
		// Grab previous PES if we're finished with this one
		// or we don't have any data
		if( m_nTempVideoBufferPtr <= 0 || !m_pTempVideoBuffer )
		{
			// The buffer has been underrun.
			// Retrieve the previous buffer. This assumes that the position in the
			// stream is exactly the byte after the latest PES.

			// Rewind PES
			// If the pointer of the file is at the end
			// of the Pes rewind this PES first
			if( m_nPesPosState == PesEnd )
				bSuccess = RewindPreviousPES( streamID );
			// Rewind again to find the real Pes
			bSuccess = RewindPreviousPES( streamID );
			ui64 PESpos = CDemux::GetStreamPos();

			if( bSuccess )
			{
				// Read the PES
				// Invalid value for streamId
				dummyPES.streamID = 0xAA;
				while( bSuccess && dummyPES.streamID!=streamID )
				{
					bSuccess = ReadPES( (unsigned char **)&m_pTempVideoBuffer, &dummyPES );
					if( bSuccess )
					{
						m_nTempVideoBufferSize = dummyPES.payloadSize;
						m_nTempVideoBufferPtr  = dummyPES.payloadSize;
						// Update this two variables to be able to
						// reposition the stream after this.
						m_nPesPosState = PesBegin;
						m_nEndPesPosition = CDemux::GetStreamPos();
					}
				}
				// Because we want to emulate the going backwards 
				// behaviour, set the position to the beginning of the PES
				// again
				CDemux::SetStreamPos( PESpos );
			}
			else
				DBG_STR((str, "VideoWrapper::GetBackByte() - Couldnt rewind PES\n"));
		}

		// There are still bytes in this buffer to gather.
		if( bSuccess )
		{	
			m_nTempVideoBufferPtr--;
			*byte = *(m_pTempVideoBuffer + m_nTempVideoBufferPtr);
		}		
	}
	else
	{
		DBG_STR((str, "VideoWrapper::GetBackByte() - Buffer offset outside bounds\n"))
		bSuccess = false;
	}
	return bSuccess;
}

bool VideoWrapper::GetFordByte(ui8 *byte)
{
	bool    bSuccess = true;
  PESinfo dummyPES;

	// The offset has to be minor than the buffer size
	if( m_nTempVideoBufferPtr <= (i32)m_nTempVideoBufferSize )
	{
		// Grab previous PES if we're finished with this one
		if( m_nTempVideoBufferPtr >= (i32)m_nTempVideoBufferSize || !m_pTempVideoBuffer)
		{
			// Read the PES
			// If the pointer is a the beginning of the Pes
			// We just read, read it again to parse it. 
			if( m_nPesPosState == PesBegin )
				bSuccess = ReadPES( (unsigned char **)&m_pTempVideoBuffer, &dummyPES );

			// Invalid value for streamId
			dummyPES.streamID = 0xAA;
			while( bSuccess && dummyPES.streamID!=streamID )
			{
				bSuccess = ReadPES( (unsigned char **)&m_pTempVideoBuffer, &dummyPES );
				if( bSuccess && dummyPES.streamID==streamID )
				{
					m_nPesPosState = PesEnd;
					m_nTempVideoBufferSize = dummyPES.payloadSize;
					m_nTempVideoBufferPtr  = 0;					
				}
			}
		}
		// There are still bytes in this buffer to gather.
		if( bSuccess )
		{
			*byte = *(m_pTempVideoBuffer + m_nTempVideoBufferPtr);
			m_nTempVideoBufferPtr++;			
		}		
	}
	else
	{
		bSuccess = false;
	}
	return bSuccess;
			
}
bool VideoWrapper::GetFordDWord(ui32 *dword)
{
	ui8 byte;

	*dword = 0;

	bool bSuccess = GetFordByte( &byte );
	*dword = byte << 24;
	if( bSuccess )
		bSuccess = GetFordByte( &byte );
	*dword |= byte << 16;
	if( bSuccess )
		bSuccess = GetFordByte( &byte );
	*dword |= byte << 8;		
	if( bSuccess )
		bSuccess = GetFordByte( &byte );
	*dword |= byte;	
	
	return bSuccess;
}

bool VideoWrapper::SearchPictureHeader( int &nParams )
{
	bool bSuccess=true;
	bool bFound = false;
   i32 nParsedBytes;
  ui32 nPictureType;
  ui32 nPrevPictureType;
  ui32 nTemporalReference;
  ui64 nFirstFieldPictureStartPos=0;
  bool bLookingSecondField=false;
  ui32 nPrevTemporalReference;
  ui16 nVBVDelay;
  ui8  nPictureStructure;
	
	ui8  byte;
	
	while( !bFound && bSuccess )
	{
		ui32 nStartCode = 0xFFFFFFFF;
		
		// Select direccion of search
		if( nParams&SEARCH_BACKWARDS )
			while( nStartCode!=PICTURE_START_CODE && bSuccess )
			{
				bSuccess = GetBackByte(&byte);
				nStartCode = (nStartCode>>8) | (byte<<24);
			}
			else
				while( nStartCode!=PICTURE_START_CODE && bSuccess )
				{
					bSuccess = GetFordByte(&byte);
					nStartCode = (nStartCode<<8) | byte;
				}
				
				if( bSuccess )
				{
					// We found a picture header and we are aligned with it
					// Get start code
					// Only parse the start code if we were reading
					// backwards

					if( nParams&SEARCH_BACKWARDS )
						GetFordDWord(&nStartCode);

					GetFordByte(&byte);
          nTemporalReference = byte << 2;
					GetFordByte(&byte);
          nTemporalReference |= (byte&0xC0)>>6;

          nParsedBytes = 6;
					nPictureType = (byte&0x3F)>>3;

          // If we're not interested in this type
          // continue searching

#define FOUND_TYPE  ((nPictureType==P_TYPE && nParams&SEARCH_P) ||  \
                     (nPictureType==B_TYPE && nParams&SEARCH_B) ||  \
                     (nPictureType==I_TYPE && nParams&SEARCH_I) )

#if 0
          char cTypes[] = {'X','I','P','B','X','X','X','X','X','X','X','X','X','X','X','X'};
          DBG_STR((str, "VideoWrapper::SearchPictureHeader - %c picture found at %d\n", cTypes[nPictureType], GetStreamPos()-6))
#endif
		
          if( nParams&SEARCH_ALIGN_FIRST_FIELD )
          {
            // parse the rest of the header
            nVBVDelay  = (byte&0x07)<<13;
            GetFordByte(&byte);
            nVBVDelay |= byte<<5;
            GetFordByte(&byte);
            nVBVDelay |= (byte&0xF8)>>3;

            nParsedBytes += 2;
            // Keep track of the bits remaining in 'byte'
            // Now we have 3 bits remaining.
            ui32 nBitsRemaining = 3;

#define PARSE_BITS(x){                                    \
  if(nBitsRemaining<=x)                                   \
            {                                             \
            GetFordByte(&byte);                           \
            nParsedBytes++;                               \
            nBitsRemaining = 8 - (x - nBitsRemaining);    \
            }                                             \
            else                                          \
            nBitsRemaining-= x; }                               
            
#define SHOW_BIT()( (byte&(1<<(nBitsRemaining-1)))>>(nBitsRemaining -1) )

#define Rewind(x) { for(int i=0; i<x; i++) GetBackByte(&byte); }

            if( nPictureType == P_TYPE || nPictureType == B_TYPE )
            {
              PARSE_BITS(4);
              if( nPictureType == B_TYPE )
                PARSE_BITS(4);
            }
            // parse extra information
            while(SHOW_BIT())
            {
              PARSE_BITS(1);
              PARSE_BITS(8);
            }
            // parse the next bit that is 0.
            PARSE_BITS(1);
            // Find next start code prefix
            ui32 nStartCode = 0xFFFFFF00;
            while( nStartCode!=1 && bSuccess )
            {
              bSuccess = GetFordByte(&byte);
              nParsedBytes += 1;
              nStartCode = (nStartCode<<8) | byte;
            }
            if(!bSuccess) continue;
            GetFordByte(&byte);
            nParsedBytes += 1;
            if(byte!=0xB5)
            {
              // Either this is MPEG1 or something wrong happened
              goto validate;
            }

            GetFordByte(&byte);
            if((byte>>4)!=8) // The extension_start_code_identifier is 8 for picture
              continue; // something wrong happened
            nParsedBytes += 1;
            GetFordByte(&byte); // parse some stuff
            GetFordByte(&byte);
            nParsedBytes += 2;
            nPictureStructure = byte&0x03;

            bool bJumpToPos = false;
            if(nPictureStructure==FRAME_PICTURE){
              if(FOUND_TYPE)
              {
                bFound = true;
                SetSearchType( nParams, nPictureType );
              }
            }
            else{// we have a field picture
              if( !bLookingSecondField ){
                nFirstFieldPictureStartPos = GetStreamPos() - nParsedBytes;
                nPrevTemporalReference = nTemporalReference;
                nPrevPictureType       = nPictureType;
                bLookingSecondField = true;
              }
              else
              {
                // In this two field pictures, the temporal reference
                // matches. These two form the frame picture. Position
                // ourselves in the first one.                
                if( nPrevTemporalReference==nTemporalReference ){
                  // If we're going forward, it was the previous one
                  if(nParams&SEARCH_FORWARD)
                  {
                    bJumpToPos = true;
                    nPictureType = nPrevPictureType;
                  }
                }
                else{
                  if(nParams&SEARCH_BACKWARDS)
                  {
                    bJumpToPos = true;
                    nPictureType = nPrevPictureType;
                  }
                }
                Second_Field = 0;
                if(FOUND_TYPE)
                {
                  SetSearchType( nParams, nPictureType );                  
                  bFound = true;
                }
                bLookingSecondField = false;
              }
            }
            if(bFound || nParams&SEARCH_BACKWARDS)
            {
              if(bJumpToPos)
                  SetStreamPos(nFirstFieldPictureStartPos);
              else
                  Rewind(nParsedBytes);
            }

          }
          else
          {
validate:
            if(FOUND_TYPE)
            {
              bFound = true;
              SetSearchType( nParams, nPictureType );
            }
            if( bFound || nParams&SEARCH_BACKWARDS )
            {
              Rewind(nParsedBytes);
            }              
          }
        }
	}
	return bSuccess;
}

bool VideoWrapper::EnterCoreContext()
{
	Set_Buffer_State( m_pTempVideoBuffer, 
					  m_nTempVideoBufferPtr, 
					  m_nTempVideoBufferSize );
	return true;
}

bool VideoWrapper::ExitCoreContext()
{
	m_nTempVideoBufferPtr = Get_Buffer_Pos();
  // If the pos is <0 means that the current position
  // of the video buffer belongs to a previous buffer
  // Rewind it
  if( m_nTempVideoBufferPtr<0 )
  {
    int nRewindBytes = -m_nTempVideoBufferPtr;
    // Set the beginning of the buffer
    m_nTempVideoBufferPtr = 0;
    // Rewind
    ui8 byte;
    for(int i=0; i<nRewindBytes; i++) 
      GetBackByte(&byte);
  }
	return true;
}


////////////////////////////////////////////////////////////////////////////////
//
// Navigational retrieving methods. Timestamps are not valid within this context.
//
////////////////////////////////////////////////////////////////////////////////
bool VideoWrapper::GetNextFrame(CFrame **pFrame)
{
  YUVImageTag *pYuvImage;
  bool bSuccess = GetNextFrame(&pYuvImage);
  if(!bSuccess)
  {
    *pFrame = NULL;
    return false;
  }
  ConvertTo444();
  m_oUtilFrame.SetFrame(&m_oYuvFrame);
  *pFrame = &m_oUtilFrame;
  return false;
}

bool VideoWrapper::GetNextFrame(YUVImageTag **ppFrame)
{
    DecodePictureEx(DECEX_NEXT_PICTURE);
    Write_Frame( p.actual.picture_coding_type==B_TYPE ? auxframe : forward_reference_frame, 0 );
    *ppFrame = &YUV;

    if(m_bReadError)
        error=END_OF_STREAM;
    return !m_bReadError;
}

bool VideoWrapper::GetPrevFrame(CFrame **pFrame)
{
  YUVImageTag *pYuvImage;
  bool bSuccess = GetPrevFrame(&pYuvImage);
  if(!bSuccess)
  {
    *pFrame = NULL;
    return false;
  }
  ConvertTo444();
  m_oUtilFrame.SetFrame(&m_oYuvFrame);
  *pFrame = &m_oUtilFrame;
  return false;
}

bool VideoWrapper::GetPrevFrame(YUVImageTag **ppFrame)
{
    DecodePictureEx(DECEX_PREV_PICTURE);
    Write_Frame( p.actual.picture_coding_type==B_TYPE ? auxframe : forward_reference_frame, 0 );
    *ppFrame = &YUV;
    if(m_bReadError)
        error=END_OF_STREAM;
    return !m_bReadError;
}

bool VideoWrapper::GetNextIFrame(CFrame **pFrame)
{
  YUVImageTag *pYuvImage;
  bool bSuccess = GetNextIFrame(&pYuvImage);
  if(!bSuccess)
  {
    *pFrame = NULL;
    return false;
  }
  ConvertTo444();
  m_oUtilFrame.SetFrame(&m_oYuvFrame);
  *pFrame = &m_oUtilFrame;
  return false;
}

bool VideoWrapper::GetNextIFrame(YUVImageTag **ppFrame)
{
    DecodePictureEx(DECEX_NEXT_PICTURE|DECEX_JUST_IPICTURE);
    Write_Frame( current_frame, 0 );
    *ppFrame = &YUV;
    if(m_bReadError)
        error=END_OF_STREAM;
    return !m_bReadError;
}

bool VideoWrapper::GetPrevIFrame(CFrame **pFrame)
{
  YUVImageTag *pYuvImage;
  bool bSuccess = GetPrevIFrame(&pYuvImage);
  if(!bSuccess)
  {
    *pFrame = NULL;
    return false;
  }
  ConvertTo444();
  m_oUtilFrame.SetFrame(&m_oYuvFrame);
  *pFrame = &m_oUtilFrame;
  return false;
}

bool VideoWrapper::GetPrevIFrame(YUVImageTag **ppFrame)
{
    DecodePictureEx(DECEX_PREV_PICTURE|DECEX_JUST_IPICTURE);
    Write_Frame( current_frame, 0 );
    *ppFrame = &YUV;
    if(m_bReadError)
        error=END_OF_STREAM;
    return !m_bReadError;
}


int VideoWrapper::DecodePictureEx( int nDecExParams )
{
    int nRetValue=0;
    int nPictureTypes, nDecodingModes;
    int nParams;
    
    nPictureTypes = nDecExParams&DECEX_JUST_IPICTURE ? 
                    SEARCH_I : 
                    SEARCH_I | SEARCH_B | SEARCH_P;

    // If just I picture was demanded, skip references search
    nDecodingModes = nDecExParams&DECEX_JUST_IPICTURE ? 
                     SKIP_REF_SEARCH|DECODING_PICTURE : DECODING_PICTURE;

    if( nDecExParams&DECEX_NEXT_PICTURE )
    {
        nParams = nPictureTypes | SEARCH_FORWARD | SEARCH_ALIGN_FIRST_FIELD;
        if( SearchPictureHeader( nParams ) )
        {
            if( m_bFirstPictureDecoded )
            {
              nDecodingModes |= SKIP_REF_SEARCH;
              nRetValue = DecodePicture(nDecodingModes);
            }
            else
            {
                m_bFirstPictureDecoded = true;
                nRetValue = DecodePicture(nDecodingModes);
                // This is here in the case of an I frame. Likely the first
                // of the stream 
                // Decode next picture. Should be a P.
                while( nRetValue==DEC_NOTENOUGHREF && 
                       !m_bReadError &&
                       !(nDecExParams&DECEX_FORCEDECODING)) // this will return DEC_NOTENOUGHREF
                    nRetValue=DecodePicture(nDecodingModes);
            }
            m_bPreviousDirWasForward = true;
        }
    }
    else
    {
        // Now rewind picture just decoded
        nParams = nPictureTypes | SEARCH_BACKWARDS | SEARCH_ALIGN_FIRST_FIELD;
        SearchPictureHeader( nParams );
        
        nParams = nPictureTypes | SEARCH_BACKWARDS | SEARCH_ALIGN_FIRST_FIELD;
        DBG_STR((str, "VideoWrapper::DecodePictureEx - back decoding starting at %d\n", GetStreamPos()))        
        if( SearchPictureHeader( nParams ) )
        {
            m_bFirstPictureDecoded = false;
            nRetValue = DecodePicture(nDecodingModes);
            // This is here in the case of an I frame. Likely the first
            // of the stream 
            // Decode next picture. Should be a P.
            while( nRetValue==DEC_NOTENOUGHREF && 
                  !m_bReadError &&
                  !(nDecExParams&DECEX_FORCEDECODING)) // this will return DEC_NOTENOUGHREF
              nRetValue=DecodePicture(nDecodingModes);
        }
        //DBG_STR((str, "VideoWrapper::DecodePictureEx - back decoding stopping at %d\n", GetStreamPos()))
        m_bPreviousDirWasForward = false;
    }
    // If just and I picture was demanded, reposition the stream
    // to the frame after the next I or P picture, because DECEX_JUST_IPICTURE makes
    // the decoder to output the I picture without references.
    if( nDecExParams&DECEX_JUST_IPICTURE )
    {
      // Search the next I or P
      nParams = SEARCH_P | SEARCH_I | SEARCH_FORWARD | SEARCH_ALIGN_FIRST_FIELD;
      SearchPictureHeader( nParams );
      // Get one byte to avoid aligning to the I or P again
      ui8 byte;
      GetFordByte(&byte);
      // Align to the frame after it
      nParams = SEARCH_P | SEARCH_I | SEARCH_B | SEARCH_FORWARD | SEARCH_ALIGN_FIRST_FIELD;
      SearchPictureHeader( nParams );       
    }

    return nRetValue;
}

int VideoWrapper::DecodePicture(int nDecParams)
{
	ui64 nPicturePos;
	int  nParams, cc;
	unsigned char *tmp;  /* temporary swap pointer */
	bool bSuccess;
  int  nReturn = DEC_OK;
	
  // Our goal is to decode the picture that is this position.
  // Parse picture header
  nParams  = SEARCH_P | SEARCH_I | SEARCH_B | SEARCH_FORWARD | SEARCH_ALIGN_FIRST_FIELD;
  if(!(nDecParams&SKIP_REF_SEARCH))
  {
    if( SearchPictureHeader( nParams )  )
    {
      // If we are told NOT to search
      // for references skip this section.
      // This is flagged when we already have
      // the correct references.
      
      // Retrieve position of this picture
      nPicturePos = GetStreamPos();
      
      switch( GetSearchType( nParams ) )
      {
      case I_TYPE:
        // No references pictures needed. Decode picture right away.
        // However the output will be the previous I or P
        // Retrieve it only if we this is the first call to DecodePicture,
        // i.e. we're not doing recursive calls to DecodePicture to retrieve 
        // references.
        // Backward reference needed
        if( nDecParams&DECODING_PICTURE )
        {
          nParams = SEARCH_I | SEARCH_P | SEARCH_BACKWARDS | SEARCH_ALIGN_FIRST_FIELD;
          if( SearchPictureHeader( nParams ) )
            // We found the previous P or I picture. Decode it.
            nReturn = DecodePicture();
          else
            nReturn = DEC_NOTENOUGHREF;
        }
        break;			
      case P_TYPE:
        // Backward reference needed
        nParams = SEARCH_I | SEARCH_P | SEARCH_BACKWARDS  | SEARCH_ALIGN_FIRST_FIELD;
        if( SearchPictureHeader( nParams ) )
        {
          // We found the previous P or I picture. Decode it.
          nReturn = DecodePicture();
          // Put this P or I frame as reference 
        }
        else
          nReturn = DEC_NOTENOUGHREF;
        // Rewind stream to find previous I or P picture.
        // Decode it and put it as reference. Come back to this place.
        break;
      case B_TYPE:
        ui64 nFirstRefPos = -1;
        bool bFirstRefOk=false, bSecondRefOk=false;
        // Search for the previous I or P
        nParams = SEARCH_P | SEARCH_I | SEARCH_BACKWARDS  | SEARCH_ALIGN_FIRST_FIELD;
        bSuccess = SearchPictureHeader( nParams );
        if(bSuccess)
        {
          nFirstRefPos = GetStreamPos();
          
          // Search for the previous I or P picture again
          nParams = SEARCH_P | SEARCH_I | SEARCH_BACKWARDS  | SEARCH_ALIGN_FIRST_FIELD;
          bSuccess = SearchPictureHeader( nParams );
          if( bSuccess )
            bFirstRefOk = DecodePicture()!=DEC_NOTENOUGHREF;
        }
        // If we found the first reference decode it
        if( nFirstRefPos != -1 )
        {
          // Set the stream to the first reference
          // we found
          SetStreamPos( nFirstRefPos );
          // Decode it
          bSecondRefOk = DecodePicture()!=DEC_NOTENOUGHREF;
        }
        
        nReturn = ( bSuccess && bFirstRefOk && bSecondRefOk ) ? DEC_OK : DEC_NOTENOUGHREF;
        break;
      }
      
      // Restore position of the beginning of this picture
      SetStreamPos( nPicturePos );
    }
  } 

  if(!Second_Field)
    m_nLastDecodedPictureStart = GetStreamPos();
  
  // Don't enter and exit context if we are in continous reading mode!
  if(!(nDecParams&CONTINOUS_READING))  
    EnterCoreContext();

  // We should be align to the first field by now.
  Second_Field = 0;
  
start_decode:
  if(Get_Hdr())
  {
    if (picture_structure==FRAME_PICTURE && Second_Field)
    {
      DBG_STR((str, "VideoWrapper::DecodePicture - Odd number of field pictures\n"));
      Second_Field = 0;
    }			

    if( !(nDecParams&DONT_UPDATE_REF) )
    {
      // DBG_STR((str, "VideoWrapper::DecodePicture - pic type %d  sec field %d\n", picture_coding_type,Second_Field));
      // Update picture buffers
      for (cc=0; cc<3; cc++)
      {
        /* B pictures do not need to be saved for future reference */
        if (picture_coding_type==B_TYPE)
        {
          current_frame[cc] = auxframe[cc];
        }
        else
        {
          /* only update at the beginning of the coded frame */
          if (!Second_Field)
          {
            tmp = forward_reference_frame[cc];
            
            /* the previously decoded reference frame is stored
            coincident with the location where the backward 
            reference frame is stored (backwards prediction is not
            needed in P pictures) */
            forward_reference_frame[cc] = backward_reference_frame[cc];
            
            /* update pointer for potential future B pictures */
            backward_reference_frame[cc] = tmp;				
          }
          
          /* can erase over old backward reference frame since it is not used
          in a P picture, and since any subsequent B pictures will use the 
          previously decoded I or P frame as the backward_reference_frame */
          current_frame[cc] = backward_reference_frame[cc];
        }
        /* IMPLEMENTATION:
        one-time folding of a line offset into the pointer which stores the
        memory address of the current frame saves offsets and conditional 
        branches throughout the remainder of the picture processing loop */
        if (picture_structure==BOTTOM_FIELD)
          current_frame[cc]+= (cc==0) ? Coded_Picture_Width : Chroma_Width;
      }
    }
    // Finally decode the picture data
    picture_data();
    
    if (picture_structure!=FRAME_PICTURE)
      Second_Field = !Second_Field;

    if (picture_structure!=FRAME_PICTURE && Second_Field)
      goto start_decode;

  }
  else
  {
    DBG_STR((str, "VideoWrapper::DecodePicture - GetHdr() failed.\n"));    
  }
  // Don't enter and exit context if we are in continous reading mode!
  if(!(nDecParams&CONTINOUS_READING))    
    ExitCoreContext();

  return nReturn;
}



bool VideoWrapper::get_frame()
{  
    updateVideoStats();
    
    time= (ui32)(double(GetTime())/ (double)27000); //miliseconds
    //In case  there is a problem
    YUV.U=NULL;
    YUV.V=NULL;
    YUV.Y=NULL;
    YUV.Yxsize=-1;
    timeCode.hour= hour;
    timeCode.minute= minute;
    timeCode.second= sec;
    timeCode.frame=  frame;
    
    if(stopDecoding){
        error=PLAYER_STOPPED;
        return 0;
    }
    
    if(m_bSyncedDecoding)
    {
      int nDecodingModes = SKIP_REF_SEARCH|DECODING_PICTURE|CONTINOUS_READING;
      // Decode an extra picture if we don't have enough references
      DecodePicture(nDecodingModes);

    }
    else
    {
      DecodePictureEx(DECEX_NEXT_PICTURE);
    }

    Write_Frame( p.actual.picture_coding_type ==B_TYPE ? auxframe : forward_reference_frame, 0 );
    p.image = p.actual.picture_coding_type==B_TYPE? p.actual : p.forward; 
    if(m_bReadError)
        error=END_OF_STREAM;

    return !m_bReadError;
}


int VideoWrapper::Stop()
{


  if(m_pFrontFrame)
    m_pFrontFrame->Release();
  if(m_pBackFrame)  
    m_pBackFrame->Release();

	time=0;
  if( subpic_streamID!=-1 )
	  subpic_free(); /* reset subpics */

  /* IDCT */
  if( myVideo->DeInitIdct )
    myVideo->DeInitIdct();
	return 0;
}

void DeInitClip(){
	free(Clip-384);
}
void InitClip(){

  int i;

  /* Clip table */
  if (!(Clip=(unsigned char *)malloc(1024)))
    Error("Clip[] malloc failed\n");

  Clip += 384;

  for (i=-384; i<640; i++)
    Clip[i] = (i<0) ? 0 : ((i>255) ? 255 : i);
}

/* mostly IMPLEMENTAION specific rouintes */
static void Initialize_Sequence()
{
  int cc, size;
  static int Table_6_20[3] = {6,8,12};

  /* check scalability mode of enhancement layer */
  if (Two_Streams && (enhan.scalable_mode!=SC_SNR) && (base.scalable_mode!=SC_DP))
    Error("unsupported scalability mode\n");

  /* force MPEG-1 parameters for proper decoder behavior */
  /* see ISO/IEC 13818-2 section D.9.14 */
  if (!base.MPEG2_Flag)
  {
    progressive_sequence = 1;
    progressive_frame = 1;
    picture_structure = FRAME_PICTURE;
    frame_pred_frame_dct = 1;
    chroma_format = CHROMA420;
    matrix_coefficients = 5;
  }

  /* round to nearest multiple of coded macroblocks */
  /* ISO/IEC 13818-2 section 6.3.3 sequence_header() */
  mb_width = (horizontal_size+15)/16;
  mb_height = (base.MPEG2_Flag && !progressive_sequence) ? 2*((vertical_size+31)/32)
                                        : (vertical_size+15)/16;

  Coded_Picture_Width = 16*mb_width;
  Coded_Picture_Height = 16*mb_height;

    //Allocate space for Output Bitmap
   //DibArray=malloc( 		Coded_Picture_Width * Coded_Picture_Height * 3 );  
	//DibArray = TempArray;

  /* ISO/IEC 13818-2 sections 6.1.1.8, 6.1.1.9, and 6.1.1.10 */
  Chroma_Width = (chroma_format==CHROMA444) ? Coded_Picture_Width
                                           : Coded_Picture_Width>>1;
  Chroma_Height = (chroma_format!=CHROMA420) ? Coded_Picture_Height
                                            : Coded_Picture_Height>>1;
  
  /* derived based on Table 6-20 in ISO/IEC 13818-2 section 6.3.17 */
  block_count = Table_6_20[chroma_format-1];

  for (cc=0; cc<3; cc++)
  {
    if (cc==0)
      size = Coded_Picture_Width*Coded_Picture_Height;
    else
      size = Chroma_Width*Chroma_Height;

    if (!(backward_reference_frame[cc] = (unsigned char *)malloc(size)))
      Error("backward_reference_frame[] malloc failed\n");

    if (!(forward_reference_frame[cc] = (unsigned char *)malloc(size)))
      Error("forward_reference_frame[] malloc failed\n");

    if (!(auxframe[cc] = (unsigned char *)malloc(size)))
      Error("auxframe[] malloc failed\n");

    if(Ersatz_Flag)
      if (!(substitute_frame[cc] = (unsigned char *)malloc(size)))
        Error("substitute_frame[] malloc failed\n");


    if (base.scalable_mode==SC_SPAT)
    {
      /* this assumes lower layer is 4:2:0 */
      if (!(llframe0[cc] = (unsigned char *)malloc((lower_layer_prediction_horizontal_size*lower_layer_prediction_vertical_size)/(cc?4:1))))
        Error("llframe0 malloc failed\n");
      if (!(llframe1[cc] = (unsigned char *)malloc((lower_layer_prediction_horizontal_size*lower_layer_prediction_vertical_size)/(cc?4:1))))
        Error("llframe1 malloc failed\n");
    }
  }

  /* SCALABILITY: Spatial */
  if (base.scalable_mode==SC_SPAT)
  {
    if (!(lltmp = (short *)malloc(lower_layer_prediction_horizontal_size*((lower_layer_prediction_vertical_size*vertical_subsampling_factor_n)/vertical_subsampling_factor_m)*sizeof(short))))
      Error("lltmp malloc failed\n");
  }



}

void Error(char *text)
{
  fprintf(stderr,text);
//  exit(1);
}

/* Trace_Flag output */
void Print_Bits(int code,int bits,int len)
{
  int i;
  for (i=0; i<len; i++)
    printf("%d",(code>>(bits-1-i))&1);
}


void Deinitialize_Sequence()
{
  int i;

  /* clear flags */
  base.MPEG2_Flag=0;

  for(i=0;i<3;i++)
  {
    free(backward_reference_frame[i]);
    free(forward_reference_frame[i]);
    free(auxframe[i]);

    if (base.scalable_mode==SC_SPAT)
    {
     free(llframe0[i]);
     free(llframe1[i]);
    }
  }

  if (base.scalable_mode==SC_SPAT)
    free(lltmp);

#ifdef DISPLAY
  if (Output_Type==T_X11) 
    Terminate_Display_Process();
#endif
}




bool VideoWrapper::SetStreamPos(ui64 pos)
{
	// Lock the decoder
	CAutoLock lock(&m_csLockSection);
  PESinfo   dummyPes;

	bool bSuccess = false;

	i64 nFirstJump = (i64)pos - nJumpBackwards;
	ui64 nCurrentPos;

	if( nFirstJump<=0 )
		nFirstJump = 0;

  // Align stream to the pack header near the first Jump
	CDemux::SetStreamPos( CDemux::GetSyncPoint(pos) );
	
  m_bReadSuccess = true;
	// Read a PES of video
	while( m_bReadSuccess && (pos >= CDemux::GetStreamPos()) )
	{
		while( (m_bReadSuccess = ReadPES((unsigned char **)&m_pTempVideoBuffer, &dummyPes)) &&
			(dummyPes.streamID == streamID) )
		{
			// Wet got a video PES.
			// Get current Position
			nCurrentPos = CDemux::GetStreamPos();
			
			// Is the position requested inside our PES?
			if( pos <   nCurrentPos &&
				pos >=  (nCurrentPos - dummyPes.payloadSize) )
			{
				//Yep, it is.
				m_nTempVideoBufferSize = dummyPes.payloadSize;
				m_nTempVideoBufferPtr  = (ui32)(dummyPes.payloadSize - (nCurrentPos - pos));
				m_nPesPosState = PesEnd;
				bSuccess = true;
				break;
			}
			else if( pos < nCurrentPos )
			{
				// we missed the position requested without
				// finding an appropiate PES position
				DBG_STR((str, "VideoWrapper::SetStreamPos - Position not found\n"))
				m_nTempVideoBufferSize = dummyPes.payloadSize;
				m_nTempVideoBufferPtr = 0;
				bSuccess = false;
				break;
			}
		}
	}

  // Because this was a seek operation and we didnt succeed
  // remove the present data in the buffer because its no longer valid.
  if( !bSuccess )
    FlushBuffer();

	return bSuccess;
}

ui64 VideoWrapper::GetStreamPos()
{
	// Lock the decoder object in this function
	CAutoLock lock(&m_csLockSection);

	// Get real current position
	// If this Pes was read backwards the
	// end position was already retrieved
	ui64 nCurrentPos = m_nPesPosState==PesBegin ? m_nEndPesPosition : CDemux::GetStreamPos();

	return nCurrentPos - (m_nTempVideoBufferSize - m_nTempVideoBufferPtr );
}
	


int VideoWrapper::GetError()
{
	int temp;
	temp=error;
	error=0;
	return temp;
}

void VideoWrapper::updateVideoStats()
{

		if(isMPEG2)
    {
			DAR = MPEG2aspect_ratio_Table[aspect_ratio_information];
		}
		else{
			DAR = (MPEG1aspect_ratio_Table[aspect_ratio_information]*(double)pictureHeight)/((double)pictureWidth);
		}
		
		frameRate=frameRateTable[frame_rate_code];
		
		detectedFrameDelay=1/frameRateTable[detectedFrameRateCode];

		frameDelay=1/frameRate;

}

// Stub for feeding data to MSSG video decoder
bool VideoWrapper::ReadVideoData( ui8** ppBuffer, ui32* pBufferSize )
{
    if( m_nPesPosState == PesBegin )
        CDemux::SetStreamPos( m_nEndPesPosition );
    do{
        m_bReadSuccess = ReadLPES((unsigned char **)&m_pTempVideoBuffer, &myPES);
        if(!m_bReadSuccess) break;
        if( myPES.streamID    == subpic_streamID &&
            myPES.subStreamID == subpic_substreamID ) {
            /* this is a subpic; parse it, and discard it so the video decoder
            * doesn't see it */
            
            subpic_decode((unsigned char *)m_pTempVideoBuffer, myVideo->myPES.payloadSize, (int)((double)myVideo->myPES.PTS/27000.0) );
            continue;
        }
    }while( myPES.streamID!=streamID);
    
    // We are feeding the video decoder
    //   set the offset to what we've read.
    m_nTempVideoBufferSize   = myPES.payloadSize;
    m_nTempVideoBufferPtr    = myPES.payloadSize;
    m_nPesPosState = PesEnd;
    
    // Update buffers
    *ppBuffer     = m_pTempVideoBuffer;
    *pBufferSize = myPES.payloadSize;
    return m_bReadSuccess;
}

bool VideoWrapper::Is24Progressive()
{
  //This megamatic function will try to detect
  // the framerate of a stream, just looking into
  // the PTS from the incoming pictures.
  // This is necessary to detect 24 fps progressive sequences
  // that have the frame_rate set to 29.97 and use repeat_first_field flag
  //
  bool bSuccess = false;

  ui64 nStreamSize = GetStreamSize();
  // Jump to the middle
  CDemux::SetStreamPos(nStreamSize>>1);
	Initialize_Buffer();
  int nFramesToTry = 10;
  while( Get_Hdr() && nFramesToTry-- )
  {
    if(picture_structure!=FRAME_PICTURE)
      break;
    // If its not 29.97
    if( frame_rate_code!= 4 )
      break;
    if(repeat_first_field){
      bSuccess = true;
      break;
    }
  }
  return bSuccess;
}

TMPGVideoInfo * VideoWrapper::GetVideoInfo()
{
    video_info.width                    = Coded_Picture_Width;
    video_info.height                   = Coded_Picture_Height;
    video_info.aspect_ratio_information = aspect_ratio_information;
    video_info.bit_rate_value           = bit_rate_value;
    video_info.detected_frame_rate_code = detectedFrameRateCode;
    video_info.frame_rate_code          = frame_rate_code;
    video_info.isMPEG2                  = isMPEG2;
    video_info.progressive_frame        = progressive_frame;
    video_info.progressive_sequence     = progressive_sequence;

    return &video_info;
}

