/* 
 *  Deinterlacer.cpp - Original code by Donald Graft
 *                                      neuron2@home.com.
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "..\..\FlasKMPEG.h"
#include "Deinterlacer.h"
 
struct MyFilterData {
	int					*prevFrame;
	unsigned long		*moving;
	int					motionOnly;
	int 				Blend;
	int 				threshold;
	int					fieldShift;
	int					bitpitch;
	bool				skip;
	int					first;
} my_mfd;

MyFilterData *mfd = &my_mfd;

int StartDeinterlacer(int w, int h, TDeinterlacerConfig *config) {

	if(!config)
		return 0;
	mfd->prevFrame		= new int[w*h];
	mfd->fieldShift     = 0;
	mfd->Blend          = config->blend;
	mfd->threshold      = config->threshold;
	mfd->motionOnly     = 0;

	memset(mfd->prevFrame, 0, w*h*sizeof(int));


	mfd->bitpitch		= (w + 32 - 1) / 32;
	mfd->moving			= new unsigned long[mfd->bitpitch * h];
	memset(mfd->moving, 0, sizeof(unsigned long)*mfd->bitpitch * h);
	mfd->skip = FALSE;
	mfd->first = TRUE;
	
	return 0;
}

#define ASM_COPY(o,i,runsize)  __asm             \
{									\
__asm			mov ecx, runsize    \
__asm			mov esi, i          \
__asm           mov edi, o          \
__asm			rep movsd           \
}

int DeInterlace(VBitmap *source, VBitmap *dest) {

	const int		bitpitch = mfd->bitpitch;
	const long		pitch = source->pitch;
	const PixDim	w = source->w;
	const PixDim	h = source->h;
	Pixel32 *		src = source->data;
	Pixel32 *		dst = dest->data;


	if(!source || !dest)
		return 0;

	int x, y;
	long prevValue;
	Pixel32 p0, p1, p2;
	long r, g, b, r0, g0, b0;
	int *lumptr;

	/* If we are shifting field phase by one field... */
	if (mfd->fieldShift)
	{
		/* This mode is used typically to clean up PAL video which
		   has erroneouslt been digitized with the field phase off by
		   one field. The result is that the frames look interlaced,
		   but really if we can phase shift by one field, we'll get back
		   the original progressive frames. This code does that and then
		   skips motion processing. */
		/* Copy the even field of the current frame to the output. */
		for (y = 0; y < h/2; y++)
		{
			//memcpy(dst, src, w*4);
			ASM_COPY(src, dst, w);

			src = (Pixel *)((char *)src + 2 * source->pitch);
			dst = (Pixel *)((char *)dst + 2 * source->pitch);
		}
		/* If this is not the first frame, copy the buffered odd field
		   of the last frame to the output. This creates a correct progressive
		   output frame. If this is the first frame, a buffered field is not
		   available, so interpolate the odd field from the current even field. */
		if (mfd->first == TRUE)
		{
			src = source->data;
			dst = (Pixel *)((char *)dest->data + dest->pitch);
			for (y = 0; y < h/2; y++)
			{
				//memcpy(dst, src, w*4);
				
				ASM_COPY(dst,src , w);

				src = (Pixel *)((char *)src + 2 * source->pitch);
				dst = (Pixel *)((char *)dst + 2 * source->pitch);
			}
			mfd->first = FALSE;
		}
		else
		{
			lumptr = mfd->prevFrame + w;
			dst = (Pixel *)((char *)dest->data + dest->pitch);
			for (y = 0; y < h/2; y++)
			{
				//memcpy(dst, lumptr, w*4);
				
				ASM_COPY(dst, lumptr, w);

				lumptr += 2 * w;
				dst = (Pixel *)((char *)dst + 2 * dest->pitch);
			}
		}
		/* Finally, save the odd field of the current frame in the buffer.
		   It will be used to creat the next frame. */
		src = (Pixel *)((char *)source->data + source->pitch);
		lumptr = mfd->prevFrame + w;
		for (y = 0; y < h/2; y++)
		{
			//memcpy(lumptr, src, w*4);
			
			ASM_COPY(lumptr, src, w);

			lumptr += 2 * w;
			src = (Pixel *)((char *)src + 2 * source->pitch);
		}
		return 0;
	}

	/* End special mode code. Doing full motion-adaptive deinterlacing. */

	if (h<2) return 0;

	// Compute differences for all pixels by comparing each pixel
    // to its corresponding pixel in the previous frame. Then do a threshold
    // test of the difference to decide if the pixel is 'moving'.
	// Then create an array of flags indicating pixels that have moved since
	// the last frame.

	lumptr = mfd->prevFrame;
	unsigned long *maskptr = mfd->moving + mfd->bitpitch;

	for (y = 1; y < h - 1; y++)
	{
		long pixmask = 0;

		src = (Pixel *)((char *)src + source->pitch);

		x = 0;
		do
		{
			// Set the moving flag if the diff exceeds the configured
            // threshold.
			pixmask <<= 1;
			prevValue = *lumptr;
			*lumptr++ = src[x];
			b = (src[x] & 0xff);
			b0 = (prevValue & 0xff);
			if (abs(b - b0) > mfd->threshold)
				goto moving;
			r = (src[x] & 0xff0000) >> 16;
			r0 = (prevValue & 0xff0000) >> 16;
			if (abs(r - r0) > mfd->threshold)
				goto moving;
			g = (src[x] & 0xff00)   >> 8;
			g0 = (prevValue & 0xff00)   >> 8;
			if (abs(g - g0) > mfd->threshold)
				goto moving;			
			goto notmoving;
moving:
			pixmask |= 1;
notmoving:
			if (!(++x & 31))
				*maskptr++ = pixmask;
		} while(x<w);

		if (x & 31)
			*maskptr++ = pixmask << (-x&31);
	}

	// Remove combing from the motion areas and render.
    // The first line gets a free ride.
	src = source->data;
	dst = dest->data;
	//memcpy(dst, src, w*4);

	ASM_COPY(dst, src, w);

	maskptr = mfd->moving;

	for (y = 1; y < h - 1; y++)
	{
		unsigned long mask;

		src = (Pixel *)((char *)src + source->pitch);
		dst = (Pixel *)((char *)dst + dest->pitch);

		if (mfd->motionOnly) {
			if (mfd->Blend) {
				x = 0;
				do {
					if (!(x & 31)) {
						mask = maskptr[bitpitch] | maskptr[0] | maskptr[bitpitch*2];
						++maskptr;
					}

					if ((signed long)mask >= 0)
						dst[x] = 0;
					else {
						
						/* Blend fields. */
						p0 = src[x];
						p0 &= 0x00fefefe;

						p1 = ((Pixel32 *)(((char *)src)-pitch))[x];
						p1 &= 0x00fcfcfc;

						p2 = ((Pixel32 *)(((char *)src)+pitch))[x];
						p2 &= 0x00fcfcfc;

						dst[x] = (p0>>1) + (p1>>2) + (p2>>2);
					}
					mask <<= 1;

				} while(++x<w);
			} else {
				x = 0;
				do {
					if (!(x & 31)) {
						mask = maskptr[bitpitch] | maskptr[0] | maskptr[bitpitch*2];
						++maskptr;
					}

					if ((signed long)mask >= 0)
						dst[x] = 0x0;
					else if (y&1) {
						p1 = ((Pixel32 *)(((char *)src)-pitch))[x];
						p1 &= 0x00fefefe;

						p2 = ((Pixel32 *)(((char *)src)+pitch))[x];
						p2 &= 0x00fefefe;
						dst[x] = (p1>>1) + (p2>>1);
					} else
						dst[x] = src[x];

					mask <<= 1;

				} while(++x<w);
			}
		} else {
			if (mfd->Blend) {
				x = 0;
				do {
					if (!(x & 31)) {
						mask = maskptr[bitpitch] | maskptr[0] | maskptr[bitpitch*2];
						++maskptr;
					}

					if ((signed long)mask >= 0)
						dst[x] = src[x];
					else {

						/* Blend fields. */
						p0 = src[x];
						p0 &= 0x00fefefe;

						p1 = ((Pixel32 *)(((char *)src)-pitch))[x];
						p1 &= 0x00fcfcfc;

						p2 = ((Pixel32 *)(((char *)src)+pitch))[x];
						p2 &= 0x00fcfcfc;

						dst[x] = (p0>>1) + (p1>>2) + (p2>>2);
					}
					mask <<= 1;

				} while(++x<w);
			} else {
				// Doing line interpolate. Thus, even lines are going through
				// for moving and non-moving mode. Odd line pixels will be subject
				// to the motion test.
				if (y&1) {
					x = 0;
					do {
						if (!(x & 31)) {
							mask = maskptr[bitpitch] | maskptr[0] | maskptr[bitpitch*2];
							++maskptr;
						}

						if ((signed long)mask >= 0)
							dst[x] = src[x];
						else {
							p1 = ((Pixel32 *)(((char *)src)-pitch))[x];
							p1 &= 0x00fefefe;

							p2 = ((Pixel32 *)(((char *)src)+pitch))[x];
							p2 &= 0x00fefefe;

							dst[x] = (p1>>1) + (p2>>1);
						}

						mask <<= 1;

					} while(++x<w);
				} else {
					// Even line; pass it through.
					//memcpy(dst, src, w*4);

					ASM_COPY(dst, src, w);

					maskptr += bitpitch;
				}
			}
		}

	}
	
	// The last line gets a free ride.
	src = (Pixel *)((char *)src + source->pitch);
	dst = (Pixel *)((char *)dst + dest->pitch);
	memcpy(dst, src, w*4);

	return 0;
}

int StopDeinterlacer() {

	delete[] mfd->prevFrame;	mfd->prevFrame = NULL;
	delete[] mfd->moving;		mfd->moving = NULL;

	return 0;
}