//========================================================================================
//
// prRt.h
//
// Part of the Adobe Premiere 5 Plug-In Developer's Toolkit.
//
// Defines the Premiere real-time structures from the timeline interface. The API defined
// in this header is valid only with the RT version of Premiere.
//
// Copyright  1992-99, Adobe Systems Incorporated, all rights reserved worldwide.
//
//========================================================================================


#ifndef __PRRT
#define __PRRT

#include "prSetEnv.h"

typedef unsigned long prtColor;

typedef struct
{
	long		x;						//	x coordinate
	long		y;						//	y coordinate
} prtPoint;


///////////////////////////////////////////////////
// This structure defines timebase information
//   the timebase information defines how trim
//   points should be interpreted in order to
//   be converted into real-world values
typedef struct
{  
   long  scale;
   long  sampleSize;  // scale divided by sampleSize is the timebase
} prtTimebase;


///////////////////////////////////////////////////
// This structure defines filename information
typedef struct {
   long		volID;		// used on Mac only
   long		parID;		// used on Mac only
   char		name[256];	// file name on Mac, full path elsewhere
} prtFileSpec;


///////////////////////////////////////////////////
// This structure defines file playback info.
//    This basically contains the filename and
//    defines the trim-in and trim-out of the file.
//    This struct is part of the tree that defines
//    a segment and is passed to the playback module
//    through the playmod_ListAddRTVideoSegment function.
typedef struct
{
   long           size;          // the size of this struct
   long           version;       // the version of the file rec struct
   prtFileSpec    fileSpec;      // filename
   long           type;          // file type id - four char code 
   long           subtype;       // file subtype - four char code
   long           singleFrame;   // non-zero if file is static
   long           trimIn;        // source in point (in file's timebase)
   long           trimInListPos; // in point's position in timeline
   double         speed;         // rate of playback
   prtTimebase    timebase;      // timebase of the source file
   Handle         fileSpecs;     // private data previously generated by the 
                                 //    import module of the file (NULL if none)
   Handle         clipData;      // clip data specific to Premiere
   long           total;         // length of whole clip in timeline
} prtFileRec;


// forward declaration of the clip structure
struct prtClipRecTag; 
typedef struct prtClipRecTag prtClipRec, *prtClipRecPtr;


///////////////////////////////////////////////////
// This structure defines a keyframe
//    A keyframe is part of the filter structure
typedef struct
{
   long     part;       // position of KF relative to start of filter
   Handle   keySpecs;   // Handle to keyframe-specific data
} prtKeyframeRec, *prtKeyframeRecPtr;



///////////////////////////////////////////////////
// This structure defines a filter
//    An filter is an operator that is 
//    performed on one clip, and contains
//    plug-in specific data (Premiere acts as a
//    container).
//    This struct is part of the tree that defines
//    a segment and is passed to the playback module
//    through the playmod_ListAddRTVideoSegment function.
typedef struct
{
   long              size;          // size of this structure
   long              version;       // The version of the filter struct
   char              name[256];     // filter type name (match name)
   prtClipRecPtr     clip;          // The operand of the filter (what it's applied to)
   long              startListPos;  // position of filter start point in timeline
   long              total;         // length of filter in timeline
   long              nbKeyframes;   // number of items in kfList
   prtKeyframeRecPtr kfList;        // array of keyframes
} prtFilterRec;


///////////////////////////////////////////////////
// This structure defines a transition
//    An transition is an operator that is 
//    performed on two clips, and contains
//    plug-in specific data (Premiere acts as a
//    container).
//    This struct is part of the tree that defines
//    a segment and is passed to the playback module
//    through the playmod_ListAddRTVideoSegment function.
typedef struct
{
   long           size;          // size of this structure
   long           version;       // The version of the transition struct
   char           name[256];     // transition type name (match name)
   prtClipRecPtr  clipA;         // Clip at the initial state of the transition (incoming)
   prtClipRecPtr  clipB;         // Clip at the final state (outgoing)
   unsigned char  arrowFlags;    // flags for direction arrow
   long           source;        // are the sources reversed (B->A)?
   long           reverse;       // is the effect running in reverse (end->start)?
   long           edgeAlias;     // edges being anti-aliased?
   long           startPercent;  // start percentage (0.00-100.00)
   long           endPercent;    // end percentage (0.00-100.00)
   long           edgeThick;     // edge thickness (.00)
   prtColor       edgeColor;     // color for the border
   prtPoint       startPoint;    // starting point
   prtPoint       endPoint;      // ending point
   prtPoint       centerPoint;   // center reference point
   long           startListPos;  // position of start point in timeline
   long           total;         // length of whole transition in timeline
   Handle         fxSpecs;       // plug-in specific data (specsHandle)
} prtTransitionRec;


///////////////////////////////////////////////////
// This struct defines a layer
//     A layer simply points to a clip, and is
//     part of a double-linked list of layers
//     pointed to by a prtSegmentRec structure.
typedef struct _prtLayerRec
{
   long                 size;       // size of this structure
   long                 version;    // version of this struct
   long                 index;      // index of the layer in timeline (0..n)
   prtClipRecPtr        clip;       // Points to an effect/file clip
   struct _prtLayerRec* layerAbove; // The layer that is above this one
   struct _prtLayerRec* layerBelow; // The layer that is below this one
} prtLayerRec, *prtLayerRecPtr;


//////////////////////////////////////////////////
// This struct defines a segment
//     A segment contains a series of layers.
//     it is passed during playmod_ListAddRTVideoSegment.
//     The structure points to a list of layers,
//     which in turn are the root of a tree defining
//     the source clips and effects to be performed
//     in the layer.

// A segment that has the following value in the "outListPos" field is
//  usually a NULL segment that signifies the end of the timeline
//  (no more material to the right of the "inListPos" field of the 
//  segment).  The segment does not have to be kept by the playback 
//  module.
#define PRT_END_OF_TIMELINE      0x0FFFFFFF

typedef struct
{
   long           size;          // size of this structure
   long           version;       // the structure/tree version.
   long           inListPos;     // start point in the play list
   long           outListPos;    // ending point in the play list (inclusive)
   prtTimebase    timebase;      // timebase of the segment
   long           nbLayers;      // The number of layers in this segment
   prtLayerRecPtr layerTop;      // The top layer
   prtLayerRecPtr layerBottom;   // The bottom layer
} prtSegmentRec, *prtSegmentRecPtr;


///////////////////////////////////////////////////
// This structure defines a clip.
//    A clip can either be defined as a transition,
//    a filter, a source file, or as a reference
//    to a segment (virtual clip).
//    It is used in a tree describing a segment to
//    the playback module, and is sent through
//    the playmod_ListAddRTVideoSegment function.

#define PRT_CLIPTYPE_FILE            0     // clip is a source file
#define PRT_CLIPTYPE_FILTER          1     // clip is a filter
#define PRT_CLIPTYPE_TRANSITION      2     // clip is a transition
#define PRT_CLIPTYPE_SEGMENT         3     // clip is a segment (virtual clip)

// special flags for the "flags" field
// clip is the one currently being edited or executed
#define PRT_CLIPFLAG_HAS_FOCUS            0x00000001  

// "Frame Hold Options -> Hold On" flag, also used
//  for "Frame Hold Options -> Hold Filters" flag.
// NOTE: When this flag is set, the "holdOnPos" field of 
//  the prtClipRec struct specifies the frame to "hold".
#define PRT_CLIPFLAG_HOLD_ON              0x00000002

// "Frame Hold Options -> Alternate Rate" flag.
//  NOTE: When this flag is set, the "altRate" field
//  specifies the alternate frame rate.
#define PRT_CLIPFLAG_ALTERNATE_RATE       0x00000004

// "Frame Hold Options -> Frame Blending" flag.
#define PRT_CLIPFLAG_FRAME_BLEND          0x00000008

// "Field Options -> Reverse field dominance" flag.
#define PRT_CLIPFLAG_REVERSE_FIELDS       0x00000010

// "Field Options -> Interlace consecutive frames" flag.
#define PRT_CLIPFLAG_INTERLACE_FRAMES     0x00000020

// "Field Options -> Always deinterlace" flag.
// (same as "Frame Hold Options -> Deinterlace" flag)
#define PRT_CLIPFLAG_ALWAYS_DEINTERLACE   0x00000040

// "Field Options -> Flicker removal" flag.
#define PRT_CLIPFLAG_CONVOLVE             0x00000080

// "Field Options -> Deinterlace when speed < 100%" flag
#define PRT_CLIPFLAG_DEINTERLACE_SLOW     0x00000100

// "Maintain Aspect Ratio" flag
#define PRT_CLIPFLAG_MAINTAIN_ASPECT      0x00000200


struct prtClipRecTag
{
   long                 type;       // Type of clip (see values above)
   union
   {
      // only one of the following is used, depending on the value of "type"
      prtFileRec        file;       // The source file description
      prtFilterRec      filter;     // The filter description
      prtTransitionRec  transition; // The transition description
      prtSegmentRec     segment;    // The segment description (virtual clip)
   };

   // additional data
   long                 flags;      // special flags defined above
   long                 altRate;    // alternate frame rate for the clip (fps)
                                    // Used only with PRT_CLIPFLAG_ALTERNATE_RATE 
                                    // flag.
   long                 holdOnPos;  // "holding position" of the clip.
                                    // This field is used only when the 
                                    // PTR_CLIPFLAG_HOLD_ON flag is specified.
                                    // The specified position is relative to the
                                    // starting position of the clip in the timeline
                                    // and is given in the clip's timebase.
   prtColor             fillColor;  // filling color for clips that should maintain 
                                    //  aspect ratio
};


//////////////////////////////////////////////////
// This struct defines a playable RT region
//    A playable region defines a region that
//    can either be played live or not by the playback
//    module.  It is used during the 
//    playmod_GetPlayableRTRange function call.
//  NOTE: If the queried frame falls past the end of
//        the current cut-list, the playback module
//        should return the PRT_END_OF_TIMELINE value
//        in the "outListPos" field.  In this case, 
//        the other fields are considered irrelevant.

// This code specifies that the range is playable in real-time
#define PRT_PLAYCODE_REALTIME                   0

// This code specifies that the region cannot be played in 
// real-time for unspecified reasons
#define PRT_PLAYCODE_NON_REALTIME_UNSPECIFIED   1

// This code specifies that the range has too many layers
// to be played in real-time
#define PRT_PLAYCODE_TOO_MANY_LAYERS            2

// This code specifies that the range has too many instances
// of a specific filter to be performed in real-time
#define PRT_PLAYCODE_TOO_MANY_FILTERS           3

// This code specifies that the range has too many instances
// of a specific transition to be performed in real-time
#define PRT_PLAYCODE_TOO_MANY_TRANSITIONS       4

// This code specifies that a specific file type cannot be 
// played in real-time
#define PRT_PLAYCODE_UNKNOWN_FILE_TYPE          5

// This code specifies that a specific filter type cannot be 
//  performed in real-time
#define PRT_PLAYCODE_UNKNOWN_FILTER_TYPE        6

// This code specifies that a specific transition type cannot
// be performed in real-time
#define PRT_PLAYCODE_UNKNOWN_TRANSITION_TYPE    7

// This code specifies that certain clip flags are not supported
#define PRT_PLAYCODE_UNSUPPORTED_FLAGS          8

// This code says that the specified clip speed is not supported
#define PRT_PLAYCODE_UNSUPPORTED_SPEED          9

// This code specifies that a virtual clip cannot be accelerated
#define PRT_PLAYCODE_UNSUPPORTED_VIRTUAL        10

typedef struct
{
// the following members are filled by Premiere before the call
   long size;              // size of this structure
   long version;           // the version of the struct (set to 1)
   long frame;             // frame being tested
   prtTimebase timebase;   // timebase of "frame"

// the following members are filled by the playback module
   long inListPos;         // in point in the play list
   long outListPos;        // out point in the play list (inclusive)
   long playcode;          // see enum above
   long layerIndex;        // (optional) index of affected layer
} prtPlayableRangeRec, *prtPlayableRangePtr;


//////////////////////////////////////////////////
// This struct defines a frame that should be rendered
//    by the playback module, if possible.
//    If the playback module answer playmod_ErrNone
//    to the message, it is assumed that the 
//    "destination" image has been filled by the
//    playback module
// This struct is used in the playmod_RenderRTFrame call

// render only the even field
#define PRT_RENDER_FIELD_EVEN    0x00000001
// render only the odd field
#define PRT_RENDER_FIELD_ODD     0x00000002

typedef struct
{
   long           size;          // size of this structure
   long           version;       // the structure version.
   long           frame;         // the frame to render
   prtTimebase    timebase;      // timebase of "frame"
   long           flags;         // see values above
   long           nbLayers;      // The number of layers in this frame
   prtLayerRecPtr layerTop;      // The top layer
   prtLayerRecPtr layerBottom;   // The bottom layer
   prRect         bounds;        // the size of the destination image
   PPixHand       destination;   // the destination image
} prtRenderFrameRec, *prtRenderFrameRecPtr;


#include "prResetEnv.h"
#endif //__PRRT
