/* 
 *  PostProcessing.cpp 
 *
 *	Copyright (C) Alberto Vigata - January 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <windows.h>
#include "flaskmpeg.h"
#include ".\resizer\vbitmap.h"
#include ".\resizer\resizer.h"
#include ".\Video\Deinterlacer\Deinterlacer.h"
#include ".\postprocessing.h"

struct TPostRun{
	int			 cropWidthFlag;
	int			 cropHeightFlag;
	Pixel32      *buffer;
	VBitmap      temp;
	VBitmap      deinterlacer_output;
	Pixel32      *deinterlacer_output_data;

} PostRun;

#include ".\Video\VideoWrapper.h"
extern VideoWrapper *myVideo;

static VBitmap vbSrc, vbDst;

//Post processing functions
int PostProcessingStart(VBitmap *source, VBitmap *dest, TPPost *pp){
	

	if( !source || !dest  ||  !pp )
		return 0;
	
	PostRun.cropWidthFlag  = 0;
	PostRun.cropHeightFlag = 0;
	PostRun.buffer         = NULL;
	dest->offset = 0;
	// init resizer
	PrepareResizer(pp->resWidth, pp->resHeight, pp->doAR, pp->iDAR, pp->filterMode, source);

	if(pp->deinterlace)
	{
		TDeinterlacerConfig config;
		config.blend	= pp->blend_fields;
		config.threshold= pp->threshold;

		// Start deinterlacer
		StartDeinterlacer(source->w, source->h, &config);
		PostRun.deinterlacer_output_data = new Pixel32[source->w*source->h];
		PostRun.deinterlacer_output.init( PostRun.deinterlacer_output_data
										, source->w
										, source->h
										, 32);
		PostRun.deinterlacer_output.AlignTo4();
	}

	if( pp->crop )
		if(pp->cropHeight < pp->resHeight )
			PostRun.cropHeightFlag = 1;

	// prepare buffer if crop width
	if( pp->crop )
		if(pp->cropWidth < pp->resWidth){
			if( !(PostRun.buffer = (Pixel32 *)malloc(pp->resWidth*pp->resHeight*4)) )
				return 0;
			PostRun.cropWidthFlag=1;
		}

	return 1;
}

int PostProcess(CFrame *pFrameSrc, CFrame *pFrameDst, TPPost *pp)
{
 vbSrc.init( pFrameSrc->GetBuffer(), 
    pFrameSrc->GetWidth(), 
    pFrameSrc->GetHeigth(), 32);
  
  vbDst.init( pFrameDst->GetData(), 
    pFrameDst->GetWidth(), 
    pFrameDst->GetHeigth(), 32);

  int ret=PostProcess(&vbSrc, &vbDst, pp);
  // FIXME: dependencies with vbitmaps and offsets
  pFrameDst->SetOffset( vbDst.offset*4 );

  return ret;
}

int PostProcess(VBitmap *source_bmp, VBitmap *dest, TPPost *pp){
	int w,h,diff,firstline, width, height;
	VBitmap *source;
	Pixel32 *src, *dst;

	if( !source_bmp || !dest  ||  !pp )
		return 0;

	if(pp->deinterlace){
		DeInterlace(source_bmp, &PostRun.deinterlacer_output);
		source = &PostRun.deinterlacer_output;
	}
	else
		source = source_bmp;

	// subpic_apply(source, (double)pp->myClock/27000.0);

	// Resize
	if(PostRun.cropWidthFlag){
        PostRun.temp.init(PostRun.buffer, pp->resWidth, pp->resHeight, 32);
		ResizerRun( source, &PostRun.temp);
	}
	else if(PostRun.cropHeightFlag){
		// Run resizer over dest->data
		PostRun.temp.init(dest->data, pp->resWidth, pp->resHeight, 32);
		ResizerRun( source, &PostRun.temp);
	}
	else
		ResizerRun( source, dest);
	
	// Crop
	if( pp->crop ){
		if( PostRun.cropHeightFlag )			//Height cropping
			dest->offset = (pp->resHeight - pp->cropTopOffset - pp->cropHeight ) * pp->resWidth;
		if( PostRun.cropWidthFlag  ){
			// Width or Width & Height cropping. Resized image is at PostRun.temp
			dest->offset=0;
			h = pp->cropHeight;

			firstline = pp->resHeight - pp->cropTopOffset;
			diff      = pp->cropLeftOffset + (pp->resWidth -pp->cropWidth - pp->cropLeftOffset);
			dst       = (Pixel32 *)dest->data + pp->cropWidth * pp->cropHeight -1;
			src       = PostRun.temp.data + pp->cropLeftOffset + (firstline - 1)*pp->resWidth + pp->cropWidth - 1;


      do{
        
        w   = pp->cropWidth;
        do{
          *dst-- = *src--;
        }while(--w);
        src = src - diff;
      }while(--h);


		}
	}
	else
		dest->offset = 0;       // No cropping


	// Letterbox
	width  = pp->crop ? pp->cropWidth  : pp->resWidth;
    height = pp->crop ? pp->cropHeight : pp->resHeight;

	if(pp->letterbox){
		//Bottom bar
		if (pp->letterboxBottom>0){
			h = pp->letterboxBottom;
			dst = dest->Address();
			do {
				w = width;
				do {
					*dst++ = 0;
				} while(--w);
			} while(--h);
		}

		//Top bar

		if (pp->letterboxTop>0){
			h = pp->letterboxTop;
			// Point to last pixel
			dst = dest->Address() + width * height - 1;
			do {
				w = width;
				do {
					*dst-- = 0;
				} while(--w);
			} while(--h);
		}

		// Left Bar
		if (pp->letterboxLeft >0 ){
			h = height;
			do {
				// Point to first pixel of last row
				w = pp->letterboxLeft;
				dst = dest->Address() + (h-1) * width;
				do {
					*dst++ = 0;
				} while(--w);
			} while(--h);
		}

		// Right Bar
		if (pp->letterboxRight >0 ){
			h = height;
			do {
				w = pp->letterboxRight;
				// Point to last pixel
				dst = dest->Address() + h * width - 1;
				do {
					*dst-- = 0;
				} while(--w);
			} while(--h);
		}
	}
	// That was it

	return 1;
}

int PostProcessingStop(TPPost *pp){

	UnPrepareResizer();
	// Free horizontal crop buffer if necessary
	if(PostRun.cropWidthFlag)
		if(PostRun.buffer){
			free(PostRun.buffer);
			PostRun.buffer =NULL;
		}
	if(pp->deinterlace)
	{
		StopDeinterlacer();
		delete [] PostRun.deinterlacer_output_data;
	}


	return 0;
}


// Auxiliary Post processing functions

// Fix absurd post processing settings
void FixPPostSettings(TPPost *pp){

		if(pp->cropWidth>pp->resWidth)
			pp->cropWidth  = pp->resWidth;
		if(pp->cropHeight > pp->resHeight)
			pp->cropHeight = pp->resHeight;

}

int CheckVideoParameters(TPPost *pp){
  int w,h;
  
  //FixPPostSettings(pp);
  // Image dimension settings
  if( (pp->cropHeight%16 != 0) || (pp->cropWidth%16 != 0) )
    return 0;
  
  if (! pp->crop) { 
    // if we don't crop, the resize dims must be multipl of 16
    if( (pp->resWidth%16 != 0) || (pp->resHeight%16 != 0) )
      return 0;
  }
  else {
    // if we are cropping, the only requirement is that the resize width be
    // a multiple of 2 (needed by the bicubic code and friends)
    if (pp->resWidth%2 != 0)
      return 0;
  }
  
  //Crop Checkings
  // Crop widht&height must be less than input image resolution
  if( pp->crop && (pp->cropHeight > pp->resHeight) )
    return 0;
  if( pp->crop && (pp->cropWidth  > pp->resWidth) )
    return 0;
  if( pp->crop && ((pp->cropTopOffset + pp->cropHeight) > pp->resHeight)  )
    return 0;
  if( pp->crop && ((pp->cropLeftOffset + pp->cropWidth) > pp->resWidth)  )
    return 0;
  
  if( (pp->cropHeight < 0) || (pp->cropWidth < 0) )
    return 0;
  if( pp->crop && ((pp->cropTopOffset<0) ||  (pp->cropTopOffset > pp->cropHeight-16)) )
    return 0;
  if( pp->crop && ((pp->cropLeftOffset<0) ||  (pp->cropLeftOffset > pp->cropWidth-16)) )
    return 0;
  
  // image size
  if( pp->resWidth > 1024 || pp->resHeight > 1024)
    return 0;
  
  // Letterboxing
  w = pp->crop ? pp->cropWidth  : pp->resWidth;
  h = pp->crop ? pp->cropHeight : pp->resHeight;
  
  if( pp->letterbox && (pp->letterboxLeft>w || pp->letterboxRight >w
    || pp->letterboxTop >h || pp->letterboxBottom>h))
    return 0;
  if( pp->letterbox && (pp->letterboxLeft<0 || pp->letterboxRight <0
    || pp->letterboxTop <0 || pp->letterboxBottom<0))
    return 0;
  

	return 1;
}

void FromPPostToConfig( TProfile *prof, TPPost *pp){

  prof->crop           =   pp->crop;
  prof->cropWidth      =   pp->cropWidth;
  prof->cropHeight     =   pp->cropHeight;
  prof->cropTopOffset  =   pp->cropTopOffset;
  prof->cropLeftOffset =   pp->cropLeftOffset;
  
  prof->letterbox      =   pp->letterbox;
  prof->letterboxBottom=   pp->letterboxBottom;
  prof->letterboxTop   =   pp->letterboxTop;
  prof->letterboxLeft  =   pp->letterboxLeft;
  prof->letterboxRight =   pp->letterboxRight;
  
  prof->InterpolatedWidth   = pp->resWidth;
  prof->InterpolatedHeight  = pp->resHeight;
  prof->filter              = pp->filterMode;
  prof->keepAspect          = pp->doAR;
  
  prof->deinterlace = pp->deinterlace;
  prof->blend = pp->blend_fields;
  prof->threshold = pp->threshold;
}

void FromConfigToPPost( TProfile *prof, TPPost *pp){
	
  pp->crop          =  prof->crop;
  pp->cropWidth     =  prof->cropWidth;
  pp->cropHeight    =  prof->cropHeight;
  pp->cropTopOffset =  prof->cropTopOffset;
  pp->cropLeftOffset=  prof->cropLeftOffset;
  
  
  pp->letterbox       =  prof->letterbox;
  pp->letterboxBottom =  prof->letterboxBottom;
  pp->letterboxTop    =  prof->letterboxTop;
  pp->letterboxLeft   =  prof->letterboxLeft;
  pp->letterboxRight  =  prof->letterboxRight;
  
  pp->resWidth   = prof->InterpolatedWidth ;
  pp->resHeight  = prof->InterpolatedHeight;
  pp->filterMode = prof->filter;
  pp->doAR       = prof->keepAspect;
  
  pp->deinterlace  = prof->deinterlace;
  pp->blend_fields = prof->blend;
  pp->threshold    = prof->threshold;

}

