/* 
 *  Player.cpp 
 *
 *	Copyright (C) Alberto Vigata - July 2000 - ultraflask@yahoo.com
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "FlasKMPEG.h"
#include "auxiliary.h"
#include "resource.h"
#include "Player.h"
#include "./Misc/StatsStrings.h"
#include "./Video/VideoWrapper.h"
#include "./RunState.h"
#include <string>

// Extern globals
extern TConfig          o;
extern TRunState        rs;
extern HINSTANCE        hInst;
extern unsigned char    MPEGImage[MAX_IMAGE_MEM];
extern unsigned char    DisplayImage[MAX_IMAGE_MEM];
extern VBitmap			Display;
extern VBitmap			DecodedImage;
extern HWND				hMainWnd;
extern BITMAPINFO		DibInfo;
extern BOOL             playerStopped; //Needed for main window to redraw
									   //   FlasKMPEG logo.

extern void InitDibDisplay(int oxsize, int oysize);
extern void SetWindowSize( void );    // This sets the window size to video size
// Local variables
static HWND                 hPlayerDlg;
static HANDLE               evPlayerStopped; 
static HANDLE               evFrameCaptured;
static CRITICAL_SECTION     GlobalCriticalSection; 
static bool                 SeekInProgress;
static i64		             diferencia;
static TStatsStrings        stats;      //This is a big one
DWORD play_start_time;
unsigned play_frame_count;

const int nTrackBarRange = 1000;

void ShowPlayer()
{
  if(!rs.hPlayerWnd)
  {
    rs.hPlayerWnd = CreateDialog(hInst,MAKEINTRESOURCE(IDD_PLAYER), hMainWnd, (DLGPROC) PlayerDlg);
    ShowWindow(rs.hPlayerWnd, SW_SHOW);
  }  
}

static CFlEvent evPlayerEnded;

void HidePlayer()
{
  if(rs.hPlayerWnd)
  {
    evPlayerEnded.Reset();
    SendMessage( rs.hPlayerWnd, WM_COMMAND, MAKEWPARAM(IDOK,0), 0);
    evPlayerEnded.Wait();
  }
}

void Localize(HWND hDlg)
{
    DlgSetText(hDlg, R_PLAYER_PLAY_BUTTON,             GS(R_PLAYER_PLAY_BUTTON) );
    DlgSetText(hDlg, R_PLAYER_STOP_BUTTON,             GS(R_PLAYER_STOP_BUTTON) );
    DlgSetText(hDlg, R_PLAYER_VIDEOSTATS_BOX,          GS(R_PLAYER_VIDEOSTATS_BOX) );
    DlgSetText(hDlg, R_PLAYER_FILETIME_TITLE,          GS(R_PLAYER_FILETIME_TITLE) );
    DlgSetText(hDlg, R_PLAYER_FILE_TITLE,              GS(R_PLAYER_FILE_TITLE) );
    DlgSetText(hDlg, R_PLAYER_VIDEOSIZE_TITLE,         GS(R_PLAYER_VIDEOSIZE_TITLE) );
    DlgSetText(hDlg, R_PLAYER_FPS_TITLE,               GS(R_PLAYER_FPS_TITLE) );
    DlgSetText(hDlg, R_PLAYER_TOTALFILESIZE_TITLE,     GS(R_PLAYER_TOTALFILESIZE_TITLE) );
    DlgSetText(hDlg, R_PLAYER_ASPECTRATIO_TITLE,       GS(R_PLAYER_ASPECTRATIO_TITLE) );
    DlgSetText(hDlg, R_PLAYER_BITRATE_TITLE,           GS(R_PLAYER_BITRATE_TITLE) );
    DlgSetText(hDlg, R_PLAYER_VIDEOSTRUCTURE_TITLE,    GS(R_PLAYER_VIDEOSTRUCTURE_TITLE) );
    DlgSetText(hDlg, R_PLAYER_SEEKFIRST_BUTTON,        GS(R_PLAYER_SEEKFIRST_BUTTON) );
    DlgSetText(hDlg, R_PLAYER_DETECTEDFPS_TITLE,       GS(R_PLAYER_DETECTEDFPS_TITLE) );
    DlgSetText(hDlg, R_PLAYER_VIDEOFORMAT_TITLE,       GS(R_PLAYER_VIDEOFORMAT_TITLE) );

    return;
}



#define OUTPUT_POPUP_BASE 10000
static void ShowOutputPopup(HWND hDlg, HWND hButton)
{
  HMENU hPopup = CreatePopupMenu();
  for(int i=0; i<rs.plugs.OutPluginCount; i++)
  {
    MenuAddItem( hPopup, i, OUTPUT_POPUP_BASE+i, rs.plugs.outPlugs[i].ID );
  }
  CheckMenuItem(hPopup,rs.selected_out_plug, MF_BYPOSITION|MF_CHECKED);
  MenuAddSeparator(hPopup, i);
  MenuAddItem(hPopup, i+1, OUTPUT_POPUP_BASE+i+1, "Configure Output Module");

  // Get the rect of the button
  RECT rcButton;
  GetWindowRect(hButton, &rcButton);
  // Now show the popup
  TrackPopupMenu( hPopup, 
                  TPM_LEFTALIGN|TPM_TOPALIGN|TPM_LEFTBUTTON,
                  rcButton.right, rcButton.top,
                  0, hDlg, NULL);
  // thats it
}

void CreateDefaultJob()
{
  // If there are no jobs create one
  if(!rs.vBatchList.size())
  {
    CFrame *pDummyFrame;
    
    TBatchListElement el;
    // Calculate start position
    // Go to the beginning
    rs.video->SetStreamPos(0);
    // Decode the first frame available
    TVideoOptions sStartOpt;
    memset( &sStartOpt, 0, sizeof(TVideoOptions));
    sStartOpt.bStartInSync = false;
    sStartOpt.idctIndex    = rs.prof.idctIndex;
    sStartOpt.pFrameBuffer = NULL;
    
    rs.video->Start(&sStartOpt);
    rs.video->GetNextFrame(&pDummyFrame);
    rs.video->Stop();
    
    el.nStart = rs.video->GetLastDecodedFrameStart();
    el.nEnd   = rs.video->GetStreamSize();
    rs.vBatchList.push_back(el);
  }  
}

static void DeInitializePlayer(HWND hDlg, CPlayerThread *pPlayer )
{
  // Stop playing thread
  pPlayer->Destroy();
  
  // Create default job
  CreateDefaultJob();
  
  PostProcessingStop(&rs.pp);
  KillTimer( hDlg, 2);
  SetWindowSize();
  InvalidateRect(hMainWnd, NULL, true);
}

void RenderBatchList( HWND hList )
{
  int i, nSize = rs.vBatchList.size();
  char str[256];

  SendMessage(hList, LB_RESETCONTENT, 0, 0);
  for(i=0; i<nSize; i++)
  {
    int nJobSize = (rs.vBatchList[i].nEnd - rs.vBatchList[i].nStart)>>20;
    sprintf( str, "Job %d. size %d MB.", i+1, nJobSize );
    SendMessage(hList, LB_ADDSTRING, 0, (WPARAM)str);
  }
}

void RenderTrackbarSel( HWND hTrackbar , TBatchListElement *pItem )
{
  LONG start = (LONG)(((double)pItem->nStart/(double)rs.video->GetStreamSize()) * (double)nTrackBarRange);
  LONG end   = (LONG)(((double)pItem->nEnd  /(double)rs.video->GetStreamSize()) * (double)nTrackBarRange);

  SendMessage( hTrackbar, TBM_SETSEL, (WPARAM)(BOOL)true, (LPARAM)MAKELONG(start, end) );
}

LRESULT CALLBACK PlayerDlg(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	static HANDLE hPlayer;
  i64 nFilePos;
  int nSel;
	static int playerTimer;
	int fPosition, lPosition;
	static bool userIsTracking;
    char szTemp[1024];
  static CPlayerThread *pPlayer;
  static TBatchListElement trackBarSpan;
LONG nPlayFrameCount;
  CPlayerThread::playerStates nState;


	switch (message)
	{
		case WM_TIMER:
			// time
      RenderStatsStrings(&stats);
			MillisecondsToTime(szTemp, rs.video->time);
			SetDlgItemText(hDlg, IDC_PLAYTIME, szTemp);
			// Total frames
			// File
			DlgSetText(hDlg, IDC_FILE, rs.video->GetFileName());
			// Total file size
			sprintf( szTemp, "%d Mbytes", (unsigned)(rs.video->GetStreamSize() >> 20));
			SetDlgItemText(hDlg, IDC_FILESIZE, szTemp);
			// Video size
			DlgSetText(hDlg,IDC_VIDEOSIZE ,stats.InMedia.video_size);
			// Frame rate
			DlgSetText(hDlg,IDC_FRAMERATE ,stats.InMedia.video_frame_rate);
			//Detected FPS
			DlgSetText(hDlg,IDC_DETECTEDFPS ,stats.InMedia.video_detected_frame_rate);
			// Aspect Ratio
			DlgSetText(hDlg,IDC_ASPECTRATIO ,stats.InMedia.video_aspect_ratio_information);
			// Bitrate
			DlgSetText(hDlg,IDC_BITRATE ,stats.InMedia.video_bit_rate);
			// Progressive sequence
			DlgSetText(hDlg,IDC_PROGRESSIVE ,stats.InMedia.video_structure);
			//DlgSetText(hDlg,IDC_PROGRESSIVE ,video->internalPTS/27000.0);
			//DlgSetText(hDlg,IDC_BITRATE ,diferencia);

      nPlayFrameCount = pPlayer->GetPlayFrameCount();
      nState = pPlayer->GetState();

      if (nState==CPlayerThread::stPlaying) {
				DlgSetText(hDlg, IDC_FRAMES, nPlayFrameCount);
				sprintf( szTemp, "%.02f", nPlayFrameCount/(0.001*(GetTickCount()-play_start_time)));
				DlgSetText(hDlg, IDC_FPS, szTemp);
			}

			fPosition=true;
			lPosition= (int)(((double)(i64)(rs.video->GetStreamPos())/(double)(i64)(rs.video->GetStreamSize()))*nTrackBarRange);
			if( !userIsTracking && 
          nState!=CPlayerThread::stNextFrame  &&
          nState!=CPlayerThread::stPrevFrame )
				SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_SETPOS, (WPARAM) (BOOL) fPosition,(LPARAM) (LONG) lPosition); 

			break;
		case WM_INITDIALOG:
			//myClock=0;
			//frameSpan= (i64)((1/((double)(rs.prof.timeBase.scale)/
			//		 (double)(rs.prof.timeBase.sampleSize)))*(double)MPEG2_CLK_REF);

      Localize(hDlg);

      // put the audio in place
      rs.video->SetStreamPos(rs.startFilePos);
			if(rs.audio)
				rs.audio->SetStreamPos(rs.startFilePos);
	
			// Get post processing options from config
			FromConfigToPPost(&rs.prof, &rs.pp);
			// Set additional parameters
			rs.pp.iDAR = rs.video->DAR;
			InitDibDisplay(PPOST_WIDTH(rs.pp) , PPOST_HEIGHT(rs.pp));

      
      // FIXME remove dependencies from VBitmap
      {
        VBitmap vbIn;
        VBitmap vbOut;
        vbIn.init(NULL, rs.video->pictureWidth, rs.video->pictureHeight, 32);
        vbOut.init(NULL, PPOST_WIDTH(rs.pp), PPOST_HEIGHT(rs.pp), 32);
		    // Start Post processing
		    PostProcessingStart(&vbIn, &vbOut, &rs.pp);
      }

			rs.startFilePos = 0;

      hPlayerDlg=hDlg;
      SeekInProgress=false;
      userIsTracking=false;
      EnableWindow( GetDlgItem(hDlg, IDC_PLAY), true); 
      EnableWindow( GetDlgItem(hDlg, IDOK), true); 
      EnableWindow( GetDlgItem(hDlg, IDC_SEEKFIRST), true); 
      EnableWindow( GetDlgItem(hDlg, IDC_SLIDER), true); 
      EnableWindow( GetDlgItem(hDlg, IDC_STOP), true); 
      SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_SETRANGE, (WPARAM) TRUE,(LPARAM) MAKELONG(0, nTrackBarRange)); 
            
			evPlayerStopped=CreateEvent(NULL, TRUE, FALSE,NULL);
			evFrameCaptured=CreateEvent(NULL, TRUE, FALSE,NULL);
			
			play_frame_count = 0;
			play_start_time = GetTickCount();

			playerTimer= SetTimer(hDlg,              // handle of window for timer messages
								     2,          // timer identifier
								   500,           // time-out value
								   NULL   // address of timer procedure
								   );

      // Snap the player at the botton of the main window
      RECT rcMain;
      GetWindowRect( hMainWnd, &rcMain );
      WindowMove( hDlg, rcMain.left, rcMain.bottom );

      // Create default job
      CreateDefaultJob();

      // Render list
      RenderBatchList(GetDlgItem(hDlg, IDC_LIST));

      // Start the renderer
      rs.pVideoRenderer->StartPlaying();
      


      // Create player thread
      pPlayer = new CPlayerThread( rs.video, hMainWnd );
      pPlayer->Create();

      // If there are jobs select jump to the first one.
      if(rs.vBatchList.size())
      {
        trackBarSpan = rs.vBatchList[0];
        pPlayer->Seek( trackBarSpan.nStart );
        RenderTrackbarSel(GetDlgItem(hDlg, IDC_SLIDER), &trackBarSpan );
      }
      
				return TRUE;
		case WM_HSCROLL:
			int nPos;
			switch(LOWORD(wParam)){
				case TB_THUMBTRACK:
          // Get position
          // No need to serialize this.
          userIsTracking = true;

          nPos = SendDlgItemMessage(hDlg, IDC_SLIDER, TBM_GETPOS, 0, 0);   
          
          nFilePos=(i64)(((double)nPos/1000.0)*(double)(i64)(rs.video->GetStreamSize()));
          
          // Seek the player to that position
          pPlayer->SeekKeyFrame( nFilePos );
          
          play_frame_count = 0;
          play_start_time = GetTickCount();
          
          break;
				case TB_THUMBPOSITION :
          userIsTracking = false;
				break;
			}
			break;
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) 
			{
        DeInitializePlayer(hDlg, pPlayer);
        // Invalidate handle of the player dialog
        rs.hPlayerWnd = NULL;
				EndDialog(hDlg, LOWORD(wParam));
        // Set end dialog event to wake anyone waiting on HidePlayer
        evPlayerEnded.Set();
        // Stop the renderer
        rs.pVideoRenderer->StopPlaying();
				return TRUE;
			}
      WORD wmId = LOWORD(wParam);
      // Customized menues
      if(wmId>=OUTPUT_POPUP_BASE && wmId<(OUTPUT_POPUP_BASE+MAX_OUT_PLUGINS+2))
      {
        if((wmId - OUTPUT_POPUP_BASE) < rs.plugs.OutPluginCount)
          rs.selected_out_plug = wmId - 10000;
        else if( (wmId - OUTPUT_POPUP_BASE) == rs.plugs.OutPluginCount + 1 ){
          SendMessage( hMainWnd, WM_COMMAND, MAKEWPARAM(IDM_MPEG1OPTIONS,0), 0 );
        }
      }

			switch( LOWORD(wParam)){
        case IDC_FLASKIT:
          SendMessage( hMainWnd, WM_COMMAND, MAKEWPARAM(IDM_START2,0), 0 );       
          break;
        case IDC_AUDIOPLAYER:
          SendMessage( hMainWnd, WM_COMMAND, MAKEWPARAM(IDM_AUDIOPLAY,0), 0 );       
          break;
        case IDC_CONFIGURE:
          SendMessage( hMainWnd, WM_COMMAND, MAKEWPARAM(IDM_OUTPUTCONFIG,0), 0 );
          break;
        case IDC_SELECTOUTPUT:
          ShowOutputPopup(hDlg, GetDlgItem(hDlg, IDC_SELECTOUTPUT));
          break;
        case IDC_SEEKFIRST:
          pPlayer->SeekBeginning();
				break;
        case IDC_SEEKEND:
          pPlayer->SeekEnd();
          break;
        case IDC_PLAYBACK:
          pPlayer->PlayBack();
          break;
				case IDC_PLAY:
          pPlayer->Play();
					break;
				case IDC_STOP:
          pPlayer->Stop();
					break;
        case IDC_NEXTFRAME:
          pPlayer->NextFrame();
          break;
        case IDC_PREVFRAME:
          pPlayer->PrevFrame();
          break;
        case IDC_FASTFOR:
          pPlayer->FastForward();
          break;
        case IDC_FASTBACK:
          pPlayer->FastRewind();
          break;
        case IDC_STARTMARKER:
          trackBarSpan.nStart = pPlayer->GetFrameStartPos();
          if(trackBarSpan.nStart > trackBarSpan.nEnd)
          trackBarSpan.nEnd   = rs.video->GetStreamSize();
          RenderTrackbarSel(GetDlgItem(hDlg, IDC_SLIDER), &trackBarSpan );
          break;
        case IDC_ENDMARKER:
          trackBarSpan.nEnd = rs.video->GetStreamPos();
          if(trackBarSpan.nEnd < trackBarSpan.nStart)
            trackBarSpan.nStart = 0;
          
          RenderTrackbarSel(GetDlgItem(hDlg, IDC_SLIDER), &trackBarSpan );
          break;
        case IDC_GOSTARTMARKER:
          pPlayer->Seek( trackBarSpan.nStart );
          break;
        case IDC_GOENDMARKER:
          pPlayer->Seek( trackBarSpan.nEnd );
          break;
        case IDC_LISTADD:
          rs.vBatchList.push_back(trackBarSpan);
          RenderBatchList(GetDlgItem(hDlg, IDC_LIST));
          break;
        case IDC_LISTDELETEALL:
          rs.vBatchList.clear();
          RenderBatchList(GetDlgItem(hDlg, IDC_LIST));                      
          break;
        case IDC_LISTDELETE:
          nSel = SendMessage(GetDlgItem(hDlg, IDC_LIST), LB_GETCURSEL, 0, 0 );
          if(nSel!=LB_ERR)
          {
            rs.vBatchList.erase( rs.vBatchList.begin() + nSel );
            RenderBatchList(GetDlgItem(hDlg, IDC_LIST));            
          }
          break;
        case IDC_LIST:
          if(HIWORD(wParam)==LBN_SELCHANGE)
          {
            nSel = SendMessage(GetDlgItem(hDlg, IDC_LIST), LB_GETCURSEL, 0, 0 );
            trackBarSpan = rs.vBatchList[nSel];
            RenderTrackbarSel(GetDlgItem(hDlg, IDC_SLIDER), &trackBarSpan);
            pPlayer->Seek( trackBarSpan.nStart );
          }
          break;
      }
      break;
	}
    return FALSE;
}

// 
// Actual running thread
DWORD CPlayerThread::ThreadProc()
{
  m_nState = stStopped;

  ui64 size;
	TVideoOptions video_opt;
  CFrame *pDecodedFrame;
  CFrame frProcessedFrame(NULL);

  // FIXME :  the buffer size of post processed frames has to be larger than the final size
  frProcessedFrame.Set( rs.pp.resWidth , rs.pp.resHeight, FRAME_RGB, 32, 0 );
  frProcessedFrame.SetSize( PPOST_WIDTH(rs.pp), PPOST_HEIGHT(rs.pp));

  // Update video options
  video_opt.idctIndex            = rs.prof.idctIndex;
	video_opt.recons_progressive   = rs.prof.recons_progressive;
  video_opt.bStartInSync         = false;
  video_opt.pFrameBuffer         = new CListFrameBuffer( rs.video->pictureWidth,
                                                           rs.video->pictureHeight,
                                                           FRAME_RGB,
                                                           32, 3);
  pDecodedFrame  = NULL;
  while(1)
  {

    // open scope to lock command
    {
      CFlAutoLock commandLock(&m_csCommand);
      m_evCommandCompleted.Reset();
      switch( m_nCommand )
      {
      case noCommand:
        break;
      case seek:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        m_pVideoDecoder->SetStreamPos(m_nParam);
        m_pVideoDecoder->Start(&video_opt);
        m_nState = stNextFrame;
        break;
      case seekKeyFrame:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        m_pVideoDecoder->SetStreamPos(m_nParam);
        m_pVideoDecoder->Start(&video_opt);
        m_nState = stNextKeyFrame;
        break;
      case seekBeginning:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        m_pVideoDecoder->SetStreamPos(0);
        m_pVideoDecoder->Start(&video_opt);
        m_nState = stNextFrame;
        break;
      case seekEnd:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Go to end
        size = m_pVideoDecoder->GetStreamSize();
        m_pVideoDecoder->SetStreamPos( size );
        // Start video
        m_pVideoDecoder->Start(&video_opt);
        m_nState = stPrevFrame;
        break;
      case nextFrame:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stNextFrame;
        break;
      case prevFrame:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stPrevFrame;
        break;
      case nextKeyFrame:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stNextKeyFrame;
        break;
      case prevKeyFrame:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stPrevKeyFrame;
        break;
      case fastForward:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stFastForward;
        break;    
      case fastRewind:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stFastRewind;
        break;
      case getFrameStartPos:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        m_nState = stStopped;
        m_nCommandResult = m_pVideoDecoder->GetLastDecodedFrameStart();
        break;
      case play:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
          m_nState = stPlaying;
        m_lPlayFrameCount = 0;
        break;
      case playBack:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        // Start video
        m_pVideoDecoder->Start(&video_opt);
        m_nState = stPlayingBack;
        break;
      case stop:
        // Stop if necessary
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        m_nState = stStopped;
        break;
      case exit:
        if(m_nState!=stStopped)
          m_pVideoDecoder->Stop();
        
        if(video_opt.pFrameBuffer)
          delete video_opt.pFrameBuffer;
        
        // Signal exit event
        m_evFinish.Set();
        // Exit
        return 0;
        break;
      }
      m_nCommand = noCommand;
      m_evCommandCompleted.Set();
    }
// Perform action according the state of the playe


    switch(m_nState)
    {
    case stStopped:
      break;
    case stPlaying:
      m_pVideoDecoder->GetFrame(&pDecodedFrame);
      m_lPlayFrameCount++;
      break;
    case stPlayingBack:
      m_pVideoDecoder->GetPrevFrame(&pDecodedFrame);
      break;
    case stNextFrame:
      m_pVideoDecoder->GetNextFrame(&pDecodedFrame);
      m_pVideoDecoder->Stop();
      m_nState = stStopped;
      break;
    case stPrevFrame:
      m_pVideoDecoder->GetPrevFrame(&pDecodedFrame);
      m_pVideoDecoder->Stop();
      m_nState = stStopped;
      break;
    case stNextKeyFrame:
      m_pVideoDecoder->GetNextIFrame(&pDecodedFrame);
      m_pVideoDecoder->Stop();
      m_nState = stStopped;
      break;
    case stPrevKeyFrame:
      m_pVideoDecoder->GetPrevIFrame(&pDecodedFrame);
      m_pVideoDecoder->Stop();
      m_nState = stStopped;      
      break;
    case stFastForward:
      m_pVideoDecoder->GetNextIFrame(&pDecodedFrame);  
      break;
    case stFastRewind:
      m_pVideoDecoder->GetPrevIFrame(&pDecodedFrame);
      break;    
    }

    // If something to draw, draw it
    if(pDecodedFrame)
    {    
      PostProcess(pDecodedFrame, &frProcessedFrame, &rs.pp);
      rs.pVideoRenderer->Draw(&frProcessedFrame);
      pDecodedFrame->Release();
    }
    // If we're going to stopeed, pause the video renderer
    if(m_nState==stStopped)
    {
      if( pDecodedFrame ) rs.pVideoRenderer->Pause(&frProcessedFrame);
    }
    else
      rs.pVideoRenderer->StartPlaying();

    // If we are stopped, wait for a command
    if( m_nState == stStopped )
      m_evCommandTrigger.Wait();
  }
}