/* 
 *  OutputManager.cpp
 *
 *	Copyright (C) Alberto Vigata - January 2000
 *
 *  This file is part of FlasKMPEG, a free MPEG to MPEG/AVI converter
 *	
 *  FlasKMPEG is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  FlasKMPEG is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "flaskmpeg.h"
#include "flasktypes.h"
#include "OutputManager.h"

#include "runstate.h"
#include "error.h"


extern void InitPremiereCallbacks();
extern compStdParms		stdParms;

extern TRunState rs;
extern TConfig   o;

extern HWND hMainWnd;
extern HINSTANCE hInst;

static COutputManager *gOutputManager;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
COutputManager::COutputManager()
{
  gOutputManager = this;
}

COutputManager::~COutputManager()
{

}

void COutputManager::ConfigureOutput(long nIndex)
{
  compGetFilePrefsRec     FilePrefsRec;
  CompileEntryFunc CompileEntry = rs.plugs.outPlugs[nIndex].CompileEntry;
  //User Preferences
  if(CompileEntry)
  {
    FilePrefsRec.compilerPrefs = rs.plugs.outPlugs[nIndex].settings;
    CompileEntry(compGetFilePrefs, &stdParms, (LONG)&FilePrefsRec,0);
    if( FilePrefsRec.compilerPrefs ){
      if(!rs.plugs.outPlugs[nIndex].settings)
        rs.plugs.outPlugs[nIndex].settings = (char *)malloc(OUT_PLUG_MEM);
      
      memcpy( rs.plugs.outPlugs[nIndex].settings, 
              FilePrefsRec.compilerPrefs, 
              _msize(FilePrefsRec.compilerPrefs) );
      
    }
  }
}
char *COutputManager::AddFileExtension(char *file)
{
  if(!file)
    return NULL;
  if(rs.plugs.outPlugs[rs.selected_out_plug].fileType==VXXtype ||
    rs.plugs.outPlugs[rs.selected_out_plug].fileType==ODMLtype )
    strcat(file,".avi");
  if(rs.plugs.outPlugs[rs.selected_out_plug].fileType=='MPEG'){
    if(rs.prof.audioMode==DO_AUDIO)
      strcat(file,".mpg");
    else
      strcat(file,".m1v");
  }
  if(rs.plugs.outPlugs[rs.selected_out_plug].fileType=='\0MPG'){
    if(rs.prof.audioMode==DO_AUDIO)
      strcat(file,".mpg");
    else
      strcat(file,".m1v");
  }
  return file;
}

bool COutputManager::Run( TOutputManagerRunInfo *pRunInfo )
{
  if(!pRunInfo)
    return false;

  m_sRunInfo = *pRunInfo;


  m_nConversionProcessed = 0;
  m_nConversionSize      = 0;
  m_nJobInProcess   = 0; 
  m_nJobCount       = 0;

  // Set the initial state of some events
  m_bIsRunning  = true;
  m_bTerminated = false;
  m_evCompileTerminated.Reset();
  // Close any resources allocated for a previous thread
  Close();
  // Create the encoding thread
  Create();
  return true;
}
bool COutputManager::Start(ui64 nStart, ui64 nStop)
{
  // Check for output files
  char  myFile[1024];
  // Init Demuxer
  m_nMyClock=0;
  m_nFrameCount =0 ;

  rs.pVideoRenderer->StartPlaying();

  if(rs.prof.warn_overwrite){
    FILE *input = NULL;
    strcpy( myFile, pOutputFileName );
    if(input=fopen(AddFileExtension(myFile), "rb")){
      fclose(input);
      if(MessageBox(hMainWnd, GS(WARN_OVERWRITE), GS(WARNING), MB_YESNO) == IDNO)
        return false;
    }
    strcpy( myFile, rs.prof.audioOutFile );
    if(input=fopen(AddFileExtension(myFile), "rb")){
      fclose(input);
      if(MessageBox(hMainWnd, GS(WARN_OVERWRITE), GS(WARNING), MB_YESNO) == IDNO)
        return false;
    }
    
  }
  // Get post processing options from config
  FromConfigToPPost(&rs.prof, &rs.pp);
  // Set additional parameters
  rs.pp.iDAR = rs.video->DAR;
  
  //InitDibDisplay(PPOST_WIDTH(rs.pp) , PPOST_HEIGHT(rs.pp));
  // Start Post processing

  // FIXME remove dependencies
  VBitmap vbIn;
  VBitmap vbOut;
  vbIn.init(NULL, rs.video->pictureWidth, rs.video->pictureHeight, 32);
  vbOut.init(NULL, PPOST_WIDTH(rs.pp), PPOST_HEIGHT(rs.pp), 32);

  PostProcessingStart(&vbIn, &vbOut, &rs.pp);
  rs.pp.myClock = 0;
  
  m_nFrameSpan= (i64)((1/((double)(rs.prof.timeBase.scale)/
                          (double)(rs.prof.timeBase.sampleSize)))*(double)MPEG2_CLK_REF);

  m_bImageWaiting = false;
  m_bFirstTime    = true;

  m_nAudPrevFrame = 0;


  
  TVideoOptions video_opt;
  //Start video
  video_opt.idctIndex            = rs.prof.idctIndex;
  video_opt.recons_progressive   = rs.prof.recons_progressive;
  video_opt.bStartInSync         = true;
  // Set the syncpoint for the video
  video_opt.nSyncPoint           = rs.video->GetSyncPoint(nStart);
  video_opt.nEndPoint            = nStop;

  // Create the frame buffer for the video
  m_pDecodedFrameBuffer = new CListFrameBuffer( rs.video->pictureWidth, 
                                                rs.video->pictureHeight, 
                                                FRAME_RGB, 32, 4);

  video_opt.pFrameBuffer = m_pDecodedFrameBuffer;

  if(rs.video) rs.video->Start(&video_opt);
    else return false;

  if(rs.prof.sampleFreqSameAsInput)
    rs.prof.outSampleFrequency = rs.audio->sampleRate;

  //Seek mobs at the begginnig of the compile
  if(rs.audio)
    rs.audio->SetStreamPos(video_opt.nSyncPoint);

  if(rs.audio) {
    switch( rs.prof.audioMode){
    case DSC:
      rs.audio->Start(rs.prof.audioOutFile, rs.prof.audioMode);
      break;
    case DO_AUDIO:
      rs.audio->Start(rs.prof.outSampleFrequency, 
                      rs.prof.audioMode, 
                      &rs.prof.sAudioProperties , 
                      &rs.sAudioTrack);
      break;
    }
  }
  // Create the video source
  m_pVideoSource = new CVideoSource(rs.video,  
                                    rs.pp.resWidth, rs.pp.resHeight,
                                    PPOST_WIDTH(rs.pp), PPOST_HEIGHT(rs.pp) );



  //EncoderMessage( ENCODE );
  //DisableAllMenu();
  //hDlg=CreateDialog(hInst,MAKEINTRESOURCE(IDD_PROGRESS), hWnd, (DLGPROC) ProgressDlg);
  //ShowWindow(hDlg, SW_SHOW);
  return true;
}

void COutputManager::Stop()
{
  if(m_pVideoSource)
    delete m_pVideoSource;


  if(rs.video)
    rs.video->Stop();
  if(rs.audio)
    rs.audio->Stop();

  if(  m_pDecodedFrameBuffer )
    delete m_pDecodedFrameBuffer;
    
  PostProcessingStop(&rs.pp);

  rs.pVideoRenderer->StopPlaying();
}

// Actual running thread
DWORD COutputManager::ThreadProc()
{	
  bool bSuccess=true;
  ui32 nResult;
  int i;

  m_oJobList = *m_sRunInfo.pJobList;

  ui32  nJobItems = m_oJobList.size();

  // No jobs? Add a job that
  // spans the whole project
  if(!nJobItems)
  {
    TBatchListElement el;
    el.nStart = 0;
    el.nEnd   = rs.video->GetStreamSize();
    m_oJobList.push_back(el);
    nJobItems = m_oJobList.size();
  }

  // Update total job size
  for( i=0; i<nJobItems; i++)
    m_nConversionSize += m_oJobList[i].nEnd - m_oJobList[i].nStart;

  m_nJobCount = nJobItems;

  strcpy( pOutputFileName, rs.prof.outputFile );

  m_nPartialProcessed=0;
  m_nConversionProcessed = 0;
  for( i=0; i<nJobItems; i++)
  {
    // Adapt filename if multiple jobs
    if(nJobItems>1)
      sprintf(pOutputFileName,"%s_%d", rs.prof.outputFile, i+1);
    
    strcpy( m_pFinalOutputFileName, pOutputFileName );
    AddFileExtension( m_pFinalOutputFileName );

    // update job in process
    m_nJobInProcess = i;
    m_nJobStartPos = m_oJobList[i].nStart;

    // start conversion
    if(!Start(m_oJobList[i].nStart, m_oJobList[i].nEnd ))
      break;

    // Do conversion
    nResult = Convert();

    // Update conversion processed
    m_nConversionProcessed = 0;
    m_nPartialProcessed += m_oJobList[i].nEnd - m_oJobList[i].nStart;
    
    // stop conversion
    Stop();

    // Exit if error in conversion
    if(nResult!=comp_ErrNone && 
       nResult!=comp_CompileAbort &&
       nResult!=comp_CompileDone)
    {
      bSuccess = false;
      if(!bSuccess)
        PrintError(ERROR_COMPILE, (int)NULL, 0);
      break;
    }
      
  }
  
  m_evCompileTerminated.Set();
  m_evCompileCanceled.Set();
  m_bTerminated = true;

  return true;

}

ui32 COutputManager::Convert()
{
  //Setup compiler plugin info
  CompileInfo.compilerID    = rs.plugs.outPlugs[rs.selected_out_plug].CompilerID;
  CompileInfo.compilerPrefs = rs.plugs.outPlugs[rs.selected_out_plug].settings;
  
  //CompOutputRec
  CompileInfo.outputRec.doVideo = 1;
  CompileInfo.outputRec.doAudio = (rs.prof.audioMode==DO_AUDIO);
  
  //vidCompression
  CompileInfo.outputRec.vidCompression.subtype  = rs.plugs.outPlugs[rs.selected_out_plug].fileType;
  CompileInfo.outputRec.vidCompression.depth    = 16;
  CompileInfo.outputRec.vidCompression.recompressWhen=recompNever;
  
  //CompileInfo.subtype=un
  //PostProcessing int gammaCorrection;
  CompileInfo.postProcessing.gammaCorrection = 1;
  CompileInfo.outputRec.width  =  PPOST_WIDTH(rs.pp) ;
  CompileInfo.outputRec.height =  PPOST_HEIGHT(rs.pp);
  
  //compTimebaseRec
  CompileInfo.outputRec.timebase.scale=			    rs.prof.timeBase.scale;
  CompileInfo.outputRec.timebase.sampleSize=		rs.prof.timeBase.sampleSize;
  CompileInfo.outputRec.timebase.value=			    rs.prof.timeBase.value;
  CompileInfo.outputRec.fieldType = compFieldsNone;
  
  //compPostProcessing
  //compFileSpec
  strcpy( CompileInfo.outputFile.name, pOutputFileName );
  AddFileExtension(CompileInfo.outputFile.name);
  
  WAVEFORMATEX AudioCompressionRecord;
  
  AudioCompressionRecord.wFormatTag      = WAVE_FORMAT_PCM;
  AudioCompressionRecord.nChannels       = 2;
  AudioCompressionRecord.nSamplesPerSec  = rs.prof.outSampleFrequency;
  AudioCompressionRecord.nAvgBytesPerSec = rs.prof.outSampleFrequency*4;
  AudioCompressionRecord.nBlockAlign     = 4;
  AudioCompressionRecord.wBitsPerSample  = 16; 
  AudioCompressionRecord.cbSize          = 0;
  
  memcpy(&CompileInfo.outputRec.audCompression.AudCompRec, (char *)&AudioCompressionRecord, sizeof(WAVEFORMATEX));
  CompileInfo.outputRec.audCompression.subtype = 0;
  
  CompileInfo.outputRec.fieldType     = compFieldsNone;
  CompileInfo.outputRec.audrate       = rs.prof.outSampleFrequency;
  CompileInfo.outputRec.audsamplesize = 16;
  CompileInfo.outputRec.stereo        = TRUE;
  CompileInfo.outputRec.audchunksize  = (int)((CompileInfo.outputRec.audrate)/((double)rs.prof.timeBase.scale/(double)rs.prof.timeBase.sampleSize)*4); //samples/Frame * 2bytes/sample * 2channels 
  //round chunksize to the next 4 boundary. A stereo sample takes 4 bytes
  int round;
  CompileInfo.outputRec.audchunksize += (round=CompileInfo.outputRec.audchunksize % 4) ? (4-round): round;
  CompileInfo.outputRec.audchunksize =  (CompileInfo.outputRec.audchunksize >> 2) << 4;
  
#define GET_FRAMERATE(x) ((double)x.scale/(double)x.sampleSize)
  m_nAudSamplesPerFrame = ((double)rs.prof.outSampleFrequency / (double)GET_FRAMERATE(rs.prof.timeBase));
  
  
  CompileInfo.outputRec.audInterleave = prInterleave1Frame;
  
  // This has to be modified to support some plugins.
  // now it is modified
  #define plug_is_cce ((rs.plugs.outPlugs[rs.selected_out_plug].ID[0]=='C') && \
                       (rs.plugs.outPlugs[rs.selected_out_plug].ID[6]=='C'))
  if (plug_is_cce)
  {
	o.options.hOutputFile = (int) CreateFile( CompileInfo.outputFile.name, 
											  GENERIC_WRITE, 
											  0, NULL, CREATE_ALWAYS,
											  FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN, NULL );
  
	if (o.options.hOutputFile == (int)INVALID_HANDLE_VALUE)
		o.options.hOutputFile=NULL;
  }
  else
	o.options.hOutputFile=NULL;
		  
  //To close: CloseHandle
  CompileInfo.outputFileRef = (compFileRef) o.options.hOutputFile;
  CompileInfo.startFrame    =  1;
  CompileInfo.endFrame      =  rs.prof.framesToCompile;
  CompileInfo.compileSeqID  =  0;
  
  
  //Update the current compileEntry routine
  CompileEntryFunc CompileEntry = rs.plugs.outPlugs[rs.selected_out_plug].CompileEntry;
  
  // set to null the frame in process 
  m_pFrameInProcess = NULL;
  
  int returnValue;
  
  if(CompileEntry)
    returnValue = CompileEntry(compDoCompile, &stdParms, (long)&CompileInfo,0);

  if (o.options.hOutputFile!=NULL&&plug_is_cce)
	CloseHandle((void*)o.options.hOutputFile);

  return returnValue;  
}

///////////////////////////////////
// fgetAudio : PREMIERE AUDIO STUB
////////////////////////////////////
int COutputManager::fgetAudio(long frame, 
                              long *frameCount, 
                              long *size, 
                              long offset, 
                              BufferReturnType theBuffer, 
                              long compileSeqID )
{
  static char audiodata[10240];
  static char *newblock;
  int pos_now, pos_next_frame;

  //Mimic Adobe Premiere behaviour
  // Current sample position
  pos_now        = (int)floor(   (double)(frame-1) *  m_nAudSamplesPerFrame );
  pos_next_frame = (int)floor(    (double)frame    *  m_nAudSamplesPerFrame );
  
  *size = (pos_next_frame - pos_now)*4;
  
  if( frame > m_nAudPrevFrame )
    rs.audio->GetSamples( frame, (short **)&newblock ,	*size>>2 );

  m_nAudPrevFrame = frame;
  *theBuffer      = &newblock;
    
  return comp_ErrNone;
}

int COutputManager::fgetFrame(long frame, 
                              void **buffer, 
                              long *rowbytes, 
                              compGetFrameReturnRec *getFrameReturn, 
                              char getCompressed, 
                              long compileSeqID )
{
	static compGetFrameReturnRec ReturnRecord;
  bool bAbortCompile = false;

  if( m_evCancelCompile.Check() )
    bAbortCompile = true;


  CFrame *pFrame;

	if(rs.plugs.outPlugs[rs.selected_out_plug].fileType == VXXtype)
		UpdateGraph(&rs.gr, (TAVIPlugGraphInfo *)compileSeqID);
	
	if(m_bFirstTime)
  {
		m_nMyClock   = 0;
		m_bFirstTime = false;
	}



  pFrame = m_pFrameInProcess;

  while(1)
  {
    if(!m_bImageWaiting)
    {  
      // Release the previous frame
      if( m_pFrameInProcess )
        m_pFrameInProcess->Release();

      if(m_pVideoSource->GetFrame(&m_pFrameInProcess)!=FRAME_OK)
      {
        bAbortCompile=true;
        break;
      }
      
      pFrame = m_pFrameInProcess;
    }
    //Check if th image has to be displayed in this frame
    if((pFrame->GetPresTime() >= m_nMyClock) && 
       (pFrame->GetPresTime() <(m_nMyClock + m_nFrameSpan)))
    {
      m_bImageWaiting=false;
      break;
    }
    else if(pFrame->GetPresTime() >= (m_nMyClock + m_nFrameSpan) )
    {
      m_bImageWaiting=true;
      break;
    }
    //Image has a PTS lower than myClock. Drop it
    m_bImageWaiting=false;
    DBG_STR((str, "OutputManager::fgetFrame - Dropping image by %d clock units\n", m_nMyClock - pFrame->GetPresTime()));
  }


	m_nMyClock += m_nFrameSpan;

  //If GetFrame returns FRAME_NOFRAME, video bitstream is over
	if(rs.audio)
		if(rs.prof.audioMode==DSC)//If we are in Direct Stream Copy audio mode
			rs.audio->GetAudioDSC(pFrame->GetPresTime(), rs.video->GetStreamPos());


		if(o.options.displayVideo)
      rs.pVideoRenderer->Draw(pFrame);
	
  if(m_evPauseCompile.Check())
  {
    m_evPauseCompile.Reset();
    m_bIsRunning = false;
    m_evResumeCompile.Wait();
    m_bIsRunning = true;
  }
		
		
	*buffer   = pFrame->GetBuffer();
	*rowbytes = 4*PPOST_WIDTH(rs.pp);

	ReturnRecord.returnVal         = comp_ErrNone;
	ReturnRecord.repeatCount       = 0;
	ReturnRecord.makeKeyFrame      = FALSE;
	ReturnRecord.frameIsCompressed = FALSE;
	ReturnRecord.startframe        = frame;
	ReturnRecord.frameOnMarker     = FALSE;


	m_nFrameCount++;
  // Update job position every 15 frames
  if(m_nFrameCount%15)
    m_nConversionProcessed = rs.video->GetStreamPos() - m_nJobStartPos;

	//DBG_STR((str, "GetFrame. Frame:%d ID:%d\n", frame, compileSeqID ));

	if(bAbortCompile || m_nFrameCount > rs.prof.framesToCompile )
  {
		ReturnRecord.returnVal = comp_CompileAbort;
    if(m_pFrameInProcess)
      m_pFrameInProcess->Release();
  }
		
	*getFrameReturn = ReturnRecord;

	return ReturnRecord.returnVal;
}

int fgetAudio ( long frame, long *frameCount, long *size, long offset, BufferReturnType theBuffer, long compileSeqID )
{
  return gOutputManager->fgetAudio(frame, frameCount, size, offset, theBuffer, compileSeqID);
}

long fgetBlipMax ( long compileSeqID ) 
{
	printf("GetBlipMax. ID:%d\n",compileSeqID);
	return gOutputManager->CompileInfo.outputRec.audchunksize;
}

///////////////////////////////////
// fgetFrame : PREMIERE VIDEO STUB
////////////////////////////////////
int fgetFrame ( long frame, void **buffer, long *rowbytes, compGetFrameReturnRec *getFrameReturn, char getCompressed, long compileSeqID )
{
  return gOutputManager->fgetFrame(frame, buffer, rowbytes, getFrameReturn, getCompressed, compileSeqID);
}